/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.tests.codegen;

import com.google.common.base.Objects;

/**
 * Abstract base class for constructs that can be abstract or concrete.
 */
@SuppressWarnings("all")
public abstract class Fragment<T extends Fragment<T>> {
  /**
   * Possible values for abstractness.
   */
  public enum Abstract {
    YES,
    
    NO;
  }
  
  /**
   * Extension methods for enumeration {@link Abstract}.
   */
  public static abstract class AbstractExtensions {
    /**
     * Return a name for the given classifier name and abstractness. The created name
     * has the classifier name as a prefix. If the given value indicates an abstract
     * construct, then the prefix is followed by an underscore and the word 'abstract'.
     * Otherwise, there is no suffix.
     * 
     * @param abstract_ whether or not the construct of interest is abstract
     * @param classifierName the classifier name prefix
     * 
     * @return the generated name
     */
    public static String makeName(final Fragment.Abstract abstract_, final String classifierName) {
      String _classifierExtension = Fragment.AbstractExtensions.getClassifierExtension(abstract_);
      return (classifierName + _classifierExtension);
    }
    
    /**
     * Return the appropriate extension for a classifier name depending on the given value.
     * 
     * @param abstract_ whether or not the construct of interest is abstract
     * 
     * @return the name extension
     */
    public static String getClassifierExtension(final Fragment.Abstract abstract_) {
      String _switchResult = null;
      if (abstract_ != null) {
        switch (abstract_) {
          case YES:
            _switchResult = "_abstract";
            break;
          case NO:
            _switchResult = "";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
    
    /**
     * Returns the generated string for the given value.
     * 
     * @param abstract_ whether or not the construct of interest is abstract
     * 
     * @return the generated string
     */
    public static String generate(final Fragment.Abstract abstract_) {
      String _switchResult = null;
      if (abstract_ != null) {
        switch (abstract_) {
          case YES:
            _switchResult = "abstract ";
            break;
          case NO:
            _switchResult = "";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
  }
  
  private Fragment.Abstract abstract_ = Fragment.Abstract.NO;
  
  /**
   * Creates a new instance.
   */
  protected Fragment() {
  }
  
  /**
   * Specifies that the this fragment should be abstract.
   */
  public T makeAbstract() {
    this.abstract_ = Fragment.Abstract.YES;
    return ((T) this);
  }
  
  /**
   * Indicates whether this construct is abstract.
   * 
   * @return <code>true</code> if this construct is abstract and <code>false</code> otherwise
   */
  protected boolean isAbstract() {
    return Objects.equal(this.abstract_, Fragment.Abstract.YES);
  }
  
  /**
   * Generates the appropriate N4JS code for this construct.
   * 
   * @return the generated code
   */
  public abstract CharSequence generate();
  
  /**
   * Generates the appropriate keyword for this fragment, depending on whether or not it is abstract.
   * 
   * @return the generated keyword, followed by a blank
   */
  protected String generateAbstract() {
    return Fragment.AbstractExtensions.generate(this.abstract_);
  }
}
