/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.n4idl.assistants;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.EqualityExpression;
import org.eclipse.n4js.n4JS.EqualityOperator;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.IfStatement;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4JS.VariableStatementKeyword;
import org.eclipse.n4js.n4idl.N4IDLGlobals;
import org.eclipse.n4js.n4idl.migrations.MigrationSwitchComputer;
import org.eclipse.n4js.n4idl.migrations.SwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeSwitchCondition;
import org.eclipse.n4js.transpiler.TransformationAssistant;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.es.n4idl.assistants.TypeSwitchTranspiler;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TMigratable;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;

/**
 * Transformation assistant for generating migration-support related ES code.
 */
@SuppressWarnings("all")
public class MigrationTransformationAssistant extends TransformationAssistant {
  @Inject
  private TypeSwitchTranspiler typeSwitchTranspiler;
  
  @Inject
  private MigrationSwitchComputer migrationSwitchComputer;
  
  private static final String SWITCH_ARGUMENT = "migrationArguments";
  
  private static final String MIGRATION_CANDIDATE_LIST = "migrationCandidates";
  
  /**
   * Creates the initializer statement for the static field which holds migrations associated with the given type declaration.
   * 
   * Returns an absent {@link Optional} if for the given type declaration, no migration support initializer
   * can be provided (e.g. non-migratable type).
   * 
   * @see N4IDLGlobals#MIGRATIONS_STATIC_FIELD
   */
  public Statement createMigrationSupportInitializer(final SymbolTableEntry steClass, final N4TypeDeclaration typeDecl) {
    SymbolTableEntryOriginal _xifexpression = null;
    if ((steClass instanceof SymbolTableEntryOriginal)) {
      _xifexpression = ((SymbolTableEntryOriginal)steClass);
    } else {
      _xifexpression = this.getState().steCache.mapOriginal.get(typeDecl);
    }
    final SymbolTableEntryOriginal originalSTE = _xifexpression;
    IdentifiableElement _originalTarget = null;
    if (originalSTE!=null) {
      _originalTarget=originalSTE.getOriginalTarget();
    }
    final IdentifiableElement typeModelElement = _originalTarget;
    if ((null == typeModelElement)) {
      String _name = typeDecl.getName();
      String _plus = ("Failed to generate migration meta-information for type " + _name);
      String _plus_1 = (_plus + ".");
      throw new IllegalStateException(_plus_1);
    }
    if ((!(typeModelElement instanceof TMigratable))) {
      String _name_1 = typeDecl.getName();
      String _plus_2 = ("Failed to generate migration meta-information for non-migratable type " + _name_1);
      String _plus_3 = (_plus_2 + ".");
      throw new IllegalStateException(_plus_3);
    }
    final Expression migrationsFieldValue = this.makeMigrationSwitch(((TMigratable) typeModelElement));
    final ParameterizedPropertyAccessExpression_IM staticFieldAccess = TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._IdentRef(steClass), this.getSymbolTableEntryInternal(N4IDLGlobals.MIGRATIONS_STATIC_FIELD, true));
    return TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._AssignmentExpr(staticFieldAccess, migrationsFieldValue));
  }
  
  private Expression makeMigrationSwitch(final TMigratable migratable) {
    final Function1<TMigration, Integer> _function = (TMigration it) -> {
      return Integer.valueOf(it.getTargetVersion());
    };
    final Map<Integer, List<TMigration>> migrationsByTargetVersion = IterableExtensions.<Integer, TMigration>groupBy(migratable.getMigrations(), _function);
    final Function1<Map.Entry<Integer, List<TMigration>>, Pair<String, Expression>> _function_1 = (Map.Entry<Integer, List<TMigration>> entry) -> {
      FunctionExpression _makeMigrationSwitchFunction = this.makeMigrationSwitchFunction(entry.getValue());
      return Pair.<String, Expression>of(
        Integer.toString((entry.getKey()).intValue()), 
        ((Expression) _makeMigrationSwitchFunction));
    };
    return TranspilerBuilderBlocks._ObjLit(((Pair<String, Expression>[])Conversions.unwrapArray(IterableExtensions.<Pair<String, Expression>>toList(IterableExtensions.<Map.Entry<Integer, List<TMigration>>, Pair<String, Expression>>map(migrationsByTargetVersion.entrySet(), _function_1)), Pair.class)));
  }
  
  private FunctionExpression makeMigrationSwitchFunction(final List<TMigration> migrations) {
    final List<Statement> bodyStatements = CollectionLiterals.<Statement>newArrayList();
    final SymbolTableEntryInternal switchParameter = this.getSymbolTableEntryInternal(MigrationTransformationAssistant.SWITCH_ARGUMENT, true);
    final SymbolTableEntryInternal migrationCandidatesSTE = this.getSymbolTableEntryInternal(MigrationTransformationAssistant.MIGRATION_CANDIDATE_LIST, true);
    VariableStatement __VariableStatement = TranspilerBuilderBlocks._VariableStatement(VariableStatementKeyword.LET, TranspilerBuilderBlocks._VariableDeclaration(MigrationTransformationAssistant.MIGRATION_CANDIDATE_LIST, TranspilerBuilderBlocks._ArrLit()));
    bodyStatements.add(__VariableStatement);
    final Function1<TMigration, List<Statement>> _function = (TMigration it) -> {
      return this.makeIfStatement(it, switchParameter, migrationCandidatesSTE);
    };
    Iterables.<Statement>addAll(bodyStatements, Iterables.<Statement>concat(ListExtensions.<TMigration, List<Statement>>map(migrations, _function)));
    ReturnStatement __ReturnStmnt = TranspilerBuilderBlocks._ReturnStmnt(TranspilerBuilderBlocks._IdentRef(migrationCandidatesSTE));
    bodyStatements.add(__ReturnStmnt);
    FormalParameter __FormalParameter = TranspilerBuilderBlocks._FormalParameter(MigrationTransformationAssistant.SWITCH_ARGUMENT);
    return TranspilerBuilderBlocks._FunExpr(false, null, new FormalParameter[] { __FormalParameter }, TranspilerBuilderBlocks._Block(((Statement[])Conversions.unwrapArray(bodyStatements, Statement.class))));
  }
  
  /**
   * Create the if-statement which checks for argument conditions and returns the migration
   * if it matches.
   * 
   * @param migration The migration.
   * @param migrationParameters STE of the migration parameters (array).
   */
  private List<Statement> makeIfStatement(final TMigration migration, final SymbolTableEntryInternal migrationParameters, final SymbolTableEntryInternal migrationCandidatesSTE) {
    final List<Expression> conditions = CollectionLiterals.<Expression>newArrayList();
    final EqualityExpression parameterCountCondition = TranspilerBuilderBlocks._EqualityExpr(
      TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._IdentRef(migrationParameters), this.getSymbolTableEntryInternal("length", true)), 
      EqualityOperator.EQ, 
      TranspilerBuilderBlocks._NumericLiteral(migration.getSourceTypeRefs().size()));
    conditions.add(parameterCountCondition);
    final Function1<TypeRef, SwitchCondition> _function = (TypeRef it) -> {
      SwitchCondition _xtrycatchfinallyexpression = null;
      try {
        _xtrycatchfinallyexpression = this.migrationSwitchComputer.compute(it);
      } catch (final Throwable _t) {
        if (_t instanceof MigrationSwitchComputer.UnhandledTypeRefException) {
          final MigrationSwitchComputer.UnhandledTypeRefException e = (MigrationSwitchComputer.UnhandledTypeRefException)_t;
          String _name = migration.getName();
          String _plus = ("Failed to compute migration source type switch for migration " + _name);
          throw new IllegalStateException(_plus, e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      return _xtrycatchfinallyexpression;
    };
    final List<SwitchCondition> parameterTypeConditions = ListExtensions.<TypeRef, SwitchCondition>map(migration.getSourceTypeRefs(), _function);
    final Function1<SwitchCondition, Iterable<TypeSwitchCondition>> _function_1 = (SwitchCondition c) -> {
      return Iterables.<TypeSwitchCondition>filter(c.subConditions(), TypeSwitchCondition.class);
    };
    final Function1<TypeSwitchCondition, Type> _function_2 = (TypeSwitchCondition typeConditions) -> {
      return typeConditions.type;
    };
    final Iterable<Type> typeConditionTypes = IterableExtensions.<TypeSwitchCondition, Type>map(Iterables.<TypeSwitchCondition>concat(ListExtensions.<SwitchCondition, Iterable<TypeSwitchCondition>>map(parameterTypeConditions, _function_1)), _function_2);
    final Consumer<Type> _function_3 = (Type i) -> {
      this.addNamedImport(i, null);
    };
    typeConditionTypes.forEach(_function_3);
    final Procedure2<SwitchCondition, Integer> _function_4 = (SwitchCondition condition, Integer index) -> {
      final IndexedAccessExpression lhs = TranspilerBuilderBlocks._IndexAccessExpr(TranspilerBuilderBlocks._IdentRef(migrationParameters), TranspilerBuilderBlocks._NumericLiteral((index).intValue()));
      List<Expression> _transform = this.typeSwitchTranspiler.transform(condition, lhs);
      Iterables.<Expression>addAll(conditions, _transform);
    };
    IterableExtensions.<SwitchCondition>forEach(parameterTypeConditions, _function_4);
    final Function1<TypeRef, String> _function_5 = (TypeRef it) -> {
      return it.getTypeRefAsString();
    };
    String _join = IterableExtensions.join(ListExtensions.<TypeRef, String>map(migration.getSourceTypeRefs(), _function_5), ", ");
    String _plus = ("// (" + _join);
    String _plus_1 = (_plus + ")");
    ExpressionStatement __SnippetAsStmnt = TranspilerBuilderBlocks._SnippetAsStmnt(_plus_1);
    IfStatement __IfStmnt = TranspilerBuilderBlocks._IfStmnt(
      TranspilerBuilderBlocks._AND(conditions), 
      this._PushArrayElement(migrationCandidatesSTE, this._MigrationCandidateElement(migration)));
    return Collections.<Statement>unmodifiableList(CollectionLiterals.<Statement>newArrayList(__SnippetAsStmnt, __IfStmnt));
  }
  
  private Expression _MigrationCandidateElement(final TMigration migration) {
    final SymbolTableEntryOriginal migrationSTE = this.getSymbolTableEntryOriginal(migration, true);
    final Function1<TypeRef, ArrayElement> _function = (TypeRef typeRef) -> {
      return this._ParameterTypeArrayElement(typeRef);
    };
    final List<ArrayElement> sourceTypeArrayElements = ListExtensions.<TypeRef, ArrayElement>map(migration.getSourceTypeRefs(), _function);
    IdentifierRef_IM __IdentRef = TranspilerBuilderBlocks._IdentRef(migrationSTE);
    Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("migration", __IdentRef);
    ArrayLiteral __ArrLit = TranspilerBuilderBlocks._ArrLit(((ArrayElement[])Conversions.unwrapArray(sourceTypeArrayElements, ArrayElement.class)));
    Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("parameterTypes", __ArrLit);
    return TranspilerBuilderBlocks._ObjLit(_mappedTo, _mappedTo_1);
  }
  
  /**
   * Returns an {@link ArrayElement} which represents the given {@link Type} at runtime.
   * 
   * For instance, for {@link TN4Classifier}s or {@link TObjectPrototype}s that is an
   * instance of {@code type{Object}} and for primitive types that is a string {@code "primitive"}.
   * 
   * {@link TypeTypeRef}s are represented as plain object of the form <code>{type: <constructor/type ref>}</code>
   * 
   * Unhandled type reference types will throw an {@link IllegalStateException}.
   */
  private ArrayElement _ParameterTypeArrayElement(final TypeRef typeRef) {
    ArrayElement _switchResult = null;
    boolean _matched = false;
    Type _declaredType = typeRef.getDeclaredType();
    boolean _tripleNotEquals = (_declaredType != null);
    if (_tripleNotEquals) {
      _matched=true;
      ArrayElement _switchResult_1 = null;
      Type _declaredType_1 = typeRef.getDeclaredType();
      boolean _matched_1 = false;
      if (_declaredType_1 instanceof TN4Classifier) {
        _matched_1=true;
        _switchResult_1 = TranspilerBuilderBlocks._ArrayElement(TranspilerBuilderBlocks._IdentRef(this.getSymbolTableEntryOriginal(typeRef.getDeclaredType(), true)));
      }
      if (!_matched_1) {
        if (_declaredType_1 instanceof PrimitiveType) {
          _matched_1=true;
          _switchResult_1 = TranspilerBuilderBlocks._ArrayElement(TranspilerBuilderBlocks._StringLiteral("primitive"));
        }
      }
      if (!_matched_1) {
        if (_declaredType_1 instanceof TObjectPrototype) {
          _matched_1=true;
          _switchResult_1 = TranspilerBuilderBlocks._ArrayElement(TranspilerBuilderBlocks._IdentRef(this.getSymbolTableEntryOriginal(typeRef.getDeclaredType(), true)));
        }
      }
      _switchResult = _switchResult_1;
    }
    if (!_matched) {
      if (typeRef instanceof TypeTypeRef) {
        _matched=true;
        IdentifierRef_IM __IdentRef = TranspilerBuilderBlocks._IdentRef(this.getSymbolTableEntryOriginal(this.getType(((TypeTypeRef)typeRef)), true));
        Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("type", __IdentRef);
        _switchResult = TranspilerBuilderBlocks._ArrayElement(TranspilerBuilderBlocks._ObjLit(_mappedTo));
      }
    }
    if (!_matched) {
      throw new IllegalStateException(("Unhandled migration source type reference " + typeRef));
    }
    return _switchResult;
  }
  
  /**
   * Returns the type which is referred to by the given {@link TypeTypeRef}.
   */
  private Type getType(final TypeTypeRef ref) {
    final TypeArgument typeArg = ref.getTypeArg();
    if ((typeArg instanceof TypeRef)) {
      return ((TypeRef)typeArg).getDeclaredType();
    }
    return null;
  }
  
  /**
   * Creates a {@code <arraySTE>.push(<element>)} expression which adds the element expression to the array
   * referenced by arraySTE.
   */
  private Statement _PushArrayElement(final SymbolTableEntry arraySTE, final Expression element) {
    return TranspilerBuilderBlocks._ExprStmnt(
      TranspilerBuilderBlocks._CallExpr(
        TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._IdentRef(arraySTE), this.getSymbolTableEntryInternal("push", true)), element));
  }
}
