/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDefinition;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.organize.imports.ScriptDependencyResolver;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.ParameterizedTypeRef_IM;
import org.eclipse.n4js.transpiler.im.ReferencingElement_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TEnumLiteral;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.StaticPolyfillHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

@SuppressWarnings("all")
public class StaticPolyfillTransformation extends Transformation {
  @Inject
  private StaticPolyfillHelper staticPolyfillHelper;
  
  /**
   * While inserting members into classes, we add here the elements referenced in the initializer expressions, fpar
   * default expressions and bodies.
   */
  private Set<SymbolTableEntryOriginal> referencedElements = CollectionLiterals.<SymbolTableEntryOriginal>newHashSet();
  
  /**
   * Each referenced element for which we add an import gets its own alias. This set is used to keep track of those
   * aliases (mainly when creating a new, unique alias).
   */
  private Set<String> referencedElementsAliases = CollectionLiterals.<String>newHashSet();
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final boolean isAware = N4JSLanguageUtils.isContainedInStaticPolyfillAware(this.getState().resource.getScript());
    if (isAware) {
      final N4JSResource fillingResource = this.staticPolyfillHelper.getStaticPolyfillResource(this.getState().resource);
      if ((fillingResource != null)) {
        this.referencedElements.clear();
        final Consumer<N4ClassDeclaration> _function = (N4ClassDeclaration it) -> {
          this.doStaticPolyfilling(it);
        };
        this.<N4ClassDeclaration>collectNodes(this.getState().im, N4ClassDeclaration.class, false).forEach(_function);
        final Consumer<SymbolTableEntryOriginal> _function_1 = (SymbolTableEntryOriginal it) -> {
          this.addImportIfRequired(it, fillingResource);
        };
        this.referencedElements.forEach(_function_1);
      }
    }
  }
  
  private void doStaticPolyfilling(final N4ClassDeclaration classDecl) {
    final TClass defCls = this.getState().info.getOriginalDefinedType(classDecl);
    final N4ClassDeclaration filler = this.staticPolyfillHelper.getStaticPolyfill(defCls);
    if ((filler != null)) {
      this.doStaticPolyfilling(classDecl, filler);
    }
  }
  
  private void doStaticPolyfilling(final N4ClassDeclaration classFilled, final N4ClassDeclaration classFiller) {
    final Function1<ParameterizedTypeRef_IM, SymbolTableEntry> _function = (ParameterizedTypeRef_IM it) -> {
      return it.getDeclaredType_IM();
    };
    final Set<TInterface> currentIfcs = IterableExtensions.<TInterface>toSet(Iterables.<TInterface>filter(IterableExtensions.<ParameterizedTypeRef_IM, SymbolTableEntry>map(Iterables.<ParameterizedTypeRef_IM>filter(classFilled.getImplementedInterfaceRefs(), ParameterizedTypeRef_IM.class), _function), TInterface.class));
    final Consumer<ParameterizedTypeRef> _function_1 = (ParameterizedTypeRef it) -> {
      this.insertImplementedInterface(classFilled, it, currentIfcs);
    };
    classFiller.getImplementedInterfaceRefs().forEach(_function_1);
    final Consumer<N4MemberDeclaration> _function_2 = (N4MemberDeclaration it) -> {
      this.insertMember(classFilled, it);
    };
    classFiller.getOwnedMembers().forEach(_function_2);
  }
  
  private void insertImplementedInterface(final N4ClassDefinition classFilled, final ParameterizedTypeRef ifcRefToBeInserted, final Set<TInterface> currentIfcs) {
    final Type ifcType = ifcRefToBeInserted.getDeclaredType();
    if ((ifcType instanceof TInterface)) {
      boolean _contains = currentIfcs.contains(ifcType);
      boolean _not = (!_contains);
      if (_not) {
        final SymbolTableEntryOriginal ifcSTE = this.getSymbolTableEntryOriginal(ifcType, true);
        EList<ParameterizedTypeRef> _implementedInterfaceRefs = classFilled.getImplementedInterfaceRefs();
        ParameterizedTypeRef_IM __ParameterizedTypeRef = TranspilerBuilderBlocks._ParameterizedTypeRef(ifcSTE);
        _implementedInterfaceRefs.add(__ParameterizedTypeRef);
      }
    }
  }
  
  /**
   * Create and insert a copy of <code>memberToBeInserted</code> into <code>classFilled</code>.
   * 
   * @param classFilled  the class declaration to be filled, must be contained in the intermediate model.
   * @param memberToBeInserted  this member is expected to be contained in the AST of the filling resource, so this is
   *            neither contained in the original AST (of the resource to compile) nor the intermediate model!
   */
  private void insertMember(final N4ClassDefinition classFilled, final N4MemberDeclaration memberToBeInserted) {
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf(((it.eClass() == memberToBeInserted.eClass()) && Objects.equal(it.getName(), memberToBeInserted.getName())));
    };
    final N4MemberDeclaration existing = IterableExtensions.<N4MemberDeclaration>findFirst(classFilled.getOwnedMembers(), _function);
    final N4MemberDeclaration copy = this.<N4MemberDeclaration>copyAlienElement(memberToBeInserted);
    final Function1<ReferencingElement_IM, SymbolTableEntry> _function_1 = (ReferencingElement_IM it) -> {
      return it.getRewiredTarget();
    };
    List<SymbolTableEntryOriginal> _list = IteratorExtensions.<SymbolTableEntryOriginal>toList(Iterators.<SymbolTableEntryOriginal>filter(IteratorExtensions.<ReferencingElement_IM, SymbolTableEntry>map(Iterators.<ReferencingElement_IM>filter(copy.eAllContents(), ReferencingElement_IM.class), _function_1), SymbolTableEntryOriginal.class));
    Iterables.<SymbolTableEntryOriginal>addAll(this.referencedElements, _list);
    if ((existing != null)) {
      this.replace(existing, copy);
    } else {
      EList<N4MemberDeclaration> _ownedMembersRaw = classFilled.getOwnedMembersRaw();
      _ownedMembersRaw.add(copy);
    }
    this.getState().info.markAsStaticlyPolyfilled(copy);
    this.getState().info.setOriginalDefinedMember(copy, memberToBeInserted.getDefinedTypeElement());
  }
  
  private void addImportIfRequired(final SymbolTableEntryOriginal ste, final N4JSResource fillingResource) {
    ImportSpecifier _importSpecifier = ste.getImportSpecifier();
    boolean _tripleEquals = (_importSpecifier == null);
    if (_tripleEquals) {
      final IdentifiableElement originalTarget = ste.getOriginalTarget();
      final boolean isNested = ((originalTarget instanceof TMember) || (originalTarget instanceof TEnumLiteral));
      if ((!isNested)) {
        Resource _eResource = originalTarget.eResource();
        final boolean isLocal = (_eResource == this.getState().resource);
        if (((!isLocal) && ScriptDependencyResolver.shouldBeImported(fillingResource.getModule(), originalTarget))) {
          this.addImport(ste, fillingResource);
        }
      }
    }
  }
  
  /**
   * Add an import for the element represented by the given SymbolTableEntry in the intermediate model.
   */
  private void addImport(final SymbolTableEntryOriginal ste, final N4JSResource fillingResource) {
    final IdentifiableElement importedElement = ste.getOriginalTarget();
    final boolean isNamespace = (importedElement instanceof ModuleNamespaceVirtualType);
    TModule _xifexpression = null;
    if (isNamespace) {
      _xifexpression = ((ModuleNamespaceVirtualType) importedElement).getModule();
    } else {
      _xifexpression = EcoreUtil2.<TModule>getContainerOfType(importedElement, TModule.class);
    }
    final TModule remoteModule = _xifexpression;
    final Function1<ImportDeclaration, Boolean> _function = (ImportDeclaration it) -> {
      TModule _module = it.getModule();
      return Boolean.valueOf((_module == remoteModule));
    };
    final Function1<ImportDeclaration, EList<ImportSpecifier>> _function_1 = (ImportDeclaration it) -> {
      return it.getImportSpecifiers();
    };
    final Iterable<ImportSpecifier> impSpecsForContainingModule = Iterables.<ImportSpecifier>concat(IterableExtensions.<ImportDeclaration, EList<ImportSpecifier>>map(IterableExtensions.<ImportDeclaration>filter(Iterables.<ImportDeclaration>filter(fillingResource.getScript().getScriptElements(), ImportDeclaration.class), _function), _function_1));
    ImportSpecifier _xifexpression_1 = null;
    if (isNamespace) {
      final Function1<NamespaceImportSpecifier, Boolean> _function_2 = (NamespaceImportSpecifier it) -> {
        Type _definedType = it.getDefinedType();
        return Boolean.valueOf((_definedType == importedElement));
      };
      _xifexpression_1 = IterableExtensions.<NamespaceImportSpecifier>findFirst(Iterables.<NamespaceImportSpecifier>filter(impSpecsForContainingModule, NamespaceImportSpecifier.class), _function_2);
    } else {
      final Function1<NamedImportSpecifier, Boolean> _function_3 = (NamedImportSpecifier it) -> {
        TExportableElement _importedElement = it.getImportedElement();
        return Boolean.valueOf((_importedElement == importedElement));
      };
      _xifexpression_1 = IterableExtensions.<NamedImportSpecifier>findFirst(Iterables.<NamedImportSpecifier>filter(impSpecsForContainingModule, NamedImportSpecifier.class), _function_3);
    }
    final ImportSpecifier impSpec_original = _xifexpression_1;
    EObject _eContainer = null;
    if (impSpec_original!=null) {
      _eContainer=impSpec_original.eContainer();
    }
    final EObject impDecl_original = _eContainer;
    if (((impDecl_original != null) && (impSpec_original != null))) {
      String _elvis = null;
      String _elvis_1 = null;
      String _alias = StaticPolyfillTransformation.getAlias(impSpec_original);
      if (_alias != null) {
        _elvis_1 = _alias;
      } else {
        String _exportedName = ste.getExportedName();
        _elvis_1 = _exportedName;
      }
      if (_elvis_1 != null) {
        _elvis = _elvis_1;
      } else {
        _elvis = "unnamed";
      }
      final String alias = this.chooseNewUniqueAlias(_elvis);
      ImportSpecifier _xifexpression_2 = null;
      if (isNamespace) {
        _xifexpression_2 = TranspilerBuilderBlocks._NamespaceImportSpecifier(alias, true);
      } else {
        _xifexpression_2 = TranspilerBuilderBlocks._NamedImportSpecifier(ste.getExportedName(), alias, true);
      }
      final ImportSpecifier impSpec = _xifexpression_2;
      final ImportDeclaration impDecl = TranspilerBuilderBlocks._ImportDecl(impSpec);
      this.getState().im.getScriptElements().add(0, impDecl);
      ste.setName(alias);
      ste.setImportSpecifier(impSpec);
      this.getState().tracer.setOriginalASTNode(impDecl, impDecl_original);
      this.getState().tracer.setOriginalASTNode(impSpec, impSpec_original);
      this.getState().info.setImportedModule(impDecl, remoteModule);
    }
  }
  
  private String chooseNewUniqueAlias(final String baseName) {
    String alias = null;
    int cnt = 0;
    do {
      {
        String _xifexpression = null;
        if ((cnt == 0)) {
          _xifexpression = "";
        } else {
          _xifexpression = Integer.toString(cnt);
        }
        final String cntStr = _xifexpression;
        alias = ((baseName + cntStr) + "$polyfilled");
        cnt++;
      }
    } while(this.referencedElementsAliases.contains(alias));
    this.referencedElementsAliases.add(alias);
    return alias;
  }
  
  private static final String getAlias(final ImportSpecifier impSpec) {
    String _switchResult = null;
    boolean _matched = false;
    if (impSpec instanceof NamedImportSpecifier) {
      _matched=true;
      _switchResult = ((NamedImportSpecifier)impSpec).getAlias();
    }
    if (!_matched) {
      if (impSpec instanceof NamespaceImportSpecifier) {
        _matched=true;
        _switchResult = ((NamespaceImportSpecifier)impSpec).getAlias();
      }
    }
    return _switchResult;
  }
}
