/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.contentassist;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.n4js.n4JS.JSXElement;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4idl.N4IDLGlobals;
import org.eclipse.n4js.resource.N4JSResourceDescriptionStrategy;
import org.eclipse.n4js.services.N4JSGrammarAccess;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.DeclaredTypeWithAccessModifier;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMemberWithAccessModifier;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ui.contentassist.AbstractN4JSProposalProvider;
import org.eclipse.n4js.ui.contentassist.N4JSCandidateFilter;
import org.eclipse.n4js.ui.labeling.N4JSLabelProvider;
import org.eclipse.n4js.ui.proposals.imports.ImportsAwareReferenceProposalCreator;
import org.eclipse.n4js.ui.proposals.linkedEditing.N4JSCompletionProposal;
import org.eclipse.n4js.ui.utils.ConfigurableCompletionProposalExtensions;
import org.eclipse.swt.graphics.Image;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.AbstractElement;
import org.eclipse.xtext.AbstractRule;
import org.eclipse.xtext.CrossReference;
import org.eclipse.xtext.GrammarUtil;
import org.eclipse.xtext.Keyword;
import org.eclipse.xtext.ParserRule;
import org.eclipse.xtext.RuleCall;
import org.eclipse.xtext.conversion.ValueConverterException;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.ui.editor.contentassist.AbstractJavaBasedContentProposalProvider;
import org.eclipse.xtext.ui.editor.contentassist.ConfigurableCompletionProposal;
import org.eclipse.xtext.ui.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ui.editor.contentassist.ICompletionProposalAcceptor;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * see http://www.eclipse.org/Xtext/documentation.html#contentAssist on how to customize content assistant
 */
@SuppressWarnings("all")
public class N4JSProposalProvider extends AbstractN4JSProposalProvider {
  @Data
  private static final class ProposalBracketInfo {
    private final String brackets;
    
    private final int cursorOffset;
    
    public ProposalBracketInfo(final String brackets, final int cursorOffset) {
      super();
      this.brackets = brackets;
      this.cursorOffset = cursorOffset;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.brackets== null) ? 0 : this.brackets.hashCode());
      return prime * result + this.cursorOffset;
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      N4JSProposalProvider.ProposalBracketInfo other = (N4JSProposalProvider.ProposalBracketInfo) obj;
      if (this.brackets == null) {
        if (other.brackets != null)
          return false;
      } else if (!this.brackets.equals(other.brackets))
        return false;
      if (other.cursorOffset != this.cursorOffset)
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("brackets", this.brackets);
      b.add("cursorOffset", this.cursorOffset);
      return b.toString();
    }
    
    @Pure
    public String getBrackets() {
      return this.brackets;
    }
    
    @Pure
    public int getCursorOffset() {
      return this.cursorOffset;
    }
  }
  
  private static final String KEY_SCANNED_SCOPES = (N4JSProposalProvider.class.getName() + "_scannedScopes");
  
  @Inject
  private ImportsAwareReferenceProposalCreator importAwareReferenceProposalCreator;
  
  @Inject
  private IQualifiedNameProvider qualifiedNameProvider;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  @Inject
  private N4JSGrammarAccess n4jsGrammarAccess;
  
  @Inject
  private N4JSLabelProvider labelProvider;
  
  @Override
  public void completeRuleCall(final RuleCall ruleCall, final ContentAssistContext contentAssistContext, final ICompletionProposalAcceptor acceptor) {
    final AbstractRule calledRule = ruleCall.getRule();
    boolean _equals = "INT".equals(calledRule.getName());
    boolean _not = (!_equals);
    if (_not) {
      super.completeRuleCall(ruleCall, contentAssistContext, acceptor);
    }
  }
  
  @Override
  protected void lookupCrossReference(final CrossReference crossReference, final ContentAssistContext contentAssistContext, final ICompletionProposalAcceptor acceptor) {
    N4JSCandidateFilter _n4JSCandidateFilter = new N4JSCandidateFilter();
    this.lookupCrossReference(crossReference, contentAssistContext, acceptor, _n4JSCandidateFilter);
  }
  
  @Override
  protected void lookupCrossReference(final CrossReference crossReference, final ContentAssistContext contentAssistContext, final ICompletionProposalAcceptor acceptor, final Predicate<IEObjectDescription> filter) {
    ParserRule containingParserRule = GrammarUtil.containingParserRule(crossReference);
    ParserRule _typeReferenceRule = this.n4jsGrammarAccess.getTypeReferenceRule();
    boolean _tripleEquals = (containingParserRule == _typeReferenceRule);
    if (_tripleEquals) {
      final String featureName = GrammarUtil.containingAssignment(crossReference).getFeature();
      String _name = TypeRefsPackage.eINSTANCE.getParameterizedTypeRef_DeclaredType().getName();
      boolean _equals = Objects.equal(featureName, _name);
      if (_equals) {
        this.lookupCrossReference(crossReference, TypeRefsPackage.eINSTANCE.getParameterizedTypeRef_DeclaredType(), contentAssistContext, acceptor, filter);
      }
    }
    super.lookupCrossReference(crossReference, contentAssistContext, acceptor, filter);
  }
  
  /**
   * For type proposals, use a dedicated proposal creator that will query the scope, filter it and
   * apply the proposal factory to all applicable {@link IEObjectDescription descriptions}.
   */
  @Override
  protected void lookupCrossReference(final CrossReference crossReference, final EReference reference, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Predicate<IEObjectDescription> filter) {
    boolean _haveScannedScopeFor = this.haveScannedScopeFor(context.getCurrentModel(), reference);
    if (_haveScannedScopeFor) {
      return;
    }
    if ((reference.getEReferenceType().isSuperTypeOf(TypesPackage.Literals.TYPE) || 
      TypesPackage.Literals.TYPE.isSuperTypeOf(reference.getEReferenceType()))) {
      String ruleName = null;
      AbstractElement _terminal = crossReference.getTerminal();
      if ((_terminal instanceof RuleCall)) {
        AbstractElement _terminal_1 = crossReference.getTerminal();
        ruleName = ((RuleCall) _terminal_1).getRule().getName();
      }
      final Function<IEObjectDescription, ICompletionProposal> proposalFactory = this.getProposalFactory(ruleName, context);
      this.importAwareReferenceProposalCreator.lookupCrossReference(context.getCurrentModel(), reference, context, acceptor, filter, proposalFactory);
    } else {
      super.lookupCrossReference(crossReference, reference, context, acceptor, filter);
    }
  }
  
  /**
   * Tells whether the scope for the given context and reference has already been scanned for completion proposals.
   * In addition, the scope for the given context and reference is marked as having been scanned (if not marked as
   * such already).
   */
  private boolean haveScannedScopeFor(final EObject context, final EReference reference) {
    boolean _announceProcessing = this.announceProcessing(Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList(N4JSProposalProvider.KEY_SCANNED_SCOPES, context, reference)));
    return (!_announceProcessing);
  }
  
  /**
   * <b>TEMPORARY WORK-AROUND</b>
   * <p>
   * Our proposal provider implementation makes heavy use of some Xtext base implementations intended for Java (esp.
   * the abstract base class AbstractJavaBasedContentProposalProvider) which in many places has '.' hard-code as the
   * delimiter of qualified names (e.g. FQNPrefixMatcher). This wasn't a problem as long as we used '.' as delimiter
   * in N4JS qualified names, but this was changed to '/' as of GHOLD-162. Thus, the base implementations intended for
   * Java now get confused.
   * </p><p>
   * <b>TEMPORARY PARTIAL SOLUTION</b>: to not confuse the implementations intended for Java, we do not provide our
   * "official" N4JSQualifiedNameComputer obtained via injection, but instead provide a fake IQualifiedNameComputer
   * that uses '.' as delimiter.<br>
   * However, this <em>will</em> break in case of project/folder names containing '.' (which is valid in Javascript).
   * </p><p>
   * TODO IDE-2227 fix handling of qualified names in content assist or create follow-up task
   * </p>
   * 
   * @see AbstractJavaBasedContentProposalProvider
   */
  @Override
  protected Function<IEObjectDescription, ICompletionProposal> getProposalFactory(final String ruleName, final ContentAssistContext contentAssistContext) {
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(contentAssistContext.getResource().getResourceSet());
    TClassifier _objectType = builtInTypeScope.getObjectType();
    TClass _n4ObjectType = builtInTypeScope.getN4ObjectType();
    final List<? extends TClassifier> secondaryTypes = Collections.<TClassifier>unmodifiableList(CollectionLiterals.<TClassifier>newArrayList(_objectType, _n4ObjectType));
    final Function1<TClassifier, EList<TMember>> _function = (TClassifier it) -> {
      return it.getOwnedMembers();
    };
    final Function1<TMember, URI> _function_1 = (TMember it) -> {
      return EcoreUtil.getURI(it);
    };
    final Set<URI> urisOfSecondaryMembers = IterableExtensions.<URI>toSet(IterableExtensions.<TMember, URI>map(IterableExtensions.flatMap(secondaryTypes, _function), _function_1));
    final IQualifiedNameConverter.DefaultImpl myConverter = new IQualifiedNameConverter.DefaultImpl();
    return new AbstractJavaBasedContentProposalProvider.DefaultProposalCreator(contentAssistContext, ruleName, myConverter) {
      @Override
      public ICompletionProposal apply(final IEObjectDescription candidate) {
        if ((candidate == null)) {
          return null;
        }
        ICompletionProposal result = null;
        String proposal = myConverter.toString(candidate.getName());
        if ((this.valueConverter != null)) {
          try {
            proposal = this.valueConverter.toString(proposal);
          } catch (final Throwable _t) {
            if (_t instanceof ValueConverterException) {
              return null;
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        } else {
          if ((ruleName != null)) {
            try {
              proposal = N4JSProposalProvider.this.getValueConverter().toString(proposal, ruleName);
            } catch (final Throwable _t) {
              if (_t instanceof ValueConverterException) {
                return null;
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          }
        }
        final StyledString displayString = N4JSProposalProvider.this.getStyledDisplayString(candidate);
        final Image image = N4JSProposalProvider.this.getImage(candidate);
        result = N4JSProposalProvider.this.createCompletionProposal(proposal, displayString, image, contentAssistContext);
        if ((result instanceof ConfigurableCompletionProposal)) {
          final Provider<EObject> _function = () -> {
            return candidate.getEObjectOrProxy();
          };
          final Provider<EObject> provider = _function;
          ((ConfigurableCompletionProposal)result).setProposalContextResource(contentAssistContext.getResource());
          ((ConfigurableCompletionProposal)result).setAdditionalProposalInfo(provider);
          ((ConfigurableCompletionProposal)result).setHover(N4JSProposalProvider.this.getHover());
          ConfigurableCompletionProposalExtensions.setSecondaryMember(((ConfigurableCompletionProposal)result), urisOfSecondaryMembers.contains(candidate.getEObjectURI()));
          final N4JSProposalProvider.ProposalBracketInfo bracketInfo = N4JSProposalProvider.this.computeProposalBracketInfo(candidate, contentAssistContext);
          if ((bracketInfo != null)) {
            ConfigurableCompletionProposalExtensions.setReplacementSuffix(((ConfigurableCompletionProposal)result), bracketInfo.brackets);
            ConfigurableCompletionProposalExtensions.setCursorOffset(((ConfigurableCompletionProposal)result), bracketInfo.cursorOffset);
          }
        }
        N4JSProposalProvider.this.getPriorityHelper().adjustCrossReferencePriority(result, contentAssistContext.getPrefix());
        return result;
      }
    };
  }
  
  @Override
  public StyledString getStyledDisplayString(final IEObjectDescription description) {
    final int version = N4JSResourceDescriptionStrategy.getVersion(description);
    QualifiedName qName = description.getQualifiedName();
    QualifiedName name = description.getName();
    boolean _equals = Objects.equal(qName, name);
    if (_equals) {
      final EObject eObj = description.getEObjectOrProxy();
      final QualifiedName qnOfEObject = this.qualifiedNameProvider.getFullyQualifiedName(eObj);
      QualifiedName _elvis = null;
      if (qnOfEObject != null) {
        _elvis = qnOfEObject;
      } else {
        _elvis = qName;
      }
      qName = _elvis;
    }
    StyledString sString = this.getStyledDisplayString(qName, name, version);
    return sString;
  }
  
  @Override
  protected StyledString getStyledDisplayString(final EObject element, final String qualifiedNameString, final String shortNameString) {
    final int version = this.getTypeVersion(element);
    final QualifiedName qualifiedName = this.qualifiedNameConverter.toQualifiedName(qualifiedNameString);
    final QualifiedName shortName = this.qualifiedNameConverter.toQualifiedName(shortNameString);
    return this.getStyledDisplayString(qualifiedName, shortName, version);
  }
  
  protected StyledString getStyledDisplayString(final QualifiedName qualifiedName, final QualifiedName shortName, final int version) {
    final StyledString result = new StyledString();
    final String shortNameString = shortName.toString();
    int _segmentCount = qualifiedName.getSegmentCount();
    boolean _greaterThan = (_segmentCount > 1);
    if (_greaterThan) {
      String _string = this.qualifiedNameConverter.toString(qualifiedName.skipLast(1));
      final String dashName = (" - " + _string);
      final String lastSegment = qualifiedName.getLastSegment();
      String _xifexpression = null;
      if ((version == 0)) {
        _xifexpression = "";
      } else {
        String _valueOf = String.valueOf(version);
        _xifexpression = (N4IDLGlobals.VERSION_SEPARATOR + _valueOf);
      }
      final String typeVersion = _xifexpression;
      String caption = null;
      String dashInfo = null;
      boolean _endsWith = shortNameString.endsWith(lastSegment);
      if (_endsWith) {
        caption = (lastSegment + typeVersion);
        dashInfo = dashName;
      } else {
        caption = shortNameString;
        dashInfo = (((dashName + " alias for ") + lastSegment) + typeVersion);
      }
      result.append(caption);
      result.append(dashInfo, StyledString.QUALIFIER_STYLER);
    } else {
      result.append(shortNameString);
    }
    return result;
  }
  
  @Override
  protected Image getImage(final IEObjectDescription description) {
    final EClass clazz = description.getEClass();
    final EObject type = EcoreUtil.create(clazz);
    if ((type instanceof TClass)) {
      ((TClass)type).setDeclaredFinal(N4JSResourceDescriptionStrategy.getFinal(description));
      ((TClass)type).setDeclaredAbstract(N4JSResourceDescriptionStrategy.getAbstract(description));
    }
    if ((type instanceof TExportableElement)) {
      String _xifexpression = null;
      boolean _exported = N4JSResourceDescriptionStrategy.getExported(description);
      if (_exported) {
        _xifexpression = " ";
      } else {
        _xifexpression = null;
      }
      ((TExportableElement)type).setExportedName(_xifexpression);
    }
    if ((type instanceof DeclaredTypeWithAccessModifier)) {
      ((DeclaredTypeWithAccessModifier)type).setDeclaredTypeAccessModifier(N4JSResourceDescriptionStrategy.getTypeAccessModifier(description));
    }
    if ((type instanceof TFunction)) {
      ((TFunction)type).setDeclaredTypeAccessModifier(N4JSResourceDescriptionStrategy.getTypeAccessModifier(description));
    }
    if ((type instanceof TMember)) {
      EObject _eObjectOrProxy = description.getEObjectOrProxy();
      final TMember member = ((TMember) _eObjectOrProxy);
      ((TMember)type).setDeclaredFinal(member.isDeclaredFinal());
      if ((type instanceof TMemberWithAccessModifier)) {
        ((TMemberWithAccessModifier)type).setDeclaredMemberAccessModifier(member.getMemberAccessModifier());
      }
    }
    final Image image = this.labelProvider.getImage(type);
    return image;
  }
  
  /**
   * If the element is an instance of {@link TClassifier} this method
   * returns a user-faced string description of the version information.
   * 
   * Otherwise, this method returns an empty string.
   */
  private int getTypeVersion(final EObject element) {
    if ((((!element.eIsProxy()) && (element instanceof TClassifier)) && (((TClassifier) element).getDeclaredVersion() != 0))) {
      return ((TClassifier) element).getDeclaredVersion();
    }
    return 0;
  }
  
  /**
   * Is also used to filter out certain keywords, e.g., operators or (too) short keywords.
   */
  @Override
  public void completeKeyword(final Keyword keyword, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    if (((context.getCurrentModel() instanceof ParameterizedPropertyAccessExpression) || 
      (context.getPreviousModel() instanceof ParameterizedPropertyAccessExpression))) {
      return;
    }
    if (((context.getCurrentModel() instanceof JSXElement) || (context.getPreviousModel() instanceof JSXElement))) {
      return;
    }
    boolean _isAlphabetic = Character.isAlphabetic(keyword.getValue().charAt(0));
    boolean _not = (!_isAlphabetic);
    if (_not) {
      return;
    }
    int _length = keyword.getValue().length();
    boolean _lessThan = (_length < 5);
    if (_lessThan) {
      return;
    }
    super.completeKeyword(keyword, context, acceptor);
  }
  
  /**
   * Produce linked mode aware completion proposals by default.
   * @see N4JSCompletionProposal#setLinkedModeBuilder
   */
  @Override
  protected ConfigurableCompletionProposal doCreateProposal(final String proposal, final StyledString displayString, final Image image, final int replacementOffset, final int replacementLength) {
    int _length = proposal.length();
    return new N4JSCompletionProposal(proposal, replacementOffset, replacementLength, _length, image, displayString, null, null);
  }
  
  private N4JSProposalProvider.ProposalBracketInfo computeProposalBracketInfo(final IEObjectDescription proposedDescription, final ContentAssistContext contentAssistContext) {
    final EClass eClass = proposedDescription.getEClass();
    boolean _isSuperTypeOf = TypesPackage.eINSTANCE.getTFunction().isSuperTypeOf(eClass);
    if (_isSuperTypeOf) {
      return new N4JSProposalProvider.ProposalBracketInfo("()", (-1));
    }
    return null;
  }
}
