/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ide.server.hover;

import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.ide.server.hover.CustomHoverLabelUtil;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.xtend2.lib.StringConcatenation;

@SuppressWarnings("all")
public class N4JSElementSignatureProvider {
  @Inject
  private N4JSTypeSystem ts;
  
  public String get(final EObject o) {
    if ((o instanceof LiteralOrComputedPropertyName)) {
      return this.get(((LiteralOrComputedPropertyName)o).eContainer());
    }
    final EObject id = this.getIdentifiableElement(o);
    final String label = this.doGetLabel(id, o);
    UtilN4.sanitizeForHTML(label);
    return label;
  }
  
  private EObject getIdentifiableElement(final EObject o) {
    EObject _switchResult = null;
    boolean _matched = false;
    if (o instanceof IdentifierRef) {
      _matched=true;
      _switchResult = ((IdentifierRef)o).getId();
    }
    if (!_matched) {
      if (o instanceof ParameterizedPropertyAccessExpression) {
        _matched=true;
        _switchResult = ((ParameterizedPropertyAccessExpression)o).getProperty();
      }
    }
    if (!_matched) {
      if (o instanceof LiteralOrComputedPropertyName) {
        _matched=true;
        _switchResult = ((LiteralOrComputedPropertyName)o).eContainer();
      }
    }
    if (!_matched) {
      _switchResult = o;
    }
    final EObject result = _switchResult;
    return result;
  }
  
  private String _doGetLabel(final EObject o, final EObject ref) {
    final EObject tElem = N4JSASTUtils.getCorrespondingTypeModelElement(o);
    return CustomHoverLabelUtil.getLabel(tElem);
  }
  
  private String _doGetLabel(final IdentifierRef ir, final EObject ref) {
    return this.getLabelFromTypeSystem(ir, ir);
  }
  
  private String _doGetLabel(final TVariable tv, final EObject ref) {
    return this.getLabelFromTypeSystem(tv, ref);
  }
  
  private String _doGetLabel(final ParameterizedPropertyAccessExpression ppae, final EObject ref) {
    return this.getLabelFromTypeSystem(ppae, ppae);
  }
  
  private String _doGetLabel(final VariableDeclaration vd, final EObject ref) {
    String _xifexpression = null;
    if ((vd instanceof ExportedVariableDeclaration)) {
      _xifexpression = this._doGetLabel(((EObject) vd), ref);
    } else {
      _xifexpression = this.getLabelFromTypeSystem(vd, ref);
    }
    return _xifexpression;
  }
  
  private String _doGetLabel(final PropertyNameValuePair nameValuePair, final EObject ref) {
    return this.getLabelFromTypeSystem(nameValuePair, ref);
  }
  
  private String _doGetLabel(final FormalParameter fp, final EObject ref) {
    String _xblockexpression = null;
    {
      String _xifexpression = null;
      boolean _isHasInitializerAssignment = fp.isHasInitializerAssignment();
      if (_isHasInitializerAssignment) {
        _xifexpression = "=…";
      } else {
        _xifexpression = "";
      }
      final String optinonalMarker = _xifexpression;
      String _labelFromTypeSystem = this.getLabelFromTypeSystem(fp, ref);
      _xblockexpression = (_labelFromTypeSystem + optinonalMarker);
    }
    return _xblockexpression;
  }
  
  private String _doGetLabel(final FunctionExpression fe, final EObject ref) {
    return this.getLabelFromTypeSystem(fe, ref);
  }
  
  private String _doGetLabel(final LiteralOrComputedPropertyName name, final EObject ref) {
    EObject _eContainer = name.eContainer();
    if ((_eContainer instanceof TypableElement)) {
      EObject _eContainer_1 = name.eContainer();
      return this.getLabelFromTypeSystem(((TypableElement) _eContainer_1), ref);
    }
    return name.getName();
  }
  
  private String getLabelFromTypeSystem(final TypableElement o, final EObject ref) {
    if (((null == o) || (null == o.eResource()))) {
      return null;
    }
    TypableElement _xifexpression = null;
    if ((ref instanceof TypableElement)) {
      _xifexpression = ((TypableElement)ref);
    } else {
      _xifexpression = o;
    }
    final TypableElement elem = _xifexpression;
    final TypeRef typeRef = this.ts.type(RuleEnvironmentExtensions.newRuleEnvironment(elem), elem);
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _name = this.getName(o);
    _builder.append(_name);
    _builder.append(": ");
    String _typeRefAsString = typeRef.getTypeRefAsString();
    _builder.append(_typeRefAsString);
    return _builder.toString();
  }
  
  private CharSequence _getName(final EObject o) {
    return "";
  }
  
  private CharSequence _getName(final NamedElement namedElement) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    String _name = namedElement.getName();
    _builder.append(_name, " ");
    return _builder;
  }
  
  private CharSequence _getName(final IdentifiableElement identifiableElement) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    String _name = identifiableElement.getName();
    _builder.append(_name, " ");
    return _builder;
  }
  
  private CharSequence _getName(final TVariable tVariable) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    String _xifexpression = null;
    boolean _isConst = tVariable.isConst();
    if (_isConst) {
      _xifexpression = "const";
    } else {
      _xifexpression = "var";
    }
    _builder.append(_xifexpression, " ");
    _builder.append(" ");
    String _name = tVariable.getName();
    _builder.append(_name, " ");
    return _builder;
  }
  
  private String doGetLabel(final EObject nameValuePair, final EObject ref) {
    if (nameValuePair instanceof PropertyNameValuePair) {
      return _doGetLabel((PropertyNameValuePair)nameValuePair, ref);
    } else if (nameValuePair instanceof FormalParameter) {
      return _doGetLabel((FormalParameter)nameValuePair, ref);
    } else if (nameValuePair instanceof FunctionExpression) {
      return _doGetLabel((FunctionExpression)nameValuePair, ref);
    } else if (nameValuePair instanceof IdentifierRef) {
      return _doGetLabel((IdentifierRef)nameValuePair, ref);
    } else if (nameValuePair instanceof VariableDeclaration) {
      return _doGetLabel((VariableDeclaration)nameValuePair, ref);
    } else if (nameValuePair instanceof TVariable) {
      return _doGetLabel((TVariable)nameValuePair, ref);
    } else if (nameValuePair instanceof ParameterizedPropertyAccessExpression) {
      return _doGetLabel((ParameterizedPropertyAccessExpression)nameValuePair, ref);
    } else if (nameValuePair instanceof LiteralOrComputedPropertyName) {
      return _doGetLabel((LiteralOrComputedPropertyName)nameValuePair, ref);
    } else if (nameValuePair != null) {
      return _doGetLabel(nameValuePair, ref);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(nameValuePair, ref).toString());
    }
  }
  
  private CharSequence getName(final EObject tVariable) {
    if (tVariable instanceof TVariable) {
      return _getName((TVariable)tVariable);
    } else if (tVariable instanceof IdentifiableElement) {
      return _getName((IdentifiableElement)tVariable);
    } else if (tVariable instanceof NamedElement) {
      return _getName((NamedElement)tVariable);
    } else if (tVariable != null) {
      return _getName(tVariable);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tVariable).toString());
    }
  }
}
