/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.tests.codegen;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.Objects;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.utils.io.FileDeleter;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Generates the code for a project.
 */
@SuppressWarnings("all")
public class Project {
  /**
   * Represents a source folder that has a name and contains modules.
   */
  public static class SourceFolder {
    private String name;
    
    private List<org.eclipse.n4js.tests.codegen.Module> modules;
    
    /**
     * Creates a new instance with the given parameters.
     * 
     * @param name the name of this source folder
     */
    public SourceFolder(final String name) {
      this.name = Objects.<String>requireNonNull(name);
    }
    
    /**
     * Adds the given module to the source folder to created.
     * 
     * @param module the module to add
     * 
     * @return this source folder
     */
    public Project.SourceFolder addModule(final org.eclipse.n4js.tests.codegen.Module module) {
      if ((this.modules == null)) {
        this.modules = CollectionLiterals.<org.eclipse.n4js.tests.codegen.Module>newLinkedList();
      }
      this.modules.add(Objects.<org.eclipse.n4js.tests.codegen.Module>requireNonNull(module));
      return this;
    }
    
    /**
     * Creates this source folder within the given parent directory, which must exist.
     * 
     * This method first creates a new folder within the given parent directory, and then
     * it creates all of its modules within that folder by calling their {@link Module#create(File)}
     * function with the newly created folder as the parameter.
     * 
     * @param parentDirectory a file representing the parent directory of this source folder
     */
    public void create(final File parentDirectory) {
      try {
        Objects.<File>requireNonNull(parentDirectory);
        boolean _exists = parentDirectory.exists();
        boolean _not = (!_exists);
        if (_not) {
          throw new IOException((("Directory \'" + parentDirectory) + "\' does not exist"));
        }
        boolean _isDirectory = parentDirectory.isDirectory();
        boolean _not_1 = (!_isDirectory);
        if (_not_1) {
          throw new IOException((("\'" + parentDirectory) + "\' is not a directory"));
        }
        File sourceFolder = new File(parentDirectory, this.name);
        sourceFolder.mkdir();
        for (final org.eclipse.n4js.tests.codegen.Module module : this.modules) {
          module.create(sourceFolder);
        }
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }
  
  private String projectName;
  
  private String vendorId;
  
  private String vendorName;
  
  private ProjectType projectType = ProjectType.LIBRARY;
  
  private String projectVersion = "1.0.0";
  
  private String outputFolder = "src-gen";
  
  private List<Project.SourceFolder> sourceFolders;
  
  private List<Project> projectDependencies;
  
  /**
   * Creates a new instance with the given parameters.
   * 
   * @param projectName the project ID
   * @param vendorId the vendor ID
   * @param vendorName the vendor name
   */
  public Project(final String projectName, final String vendorId, final String vendorName) {
    this.projectName = Objects.<String>requireNonNull(projectName);
    this.vendorId = Objects.<String>requireNonNull(vendorId);
    this.vendorName = Objects.<String>requireNonNull(vendorName);
  }
  
  /**
   * Returns the project name.
   * 
   * @return the project name.
   */
  public String getProjectName() {
    return this.projectName;
  }
  
  /**
   * Sets the project type.
   * 
   * @param projectType the project type to set
   */
  public Project setType(final ProjectType projectType) {
    this.projectType = projectType;
    return this;
  }
  
  /**
   * Sets the project version.
   * 
   * @param projectVersion the project version
   */
  public Project setVersion(final String projectVersion) {
    this.projectVersion = projectVersion;
    return this;
  }
  
  /**
   * Sets the output folder.
   * 
   * @param outputFolder the output folder to set
   */
  public Project setOutputFolder(final String outputFolder) {
    this.outputFolder = outputFolder;
    return this;
  }
  
  /**
   * Creates a source folder with the given name to this project.
   * 
   * @param name the name of the source folder to add
   * 
   * @return the added source folder
   */
  public Project.SourceFolder createSourceFolder(final String name) {
    final Project.SourceFolder result = new Project.SourceFolder(name);
    this.addSourceFolder(result);
    return result;
  }
  
  /**
   * Adds a source folder to this project.
   * 
   * @param sourceFolder the source folder to add
   */
  public Project addSourceFolder(final Project.SourceFolder sourceFolder) {
    if ((this.sourceFolders == null)) {
      this.sourceFolders = CollectionLiterals.<Project.SourceFolder>newLinkedList();
    }
    this.sourceFolders.add(Objects.<Project.SourceFolder>requireNonNull(sourceFolder));
    return this;
  }
  
  /**
   * Adds a project dependency to this project.
   * 
   * @param projectDependency the project dependency to add
   */
  public Project addProjectDependency(final Project projectDependency) {
    if ((this.projectDependencies == null)) {
      this.projectDependencies = CollectionLiterals.<Project>newLinkedList();
    }
    this.projectDependencies.add(Objects.<Project>requireNonNull(projectDependency));
    return this;
  }
  
  /**
   * Generates the {@link IN4JSProject#PACKAGE_JSON} for this project.
   */
  public CharSequence generate() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("{");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("\"name\": \"");
    _builder.append(this.projectName, "\t");
    _builder.append("\",");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("\"version\": \"");
    _builder.append(this.projectVersion, "\t");
    _builder.append("\",");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("\"n4js\": {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("\"vendorId\": \"");
    _builder.append(this.vendorId, "\t\t");
    _builder.append("\",");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("\"vendorName\": \"");
    _builder.append(this.vendorName, "\t\t");
    _builder.append("\",");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("\"projectType\": \"");
    String _projectTypeToString = Project.projectTypeToString(this.projectType);
    _builder.append(_projectTypeToString, "\t\t");
    _builder.append("\",");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(this.outputFolder);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("\"output\": \"");
        _builder.append(this.outputFolder, "\t\t");
        _builder.append("\"");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t\t");
    {
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(this.sourceFolders);
      boolean _not_1 = (!_isNullOrEmpty_1);
      if (_not_1) {
        _builder.append(",\"sources\": {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        _builder.append("\t\t");
        _builder.append("\"source\": [");
        _builder.newLine();
        {
          boolean _hasElements = false;
          for(final Project.SourceFolder sourceFolder : this.sourceFolders) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(",", "\t\t\t\t\t");
            }
            _builder.append("\t\t");
            _builder.append("\t\t\t");
            _builder.append("\"");
            _builder.append(sourceFolder.name, "\t\t\t\t\t");
            _builder.append("\"");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t\t");
        _builder.append("\t\t");
        _builder.append("]");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    {
      boolean _isNullOrEmpty_2 = IterableExtensions.isNullOrEmpty(this.projectDependencies);
      boolean _not_2 = (!_isNullOrEmpty_2);
      if (_not_2) {
        _builder.append(",\"dependencies\": {");
        _builder.newLineIfNotEmpty();
        {
          boolean _hasElements_1 = false;
          for(final Project dep : this.projectDependencies) {
            if (!_hasElements_1) {
              _hasElements_1 = true;
            } else {
              _builder.appendImmediate(",", "\t\t\t");
            }
            _builder.append("\t");
            _builder.append("\t\t");
            _builder.append("\"");
            _builder.append(dep.projectName, "\t\t\t");
            _builder.append("\": \"*\"");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }
  
  private static String projectTypeToString(final ProjectType type) {
    String _switchResult = null;
    if (type != null) {
      switch (type) {
        case API:
          _switchResult = "api";
          break;
        case APPLICATION:
          _switchResult = "application";
          break;
        case LIBRARY:
          _switchResult = "library";
          break;
        case PROCESSOR:
          _switchResult = "processor";
          break;
        case RUNTIME_ENVIRONMENT:
          _switchResult = "runtimeEnvironment";
          break;
        case RUNTIME_LIBRARY:
          _switchResult = "runtimeLibrary";
          break;
        case TEST:
          _switchResult = "test";
          break;
        case PLAINJS:
          _switchResult = "plainjs";
          break;
        case VALIDATION:
          _switchResult = "validation";
          break;
        case DEFINITION:
          _switchResult = "definition";
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  /**
   * Creates this project in the given parent directory, which must exist.
   * 
   * This method first creates a directory with the same name as the {@link #projectName} within
   * the given parent directory. If there already exists a file or directory with that name
   * within the given parent directory, that file or directory will be (recursively) deleted.
   * 
   * Afterward, the manifest file and the source folders are created within the newly created
   * project directory.
   * 
   * @param parentDirectoryPath the path to the parent directory
   * 
   * @return the project directory
   */
  public File create(final Path parentDirectoryPath) {
    try {
      File parentDirectory = Objects.<Path>requireNonNull(parentDirectoryPath).toFile();
      boolean _exists = parentDirectory.exists();
      boolean _not = (!_exists);
      if (_not) {
        throw new IOException((("\'" + parentDirectory) + "\' does not exist"));
      }
      boolean _isDirectory = parentDirectory.isDirectory();
      boolean _not_1 = (!_isDirectory);
      if (_not_1) {
        throw new IOException((("\'" + parentDirectory) + "\' is not a directory"));
      }
      final File projectDirectory = new File(parentDirectory, this.projectName);
      boolean _exists_1 = projectDirectory.exists();
      if (_exists_1) {
        FileDeleter.delete(projectDirectory);
      }
      projectDirectory.mkdir();
      this.createProjectDescriptionFile(projectDirectory);
      this.createModules(projectDirectory);
      return projectDirectory;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void createProjectDescriptionFile(final File parentDirectory) {
    try {
      final File filePath = new File(parentDirectory, IN4JSProject.PACKAGE_JSON);
      FileWriter out = null;
      try {
        FileWriter _fileWriter = new FileWriter(filePath);
        out = _fileWriter;
        out.write(this.generate().toString());
      } finally {
        if ((out != null)) {
          out.close();
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void createModules(final File parentDirectory) {
    for (final Project.SourceFolder sourceFolder : this.sourceFolders) {
      sourceFolder.create(parentDirectory);
    }
  }
}
