/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ts.validation;

import com.google.inject.Singleton;
import java.util.Arrays;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.ts.types.BuiltInType;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TStructGetter;
import org.eclipse.n4js.ts.types.TStructSetter;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeAccessModifier;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.TypingStrategy;

/**
 * Helper returning the keyword of a given AST or type element, e.g., "class" for a class declaration.
 */
@Singleton
@SuppressWarnings("all")
public class TypesKeywordProvider {
  public String keywordWithIndefiniteArticle(final EObject elem) {
    final String keyword = this.keyword(elem);
    char _xifexpression = (char) 0;
    int _length = keyword.length();
    boolean _greaterThan = (_length > 0);
    if (_greaterThan) {
      _xifexpression = Character.toLowerCase(keyword.charAt(0));
    }
    final char firstChar = _xifexpression;
    int _indexOf = "aeiou".indexOf(firstChar);
    final boolean startsWithVowel = (_indexOf >= 0);
    String _xifexpression_1 = null;
    if (startsWithVowel) {
      _xifexpression_1 = ("an " + keyword);
    } else {
      _xifexpression_1 = ("a " + keyword);
    }
    return _xifexpression_1;
  }
  
  public String keyword(final EObject elem, final TypingStrategy typingStrategy) {
    if ((typingStrategy == TypingStrategy.STRUCTURAL_FIELD_INITIALIZER)) {
      EClass _switchResult = null;
      boolean _matched = false;
      if (elem instanceof TStructGetter) {
        _matched=true;
        _switchResult = TypesPackage.eINSTANCE.getTStructSetter();
      }
      if (!_matched) {
        if (elem instanceof TStructSetter) {
          _matched=true;
          _switchResult = TypesPackage.eINSTANCE.getTStructGetter();
        }
      }
      if (!_matched) {
        if (elem instanceof TGetter) {
          _matched=true;
          _switchResult = TypesPackage.eINSTANCE.getTSetter();
        }
      }
      if (!_matched) {
        if (elem instanceof TSetter) {
          _matched=true;
          _switchResult = TypesPackage.eINSTANCE.getTGetter();
        }
      }
      final EClass replacementType = _switchResult;
      if ((replacementType != null)) {
        return this.keyword(TypesFactory.eINSTANCE.create(replacementType));
      }
    }
    return this.keyword(elem);
  }
  
  protected String _keyword(final EObject eobj) {
    final String modelName = eobj.eClass().getName();
    int _length = modelName.length();
    int _plus = (_length + 1);
    final StringBuilder strb = new StringBuilder(_plus);
    for (int i = 0; (i < modelName.length()); i++) {
      {
        final char c = modelName.charAt(i);
        boolean _isUpperCase = Character.isUpperCase(c);
        if (_isUpperCase) {
          if ((i > 0)) {
            strb.append(" ");
          }
          strb.append(Character.toLowerCase(c));
        } else {
          strb.append(c);
        }
      }
    }
    return strb.toString();
  }
  
  protected String _keyword(final TClass n4Class) {
    return "class";
  }
  
  protected String _keyword(final TEnum tEnum) {
    return "enum";
  }
  
  protected String _keyword(final TInterface n4Interface) {
    return "interface";
  }
  
  protected String _keyword(final TClassifier n4Classifier) {
    return "classifier";
  }
  
  protected String _keyword(final TMember n4Member) {
    return "member";
  }
  
  protected String _keyword(final TStructField structField) {
    return "structural field";
  }
  
  protected String _keyword(final TStructGetter structGetter) {
    return "structural getter";
  }
  
  protected String _keyword(final TStructSetter structSetter) {
    return "structural setter";
  }
  
  protected String _keyword(final TField n4Field) {
    return "field";
  }
  
  protected String _keyword(final TMethod tMethod) {
    String _xifexpression = null;
    boolean _isConstructor = tMethod.isConstructor();
    if (_isConstructor) {
      _xifexpression = "constructor";
    } else {
      _xifexpression = "method";
    }
    return _xifexpression;
  }
  
  protected String _keyword(final TFunction function) {
    return "function";
  }
  
  protected String _keyword(final TGetter n4Getter) {
    return "getter";
  }
  
  protected String _keyword(final TSetter n4Setter) {
    return "setter";
  }
  
  protected String _keyword(final TVariable variable) {
    return "variable";
  }
  
  protected String _keyword(final TFormalParameter parameter) {
    return "parameter";
  }
  
  protected String _keyword(final TypeVariable variable) {
    return "type variable";
  }
  
  protected String _keyword(final PrimitiveType primitive) {
    return "primitive";
  }
  
  protected String _keyword(final BuiltInType primitive) {
    return "built-in type";
  }
  
  protected String _keyword(final Type other) {
    return "type";
  }
  
  protected String _keyword(final TypeAccessModifier accessModifier) {
    String _switchResult = null;
    if (accessModifier != null) {
      switch (accessModifier) {
        case PUBLIC_INTERNAL:
          _switchResult = "@Internal public";
          break;
        default:
          _switchResult = accessModifier.getName();
          break;
      }
    } else {
      _switchResult = accessModifier.getName();
    }
    return _switchResult;
  }
  
  protected String _keyword(final MemberAccessModifier accessModifier) {
    String _switchResult = null;
    if (accessModifier != null) {
      switch (accessModifier) {
        case PUBLIC_INTERNAL:
          _switchResult = "@Internal public";
          break;
        case PROTECTED_INTERNAL:
          _switchResult = "@Internal protected";
          break;
        default:
          _switchResult = accessModifier.getName();
          break;
      }
    } else {
      _switchResult = accessModifier.getName();
    }
    return _switchResult;
  }
  
  public String keyword(final Object n4Class) {
    if (n4Class instanceof TClass) {
      return _keyword((TClass)n4Class);
    } else if (n4Class instanceof TInterface) {
      return _keyword((TInterface)n4Class);
    } else if (n4Class instanceof TMethod) {
      return _keyword((TMethod)n4Class);
    } else if (n4Class instanceof TStructGetter) {
      return _keyword((TStructGetter)n4Class);
    } else if (n4Class instanceof TStructSetter) {
      return _keyword((TStructSetter)n4Class);
    } else if (n4Class instanceof PrimitiveType) {
      return _keyword((PrimitiveType)n4Class);
    } else if (n4Class instanceof TClassifier) {
      return _keyword((TClassifier)n4Class);
    } else if (n4Class instanceof TEnum) {
      return _keyword((TEnum)n4Class);
    } else if (n4Class instanceof TFunction) {
      return _keyword((TFunction)n4Class);
    } else if (n4Class instanceof TGetter) {
      return _keyword((TGetter)n4Class);
    } else if (n4Class instanceof TSetter) {
      return _keyword((TSetter)n4Class);
    } else if (n4Class instanceof TStructField) {
      return _keyword((TStructField)n4Class);
    } else if (n4Class instanceof BuiltInType) {
      return _keyword((BuiltInType)n4Class);
    } else if (n4Class instanceof TField) {
      return _keyword((TField)n4Class);
    } else if (n4Class instanceof TypeVariable) {
      return _keyword((TypeVariable)n4Class);
    } else if (n4Class instanceof TVariable) {
      return _keyword((TVariable)n4Class);
    } else if (n4Class instanceof Type) {
      return _keyword((Type)n4Class);
    } else if (n4Class instanceof TFormalParameter) {
      return _keyword((TFormalParameter)n4Class);
    } else if (n4Class instanceof TMember) {
      return _keyword((TMember)n4Class);
    } else if (n4Class instanceof MemberAccessModifier) {
      return _keyword((MemberAccessModifier)n4Class);
    } else if (n4Class instanceof TypeAccessModifier) {
      return _keyword((TypeAccessModifier)n4Class);
    } else if (n4Class instanceof EObject) {
      return _keyword((EObject)n4Class);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(n4Class).toString());
    }
  }
}
