/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling;

import com.google.inject.Inject;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.IRegion;
import org.eclipse.n4js.ide.server.hover.N4JSElementSignatureProvider;
import org.eclipse.n4js.jsdoc.JSDoc2HoverSerializer;
import org.eclipse.n4js.jsdoc.N4JSDocletParser;
import org.eclipse.n4js.jsdoc.dom.Doclet;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.ui.labeling.TypesHoverProvider;
import org.eclipse.n4js.ui.internal.N4JSActivator;
import org.eclipse.n4js.ui.labeling.helper.ImageFileNameCalculationHelper;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.ui.editor.hover.html.DefaultEObjectHoverProvider;
import org.eclipse.xtext.ui.editor.hover.html.XtextBrowserInformationControlInput;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.osgi.framework.Bundle;

@SuppressWarnings("all")
public class N4JSHoverProvider extends DefaultEObjectHoverProvider {
  @Inject
  @Extension
  private N4JSElementKeywordProvider _n4JSElementKeywordProvider;
  
  @Inject
  private N4JSElementSignatureProvider signatureProvider;
  
  @Inject
  private TypesHoverProvider typesHoverProvider;
  
  @Inject
  private OperationCanceledManager cancelManager;
  
  @Inject
  private N4JSDocletParser docletParser;
  
  @Override
  protected String getFirstLine(final EObject o) {
    if ((o instanceof LiteralOrComputedPropertyName)) {
      return this.getFirstLine(((LiteralOrComputedPropertyName)o).eContainer());
    }
    final EObject id = this.getIdentifiableElement(o);
    final URL image = this.getImageURL(id);
    final String keyword = this._n4JSElementKeywordProvider.keyword(id);
    final String label = this.getLabel(o);
    return TypesHoverProvider.composeFirstLine(image, keyword, label);
  }
  
  private EObject getIdentifiableElement(final EObject o) {
    EObject _switchResult = null;
    boolean _matched = false;
    if (o instanceof IdentifierRef) {
      _matched=true;
      _switchResult = ((IdentifierRef)o).getId();
    }
    if (!_matched) {
      if (o instanceof ParameterizedPropertyAccessExpression) {
        _matched=true;
        _switchResult = ((ParameterizedPropertyAccessExpression)o).getProperty();
      }
    }
    if (!_matched) {
      if (o instanceof LiteralOrComputedPropertyName) {
        _matched=true;
        _switchResult = ((LiteralOrComputedPropertyName)o).eContainer();
      }
    }
    if (!_matched) {
      _switchResult = o;
    }
    final EObject result = _switchResult;
    return result;
  }
  
  @Override
  protected String getLabel(final EObject o) {
    String _xblockexpression = null;
    {
      final String label = this.signatureProvider.get(o);
      _xblockexpression = UtilN4.sanitizeForHTML(label);
    }
    return _xblockexpression;
  }
  
  @Override
  protected String getDocumentation(final EObject o) {
    try {
      final EObject id = this.getIdentifiableElement(o);
      String jsdocString = super.getDocumentation(id);
      if ((jsdocString == null)) {
        return null;
      }
      final Doclet doclet = this.docletParser.parse(jsdocString);
      final String hoverHTML = JSDoc2HoverSerializer.toJSDocString(doclet);
      return hoverHTML;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        return ("Error generating documentation:  " + ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Override
  protected boolean hasHover(final EObject o) {
    return this.doHasHover(o);
  }
  
  private boolean _doHasHover(final EObject o) {
    final EObject tElem = N4JSASTUtils.getCorrespondingTypeModelElement(o);
    boolean _xifexpression = false;
    if ((null == tElem)) {
      _xifexpression = super.hasHover(o);
    } else {
      _xifexpression = this.typesHoverProvider.hasHover(tElem);
    }
    return _xifexpression;
  }
  
  private boolean _doHasHover(final ParameterizedPropertyAccessExpression ppae) {
    return true;
  }
  
  private boolean _doHasHover(final IdentifierRef identifierRef) {
    return true;
  }
  
  private boolean _doHasHover(final VariableDeclaration vd) {
    return true;
  }
  
  private boolean _doHasHover(final PropertyNameValuePair nameValuePair) {
    return true;
  }
  
  private boolean _doHasHover(final FormalParameter fp) {
    return true;
  }
  
  private boolean _doHasHover(final FunctionExpression fe) {
    return true;
  }
  
  private boolean _doHasHover(final N4TypeDeclaration md) {
    return true;
  }
  
  private boolean _doHasHover(final LiteralOrComputedPropertyName name) {
    EObject _eContainer = name.eContainer();
    return (_eContainer instanceof N4MemberDeclaration);
  }
  
  @Override
  protected XtextBrowserInformationControlInput getHoverInfo(final EObject element, final IRegion hoverRegion, final XtextBrowserInformationControlInput previous) {
    Object _xtrycatchfinallyexpression = null;
    try {
      return super.getHoverInfo(element, hoverRegion, previous);
    } catch (final Throwable _t) {
      if (_t instanceof Throwable) {
        final Throwable t = (Throwable)_t;
        Object _xifexpression = null;
        boolean _isOperationCanceledException = this.cancelManager.isOperationCanceledException(t);
        boolean _not = (!_isOperationCanceledException);
        if (_not) {
          throw Exceptions.sneakyThrow(t);
        }
        _xtrycatchfinallyexpression = _xifexpression;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return ((XtextBrowserInformationControlInput)_xtrycatchfinallyexpression);
  }
  
  @Inject
  private ImageFileNameCalculationHelper h;
  
  private URL getImageURL(final EObject obj) {
    final String fn = this.h.getImageFileName(obj);
    if ((fn != null)) {
      final int lastDotIndex = fn.lastIndexOf(".");
      final String name = fn.substring(0, lastDotIndex);
      final String extn = fn.substring(lastDotIndex);
      final String fnHighRes = ((name + "@2x") + extn);
      final Bundle bundle = N4JSActivator.getInstance().getBundle();
      final Path folder = new Path("icons/");
      final IPath pathHighRes = folder.append(fnHighRes);
      final URL urlHighRes = FileLocator.find(bundle, pathHighRes, null);
      if ((urlHighRes != null)) {
        try {
          final URL file = FileLocator.toFileURL(urlHighRes);
          boolean _exists = Files.exists(Paths.get(file.toURI()));
          if (_exists) {
            return file;
          }
        } catch (final Throwable _t) {
          if (_t instanceof Exception) {
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
      final IPath path = folder.append(fn);
      final URL url = FileLocator.find(bundle, path, null);
      if ((url != null)) {
        try {
          return FileLocator.toFileURL(url);
        } catch (final Throwable _t) {
          if (_t instanceof IOException) {
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
    }
    return null;
  }
  
  private boolean doHasHover(final EObject nameValuePair) {
    if (nameValuePair instanceof PropertyNameValuePair) {
      return _doHasHover((PropertyNameValuePair)nameValuePair);
    } else if (nameValuePair instanceof FormalParameter) {
      return _doHasHover((FormalParameter)nameValuePair);
    } else if (nameValuePair instanceof FunctionExpression) {
      return _doHasHover((FunctionExpression)nameValuePair);
    } else if (nameValuePair instanceof IdentifierRef) {
      return _doHasHover((IdentifierRef)nameValuePair);
    } else if (nameValuePair instanceof N4TypeDeclaration) {
      return _doHasHover((N4TypeDeclaration)nameValuePair);
    } else if (nameValuePair instanceof VariableDeclaration) {
      return _doHasHover((VariableDeclaration)nameValuePair);
    } else if (nameValuePair instanceof ParameterizedPropertyAccessExpression) {
      return _doHasHover((ParameterizedPropertyAccessExpression)nameValuePair);
    } else if (nameValuePair instanceof LiteralOrComputedPropertyName) {
      return _doHasHover((LiteralOrComputedPropertyName)nameValuePair);
    } else if (nameValuePair != null) {
      return _doHasHover(nameValuePair);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(nameValuePair).toString());
    }
  }
}
