/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling;

import com.google.inject.Inject;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.n4js.ui.labeling.helper.ImageCalculationHelper;
import org.eclipse.n4js.ui.labeling.helper.LabelCalculationHelper;
import org.eclipse.n4js.ui.labeling.helper.StyledTextCalculationHelper;
import org.eclipse.n4js.ui.outline.N4JSOutlineTreeProvider;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.xtext.ui.label.AbstractLabelProvider;
import org.eclipse.xtext.util.CancelIndicator;

/**
 * This provider is bound as default label provider and is used e.g. for the outline view.
 * As labels in content assist and hyperlinks may look different (as different use case),
 * there are {@link N4JSContentAssistLabelProvider} and {@link N4JSHyperlinkLabelProvider}.
 * <br/><br/>
 * Everytime \@Inject ILabelProvider is used this label provider is injected.
 * <br/><br/>
 * see http://www.eclipse.org/Xtext/documentation.html#labelProvider
 * <br/><br/>
 * Main use case is the use in {@link N4JSOutlineTreeProvider}.
 * <br/><br/>
 * This class just delegates the calculation of labels and images. Please not that the
 * {@link StyledTextCalculationHelper} and {@link LabelCalculationHelper} creates in
 * combination the label, e.g. for fields, the {@link LabelCalculationHelper} writes the name
 * of the field and {@link StyledTextCalculationHelper} then just appends the styled string
 * with the type information. {@link ImageCalculationHelper} is used to calculate the icons
 * next to the labels.
 * <br/><br/>
 * Although this class delegates tasks the dependent classes sometimes requires access to
 * methods in the super class of this class. Therefore the dependent classes get this label
 * provider set and this class overrides some of its super class methods to raise visibility
 * from protected to public.
 */
@SuppressWarnings("all")
public class N4JSLabelProvider extends AbstractLabelProvider {
  @Inject
  private LabelCalculationHelper labelCalculationHelper;
  
  private ImageCalculationHelper imageCalculationHelper;
  
  private StyledTextCalculationHelper styledTextCalculationHelper;
  
  @Inject
  public N4JSLabelProvider(final AdapterFactoryLabelProvider delegate) {
    super(delegate);
  }
  
  @Inject
  public void setImageCalculationHelper(final ImageCalculationHelper imageCalculationHelper) {
    this.imageCalculationHelper = imageCalculationHelper;
    this.imageCalculationHelper.setLabelProvider(this);
  }
  
  @Inject
  public N4JSLabelProvider setStyledTextCalculationHelper(final StyledTextCalculationHelper styledTextCalculationHelper) {
    N4JSLabelProvider _xblockexpression = null;
    {
      this.styledTextCalculationHelper = styledTextCalculationHelper;
      _xblockexpression = this.styledTextCalculationHelper.setLabelProvider(this);
    }
    return _xblockexpression;
  }
  
  public CancelIndicator cancelIndicator = CancelIndicator.NullImpl;
  
  /**
   * A "real" {@link CancelIndicator} (ie, the argument) is only available during an invocation of the caller of this method.
   * That caller is expected to follow this protocol:
   * <ol>
   * <li>establish the indicator on entry, by invoking this method.</li>
   * <li>performing operations (which may access the cancel indicator put into effect above).</li>
   * <li>removing the cancel indicator on exit (in a finally block).</li>
   * </ol>
   * 
   * @see #removeCancelIndicator
   */
  public void establishCancelIndicator(final CancelIndicator indicator) {
    CancelIndicator _xifexpression = null;
    if ((null != indicator)) {
      _xifexpression = indicator;
    } else {
      _xifexpression = CancelIndicator.NullImpl;
    }
    this.cancelIndicator = _xifexpression;
  }
  
  public void removeCancelIndicator() {
    this.cancelIndicator = CancelIndicator.NullImpl;
  }
  
  /**
   * doGetText -> manually dispatch label text via dispatchDoGetText
   */
  @Override
  public String doGetText(final Object element) {
    return this.labelCalculationHelper.dispatchDoGetText(element);
  }
  
  /**
   * manually dispatch image object via dispatchDoGetImage
   */
  @Override
  public ImageDescriptor doGetImage(final Object element) {
    return this.imageCalculationHelper.dispatchDoGetImage(element);
  }
  
  /**
   * manually dispatch styled text via dispatchGetStyledText
   */
  @Override
  public StyledString getStyledText(final Object element) {
    return this.styledTextCalculationHelper.dispatchGetStyledText(element);
  }
  
  public StyledString getSuperStyledText(final Object element) {
    return super.getStyledText(element);
  }
  
  public ImageDescriptor asImageDescriptor(final String name) {
    return super.convertToImageDescriptor(name);
  }
  
  public static ImageDescriptor getDefaultImageDescriptor() {
    return PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_ELEMENT);
  }
}
