/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling;

import com.google.common.base.Strings;
import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.ui.labeling.N4JSHoverProvider;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;

/**
 * Extended hover provider which unlike {@link N4JSHoverProvider}
 * handles non-declared AST elements such as expressions.
 */
@SuppressWarnings("all")
public class N4JSTypeInformationHoverProvider extends N4JSHoverProvider {
  @Inject
  @Extension
  private N4JSTypeSystem _n4JSTypeSystem;
  
  @Override
  protected boolean hasHover(final EObject obj) {
    return this.doHasHover(obj);
  }
  
  @Override
  protected String getLabel(final EObject obj) {
    return UtilN4.sanitizeForHTML(Strings.nullToEmpty(this.doGetLabel(obj)));
  }
  
  @Override
  protected String getFirstLine(final EObject obj) {
    final String label = this.getLabel(obj);
    String _xifexpression = null;
    boolean _isNullOrEmpty = Strings.isNullOrEmpty(label);
    if (_isNullOrEmpty) {
      _xifexpression = null;
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Type of selected expression:<br>");
      {
        boolean _contains = label.contains("<b>");
        boolean _not = (!_contains);
        if (_not) {
          _builder.append("<b>");
          _builder.append(label);
          _builder.append("</b>");
        } else {
          _builder.append(label);
        }
      }
      _xifexpression = _builder.toString();
    }
    return _xifexpression;
  }
  
  /**
   * Copied from super class to add support interrupting the hover information calculation
   * when type information does not exist for an expression.
   */
  @Override
  protected String getHoverInfoAsHtml(final EObject obj) {
    boolean _hasHover = this.hasHover(obj);
    boolean _not = (!_hasHover);
    if (_not) {
      return null;
    }
    final String firstLine = this.getFirstLine(obj);
    boolean _isNullOrEmpty = Strings.isNullOrEmpty(firstLine);
    if (_isNullOrEmpty) {
      return null;
    }
    final StringBuilder sb = new StringBuilder(firstLine);
    final String documentation = this.getDocumentation(obj);
    if (((null != documentation) && (0 < documentation.length()))) {
      sb.append("<p>");
      sb.append(documentation);
      sb.append("</p>");
    }
    return sb.toString();
  }
  
  private boolean _doHasHover(final EObject obj) {
    return super.hasHover(obj);
  }
  
  private boolean _doHasHover(final Expression expression) {
    return true;
  }
  
  private String _doGetLabel(final EObject obj) {
    return super.getLabel(obj);
  }
  
  private String _doGetLabel(final Expression expression) {
    Resource _eResource = expression.eResource();
    boolean _tripleEquals = (null == _eResource);
    if (_tripleEquals) {
      return null;
    }
    final TypeRef type = this._n4JSTypeSystem.type(RuleEnvironmentExtensions.newRuleEnvironment(expression), expression);
    String _xifexpression = null;
    if ((null == type)) {
      _xifexpression = null;
    } else {
      _xifexpression = type.getTypeRefAsString();
    }
    return _xifexpression;
  }
  
  private boolean doHasHover(final EObject expression) {
    if (expression instanceof Expression) {
      return _doHasHover((Expression)expression);
    } else if (expression != null) {
      return _doHasHover(expression);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(expression).toString());
    }
  }
  
  private String doGetLabel(final EObject expression) {
    if (expression instanceof Expression) {
      return _doGetLabel((Expression)expression);
    } else if (expression != null) {
      return _doGetLabel(expression);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(expression).toString());
    }
  }
}
