/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling.helper;

import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4EnumLiteral;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.resource.N4JSResourceDescriptionStrategy;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnumLiteral;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeAccessModifier;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ui.labeling.EObjectWithContext;
import org.eclipse.n4js.ui.labeling.N4JSLabelProvider;
import org.eclipse.n4js.ui.labeling.helper.ImageDescriptionHelper;
import org.eclipse.n4js.ui.labeling.helper.ImageFileNameCalculationHelper;
import org.eclipse.n4js.ui.labeling.helper.N4JSImageDescriptionLibrary;
import org.eclipse.n4js.ui.typesearch.TypeSearchKind;
import org.eclipse.xtext.Keyword;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.ui.label.AbstractLabelProvider;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This helper class serves as replacement for the polymorphic dispatch done
 * by {@link AbstractLabelProvider} in favor of
 * Xtend dispatch methods. Here the dispatch of icons to be shown e.g. in
 * the outline view is done. It is called in {@link N4JSLabelProvider#doGetImage}.
 * <br /><br />
 * General pattern is to delegate from an AST element to the types elmenent
 * as in the types model the required information to calculate the decorated
 * image is better provided.
 * <br /><br />
 * See /org.eclipse.n4js.ui/icons/icons_origin.txt for the origins of the
 * icons used to create the images.
 * <br /><br />
 * In some cases the image file provided can be used as it is but in most cases
 * some additional image decorators have to be added to visualize properties
 * of the represented element, e.g. accessibility, static/non-static, final,
 * constant, abstract, constructor. Some images like for getter/setter are
 * created using decorators as no specific image file was available. For classes
 * and interfaces there were specific images available when they are declared
 * public or private for roles this is done with decorators.
 * <br /><br />
 * For all unexpected elements a default image is shown.
 * <br /><br />
 * Decorators can be placed top left, top right, bottom right, bottom left.
 * Its also possible to add multiple decorators in one corner. This is done by
 * using {@code N4JSDecoratorRow}. Actually the creation of
 * image descriptions is completely layed out in {@link ImageDescriptionHelper}.
 * The selection of image files used as main images is done in
 * {@link ImageFileNameCalculationHelper}. {@link N4JSImageDescriptionLibrary}
 * holds the prepared image descriptions to be used as decorators.
 * <br /><br />
 * In the bottom left corner an error or warning decorator is added if the element
 * represented has corresponding validation issues.
 */
@SuppressWarnings("all")
public class ImageCalculationHelper {
  @Inject
  @Extension
  private ImageDescriptionHelper imageDescriptionHelper;
  
  @Inject
  @Extension
  private ImageFileNameCalculationHelper imageFileNameCalculationHelper;
  
  @Extension
  private N4JSImageDescriptionLibrary imageDescriptionLibrary;
  
  public void setLabelProvider(final N4JSLabelProvider provider) {
    this.imageDescriptionHelper.setLabelProvider(provider);
    this.imageDescriptionLibrary = this.imageDescriptionHelper.getImageDescriptionLibrary();
  }
  
  /**
   * Should not happen as this will lead to consequential errors, still added here to make helper more robust.
   */
  protected ImageDescriptor _dispatchDoGetImage(final Void _null) {
    return null;
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final EObjectWithContext eObjectWithContext) {
    return this.dispatchDoGetImage(eObjectWithContext.obj);
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final N4ClassifierDeclaration n4ClassifierDeclaration) {
    return this.dispatchDoGetImage(n4ClassifierDeclaration.getDefinedType());
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final N4MemberDeclaration n4MemberDeclaration) {
    return this.dispatchDoGetImage(n4MemberDeclaration.getDefinedTypeElement());
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final N4GetterDeclaration getter) {
    return this.dispatchDoGetImage(getter.getDefinedGetter());
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final N4SetterDeclaration setter) {
    return this.dispatchDoGetImage(setter.getDefinedSetter());
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final FunctionDeclaration functionDeclaration) {
    return this.dispatchDoGetImage(functionDeclaration.getDefinedType());
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final ExportedVariableDeclaration variableDeclaration) {
    return this.dispatchDoGetImage(variableDeclaration.getDefinedVariable());
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final ExportedVariableStatement vs) {
    final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(vs, this.imageFileNameCalculationHelper.getImageFileName(vs));
    EList<VariableDeclaration> _varDecl = vs.getVarDecl();
    VariableDeclaration _head = null;
    if (_varDecl!=null) {
      _head=IterableExtensions.<VariableDeclaration>head(_varDecl);
    }
    final VariableDeclaration firstDecl = _head;
    if ((((firstDecl != null) && (firstDecl instanceof ExportedVariableDeclaration)) && (((ExportedVariableDeclaration) firstDecl).getDefinedVariable() != null))) {
      final TVariable firstDefinedVariable = ((ExportedVariableDeclaration) firstDecl).getDefinedVariable();
      ImageDescriptor _xifexpression = null;
      boolean _isConst = firstDefinedVariable.isConst();
      if (_isConst) {
        ImageDescriptor _xblockexpression = null;
        {
          final ImageDescriptor decorator = this.imageDescriptionLibrary.createConstImageDecorator();
          _xblockexpression = this.imageDescriptionHelper.createDecorationOverlayIcon(main, decorator, IDecoration.TOP_RIGHT);
        }
        _xifexpression = _xblockexpression;
      } else {
        _xifexpression = main;
      }
      final ImageDescriptor newMain = _xifexpression;
      return this.imageDescriptionHelper.addAccessibiltyImageDecorator(newMain, firstDefinedVariable.getTypeAccessModifier());
    }
    return main;
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final N4EnumLiteral n4EnumLiteral) {
    return this.dispatchDoGetImage(n4EnumLiteral.getDefinedLiteral());
  }
  
  /**
   * dispatchDoGetImage types model
   */
  protected ImageDescriptor _dispatchDoGetImage(final TClass tClass) {
    final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tClass, this.imageFileNameCalculationHelper.getImageFileName(tClass));
    boolean _isAbstract = tClass.isAbstract();
    if (_isAbstract) {
      return this.imageDescriptionHelper.createDecorationOverlayIcon(main, this.imageDescriptionLibrary.createAbstractImageDecorator(), IDecoration.TOP_RIGHT);
    }
    boolean _isFinal = tClass.isFinal();
    if (_isFinal) {
      return this.imageDescriptionHelper.createDecorationOverlayIcon(main, this.imageDescriptionLibrary.createFinalImageDecorator(), IDecoration.TOP_RIGHT);
    }
    return main;
  }
  
  /**
   * returns either the original image (provided no decorators were given) or an ImageDescriptor to display the decorators on top-right.
   */
  private ImageDescriptor decorated(final ImageDescriptor main, final List<ImageDescriptor> decorators) {
    boolean _isEmpty = decorators.isEmpty();
    if (_isEmpty) {
      return main;
    }
    int _size = decorators.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      return this.imageDescriptionHelper.createDecorationOverlayIcon(main, IterableExtensions.<ImageDescriptor>head(decorators), IDecoration.TOP_RIGHT);
    }
    return this.imageDescriptionHelper.createDecorationComposite(main, ((ImageDescriptor[])Conversions.unwrapArray(decorators, ImageDescriptor.class)));
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TInterface tInterface) {
    ImageDescriptor _xblockexpression = null;
    {
      final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tInterface, this.imageFileNameCalculationHelper.getImageFileName(tInterface));
      _xblockexpression = this.imageDescriptionHelper.addAccessibiltyImageDecorator(main, tInterface.getTypeAccessModifier());
    }
    return _xblockexpression;
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TClassifier tClassifier) {
    return this.imageDescriptionHelper.createValidationAwareImageDescriptor(tClassifier, this.imageFileNameCalculationHelper.getImageFileName(tClassifier));
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TFunction tFunction) {
    final ImageDescriptor base = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tFunction, this.imageFileNameCalculationHelper.getImageFileName(tFunction));
    return this.decorateImageForTFunction(base, tFunction.getTypeAccessModifier());
  }
  
  private ImageDescriptor decorateImageForTFunction(final ImageDescriptor baseDesc, final TypeAccessModifier accessModifier) {
    return this.imageDescriptionHelper.addAccessibiltyImageDecorator(baseDesc, accessModifier);
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TVariable tVariable) {
    final ImageDescriptor base = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tVariable, this.imageFileNameCalculationHelper.getImageFileName(tVariable));
    return this.decorateImageForTVariable(base, tVariable.isConst(), tVariable.getTypeAccessModifier());
  }
  
  private ImageDescriptor decorateImageForTVariable(final ImageDescriptor baseDesc, final boolean isConst, final TypeAccessModifier accessModifier) {
    ImageDescriptor result = baseDesc;
    if (isConst) {
      result = this.imageDescriptionHelper.createDecorationOverlayIcon(result, this.imageDescriptionLibrary.createConstImageDecorator(), IDecoration.TOP_RIGHT);
    }
    result = this.imageDescriptionHelper.addAccessibiltyImageDecorator(result, accessModifier);
    return result;
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TField tField) {
    final ArrayList<ImageDescriptor> decorators = CollectionLiterals.<ImageDescriptor>newArrayList();
    boolean _isStatic = tField.isStatic();
    if (_isStatic) {
      ImageDescriptor _createStaticImageDecorator = this.imageDescriptionLibrary.createStaticImageDecorator();
      decorators.add(_createStaticImageDecorator);
    }
    boolean _isFinal = tField.isFinal();
    if (_isFinal) {
      ImageDescriptor _createFinalImageDecorator = this.imageDescriptionLibrary.createFinalImageDecorator();
      decorators.add(_createFinalImageDecorator);
    }
    final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tField, this.imageFileNameCalculationHelper.getImageFileName(tField));
    return this.decorated(main, decorators);
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TMethod tMethod) {
    final ArrayList<ImageDescriptor> decorators = CollectionLiterals.<ImageDescriptor>newArrayList();
    boolean _isAbstract = tMethod.isAbstract();
    if (_isAbstract) {
      ImageDescriptor _createAbstractImageDecorator = this.imageDescriptionLibrary.createAbstractImageDecorator();
      decorators.add(_createAbstractImageDecorator);
    }
    boolean _isStatic = tMethod.isStatic();
    if (_isStatic) {
      ImageDescriptor _createStaticImageDecorator = this.imageDescriptionLibrary.createStaticImageDecorator();
      decorators.add(_createStaticImageDecorator);
    }
    boolean _isFinal = tMethod.isFinal();
    if (_isFinal) {
      ImageDescriptor _createFinalImageDecorator = this.imageDescriptionLibrary.createFinalImageDecorator();
      decorators.add(_createFinalImageDecorator);
    }
    boolean _isConstructor = tMethod.isConstructor();
    if (_isConstructor) {
      ImageDescriptor _createConstructorImageDecorator = this.imageDescriptionLibrary.createConstructorImageDecorator();
      decorators.add(_createConstructorImageDecorator);
    }
    final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tMethod, this.imageFileNameCalculationHelper.getImageFileName(tMethod));
    return this.decorated(main, decorators);
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final FieldAccessor tFieldAccessor) {
    final ArrayList<ImageDescriptor> decorators = CollectionLiterals.<ImageDescriptor>newArrayList();
    boolean _isAbstract = tFieldAccessor.isAbstract();
    if (_isAbstract) {
      ImageDescriptor _createAbstractImageDecorator = this.imageDescriptionLibrary.createAbstractImageDecorator();
      decorators.add(_createAbstractImageDecorator);
    }
    boolean _isStatic = tFieldAccessor.isStatic();
    if (_isStatic) {
      ImageDescriptor _createStaticImageDecorator = this.imageDescriptionLibrary.createStaticImageDecorator();
      decorators.add(_createStaticImageDecorator);
    }
    boolean _isFinal = tFieldAccessor.isFinal();
    if (_isFinal) {
      ImageDescriptor _createFinalImageDecorator = this.imageDescriptionLibrary.createFinalImageDecorator();
      decorators.add(_createFinalImageDecorator);
    }
    ImageDescriptor _xifexpression = null;
    boolean _isEmpty = decorators.isEmpty();
    if (_isEmpty) {
      _xifexpression = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tFieldAccessor, this.imageFileNameCalculationHelper.getImageFileName(tFieldAccessor));
    } else {
      ImageDescriptor _xblockexpression = null;
      {
        final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tFieldAccessor, this.imageFileNameCalculationHelper.getImageFileName(tFieldAccessor));
        _xblockexpression = this.decorated(main, decorators);
      }
      _xifexpression = _xblockexpression;
    }
    final ImageDescriptor newMain = _xifexpression;
    ImageDescriptor _xifexpression_1 = null;
    if ((tFieldAccessor instanceof TSetter)) {
      _xifexpression_1 = this.imageDescriptionLibrary.createSetterImageDecorator();
    } else {
      ImageDescriptor _xifexpression_2 = null;
      if ((tFieldAccessor instanceof TGetter)) {
        _xifexpression_2 = this.imageDescriptionLibrary.createGetterImageDecorator();
      }
      _xifexpression_1 = _xifexpression_2;
    }
    final ImageDescriptor fieldAccessorDecorator = _xifexpression_1;
    return this.imageDescriptionHelper.createDecorationOverlayIcon(newMain, fieldAccessorDecorator, IDecoration.BOTTOM_LEFT);
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final TEnumLiteral tEnumLiteral) {
    final ArrayList<ImageDescriptor> decorators = CollectionLiterals.<ImageDescriptor>newArrayList();
    ImageDescriptor _createStaticImageDecorator = this.imageDescriptionLibrary.createStaticImageDecorator();
    decorators.add(_createStaticImageDecorator);
    ImageDescriptor _createFinalImageDecorator = this.imageDescriptionLibrary.createFinalImageDecorator();
    decorators.add(_createFinalImageDecorator);
    final ImageDescriptor main = this.imageDescriptionHelper.createValidationAwareImageDescriptor(tEnumLiteral, this.imageFileNameCalculationHelper.getImageFileName(tEnumLiteral));
    return this.imageDescriptionHelper.createDecorationComposite(main, ((ImageDescriptor[])Conversions.unwrapArray(decorators, ImageDescriptor.class)));
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final Type type) {
    return this.imageDescriptionHelper.createValidationAwareImageDescriptor(type, this.imageFileNameCalculationHelper.getImageFileName(type));
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final Keyword keyword) {
    return this.imageDescriptionHelper.createSimpleImageDescriptor(this.imageFileNameCalculationHelper.getImageFileName(keyword));
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final EObject object) {
    if ((object instanceof IEObjectDescription)) {
      boolean _matches = TypeSearchKind.EVERYTHING.matches(((IEObjectDescription)object).getEClass());
      if (_matches) {
        final ImageDescriptor imageDesc = this.imageDescriptionHelper.createSimpleImageDescriptor(this.imageFileNameCalculationHelper.getImageFileName(object));
        final EClass eClass = ((IEObjectDescription)object).getEClass();
        EClass _tVariable = TypesPackage.eINSTANCE.getTVariable();
        boolean _tripleEquals = (_tVariable == eClass);
        if (_tripleEquals) {
          final boolean isConst = N4JSResourceDescriptionStrategy.getConst(((IEObjectDescription)object));
          final TypeAccessModifier accessModifier = N4JSResourceDescriptionStrategy.getTypeAccessModifier(((IEObjectDescription)object));
          return this.decorateImageForTVariable(imageDesc, isConst, accessModifier);
        } else {
          boolean _isSuperTypeOf = TypesPackage.eINSTANCE.getTFunction().isSuperTypeOf(eClass);
          if (_isSuperTypeOf) {
            final TypeAccessModifier accessModifier_1 = N4JSResourceDescriptionStrategy.getTypeAccessModifier(((IEObjectDescription)object));
            return this.decorateImageForTFunction(imageDesc, accessModifier_1);
          } else {
            EClass _tClass = TypesPackage.eINSTANCE.getTClass();
            boolean _tripleEquals_1 = (_tClass == eClass);
            if (_tripleEquals_1) {
              boolean _abstract = N4JSResourceDescriptionStrategy.getAbstract(((IEObjectDescription)object));
              if (_abstract) {
                return this.imageDescriptionHelper.createDecorationOverlayIcon(imageDesc, this.imageDescriptionLibrary.createAbstractImageDecorator(), IDecoration.TOP_RIGHT);
              }
              boolean _final = N4JSResourceDescriptionStrategy.getFinal(((IEObjectDescription)object));
              if (_final) {
                return this.imageDescriptionHelper.createDecorationOverlayIcon(imageDesc, this.imageDescriptionLibrary.createFinalImageDecorator(), IDecoration.TOP_RIGHT);
              }
            }
          }
        }
        return imageDesc;
      }
    }
    return this.imageDescriptionHelper.createValidationAwareImageDescriptor(object, this.imageFileNameCalculationHelper.getImageFileName(object));
  }
  
  protected ImageDescriptor _dispatchDoGetImage(final Object object) {
    return this.imageDescriptionHelper.createSimpleImageDescriptor(this.imageFileNameCalculationHelper.getImageFileName(object));
  }
  
  public ImageDescriptor dispatchDoGetImage(final Object tClass) {
    if (tClass instanceof TClass) {
      return _dispatchDoGetImage((TClass)tClass);
    } else if (tClass instanceof TInterface) {
      return _dispatchDoGetImage((TInterface)tClass);
    } else if (tClass instanceof TMethod) {
      return _dispatchDoGetImage((TMethod)tClass);
    } else if (tClass instanceof TClassifier) {
      return _dispatchDoGetImage((TClassifier)tClass);
    } else if (tClass instanceof TFunction) {
      return _dispatchDoGetImage((TFunction)tClass);
    } else if (tClass instanceof ExportedVariableDeclaration) {
      return _dispatchDoGetImage((ExportedVariableDeclaration)tClass);
    } else if (tClass instanceof N4ClassifierDeclaration) {
      return _dispatchDoGetImage((N4ClassifierDeclaration)tClass);
    } else if (tClass instanceof N4GetterDeclaration) {
      return _dispatchDoGetImage((N4GetterDeclaration)tClass);
    } else if (tClass instanceof N4SetterDeclaration) {
      return _dispatchDoGetImage((N4SetterDeclaration)tClass);
    } else if (tClass instanceof FieldAccessor) {
      return _dispatchDoGetImage((FieldAccessor)tClass);
    } else if (tClass instanceof TField) {
      return _dispatchDoGetImage((TField)tClass);
    } else if (tClass instanceof ExportedVariableStatement) {
      return _dispatchDoGetImage((ExportedVariableStatement)tClass);
    } else if (tClass instanceof FunctionDeclaration) {
      return _dispatchDoGetImage((FunctionDeclaration)tClass);
    } else if (tClass instanceof TVariable) {
      return _dispatchDoGetImage((TVariable)tClass);
    } else if (tClass instanceof Type) {
      return _dispatchDoGetImage((Type)tClass);
    } else if (tClass instanceof TEnumLiteral) {
      return _dispatchDoGetImage((TEnumLiteral)tClass);
    } else if (tClass instanceof N4EnumLiteral) {
      return _dispatchDoGetImage((N4EnumLiteral)tClass);
    } else if (tClass instanceof N4MemberDeclaration) {
      return _dispatchDoGetImage((N4MemberDeclaration)tClass);
    } else if (tClass instanceof Keyword) {
      return _dispatchDoGetImage((Keyword)tClass);
    } else if (tClass instanceof EObject) {
      return _dispatchDoGetImage((EObject)tClass);
    } else if (tClass == null) {
      return _dispatchDoGetImage((Void)null);
    } else if (tClass instanceof EObjectWithContext) {
      return _dispatchDoGetImage((EObjectWithContext)tClass);
    } else if (tClass != null) {
      return _dispatchDoGetImage(tClass);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tClass).toString());
    }
  }
}
