/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.util.ArrayList;
import javax.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.JSXElement;
import org.eclipse.n4js.n4JS.JSXElementName;
import org.eclipse.n4js.n4JS.JSXPropertyAttribute;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4jsx.ReactHelper;
import org.eclipse.n4js.organize.imports.RefNameUtil;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ui.organize.imports.ReferenceProxyInfo;
import org.eclipse.n4js.utils.ResourceType;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Finds unresolved cross proxy references via {@link EcoreUtil.UnresolvedProxyCrossReferencer}.
 * Also handles {@link JSXElement} bindings
 */
@SuppressWarnings("all")
public class UnresolveProxyCrossRefHelper {
  @Inject
  private ReactHelper reactHelper;
  
  /**
   * Finds unresolved cross proxies and returns tuples of reference object and used name.
   */
  public Iterable<ReferenceProxyInfo> findProxyCrossRefInfo(final Script script) {
    final ResourceType resourceType = ResourceType.getResourceType(script);
    final Iterable<EObject> unresolvedCrossRefs = this.findBrokenCrossRefs(script, resourceType);
    final Function1<EObject, ReferenceProxyInfo> _function = (EObject it) -> {
      String _refName = this.getRefName(it);
      return new ReferenceProxyInfo(it, _refName);
    };
    final Iterable<ReferenceProxyInfo> proxyInfos = IterableExtensions.<EObject, ReferenceProxyInfo>map(unresolvedCrossRefs, _function);
    return this.filter(proxyInfos, resourceType);
  }
  
  private Iterable<EObject> findBrokenCrossRefs(final Script script, final ResourceType resourceType) {
    final ArrayList<Iterable<? extends EObject>> list = CollectionLiterals.<Iterable<? extends EObject>>newArrayList();
    final Function1<EStructuralFeature.Setting, EObject> _function = (EStructuralFeature.Setting it) -> {
      return it.getEObject();
    };
    list.add(IterableExtensions.<EStructuralFeature.Setting, EObject>map(Iterables.<EStructuralFeature.Setting>concat(EcoreUtil.UnresolvedProxyCrossReferencer.find(script).values()), _function));
    if ((resourceType == ResourceType.N4JSX)) {
      list.add(this.findBrokenJSXBindings(script));
    }
    return Iterables.<EObject>concat(list);
  }
  
  /**
   * Does additional filtering of proxies based on the resource type.
   */
  private Iterable<ReferenceProxyInfo> filter(final Iterable<ReferenceProxyInfo> references, final ResourceType resourceType) {
    if (resourceType != null) {
      switch (resourceType) {
        case N4JSX:
          final Function1<ReferenceProxyInfo, Boolean> _function = (ReferenceProxyInfo it) -> {
            boolean _n4jxIsToBeIgnored = this.n4jxIsToBeIgnored(it);
            return Boolean.valueOf((!_n4jxIsToBeIgnored));
          };
          return IterableExtensions.<ReferenceProxyInfo>filter(references, _function);
        default:
          return references;
      }
    } else {
      return references;
    }
  }
  
  /**
   * If reference is using name that starts with lower case it looks like HTML tag, if on top of that it is part of
   * {@link JSXElementName} or {@link JSXPropertyAttribute} than this reference should not be subject to Organize
   * Imports.
   */
  private boolean n4jxIsToBeIgnored(final ReferenceProxyInfo proxyInfo) {
    final String usedName = proxyInfo.name;
    boolean _xifexpression = false;
    boolean _isNullOrEmpty = Strings.isNullOrEmpty(usedName);
    if (_isNullOrEmpty) {
      _xifexpression = false;
    } else {
      _xifexpression = Character.isLowerCase(usedName.charAt(0));
    }
    final boolean isLikeHTML = _xifexpression;
    if (isLikeHTML) {
      return false;
    }
    final EObject refContainer = proxyInfo.eobject.eContainer();
    if ((refContainer == null)) {
      return false;
    }
    return ((refContainer instanceof JSXElementName) || (refContainer instanceof JSXPropertyAttribute));
  }
  
  /**
   * Returns list of {@link JSXElementName} that have broken bindings
   */
  private Iterable<JSXElementName> findBrokenJSXBindings(final Script script) {
    final Function1<JSXElement, Pair<JSXElement, TypeRef>> _function = (JSXElement jsxe) -> {
      TypeRef _jsxElementBindingType = this.reactHelper.getJsxElementBindingType(jsxe);
      return Pair.<JSXElement, TypeRef>of(jsxe, _jsxElementBindingType);
    };
    final Function1<Pair<JSXElement, TypeRef>, Boolean> _function_1 = (Pair<JSXElement, TypeRef> it) -> {
      TypeRef _value = it.getValue();
      return Boolean.valueOf((_value instanceof UnknownTypeRef));
    };
    final Function1<Pair<JSXElement, TypeRef>, JSXElementName> _function_2 = (Pair<JSXElement, TypeRef> it) -> {
      return it.getKey().getJsxElementName();
    };
    return IteratorExtensions.<JSXElementName>toIterable(IteratorExtensions.<Pair<JSXElement, TypeRef>, JSXElementName>map(IteratorExtensions.<Pair<JSXElement, TypeRef>>filter(IteratorExtensions.<JSXElement, Pair<JSXElement, TypeRef>>map(Iterators.<JSXElement>filter(script.eAllContents(), JSXElement.class), _function), _function_1), _function_2));
  }
  
  private String getRefName(final EObject obj) {
    boolean _matched = false;
    if (obj instanceof IdentifierRef) {
      _matched=true;
      return RefNameUtil.findIdentifierName(((IdentifierRef)obj));
    }
    if (!_matched) {
      if (obj instanceof ParameterizedTypeRef) {
        _matched=true;
        final String name = RefNameUtil.findTypeName(((ParameterizedTypeRef)obj));
        boolean _isParameterized = ((ParameterizedTypeRef)obj).isParameterized();
        boolean _not = (!_isParameterized);
        if (_not) {
          return name;
        } else {
          final int index = name.indexOf("<");
          if ((index > (-1))) {
            return name.substring(0, index);
          } else {
            return name;
          }
        }
      }
    }
    return NodeModelUtils.getTokenText(NodeModelUtils.getNode(obj));
  }
}
