/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils.collections;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Extension for {@link Iterable iterable}s.
 */
@SuppressWarnings("all")
public class Iterables2 {
  /**
   * Creates a chain of the elements with the following rules:
   * <ul>
   * <li>Throws a NPE if the {@code elements} argument is {@code null}.</li>
   * <li>Throws a IAE if the {@code elements} argument does not contain the {@code ref} argument.</li>
   * <li>Returns with a new list which first and last element element will be the {@code ref}.</li>
   * <ul>
   * <li>If the {@code ref} argument contained multiple times in the list, then the first occurrence will be used.
   * </li>
   * </ul>
   * <li>Creates a new list with the {@code ref}. Then finds the {@code ref} argument in the {@code elements} iterable
   * and simply gets the rest of the elements from {@code elements} argument and appends it after the {@code ref},
   * then gets all the elements before the {@code ref} in the original {@code elements} and appends to the new list.
   * Finally appends the {@code ref} to the very end of the new list.
   * </ul>
   * 
   * For instance:
   * <pre>
   * #[1, 2, 3, 4, 5].chainOf(3) === #[3, 4, 5, 1, 2, 3];
   * #[1, -1, 3, -1, 4, 5].chainOf(-1) === #[-1, 3, -1, 4, 5, 1, -1];
   * </pre>
   */
  public static <T extends Object> List<T> chainOf(final Iterable<T> elements, final T ref) {
    final List<T> copy = IterableExtensions.<T>toList(Preconditions.<Iterable<T>>checkNotNull(elements));
    final int indexOf = copy.indexOf(ref);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Element ");
    _builder.append(ref);
    _builder.append(" does not contained in ");
    _builder.append(elements);
    _builder.append(".");
    Preconditions.checkArgument((0 <= indexOf), _builder);
    List<T> _subList = copy.subList(indexOf, copy.size());
    List<T> _subList_1 = copy.subList(0, indexOf);
    Iterable<T> _plus = Iterables.<T>concat(_subList, _subList_1);
    List<T> _singletonList = Collections.<T>singletonList(ref);
    return IterableExtensions.<T>toList(Iterables.<T>concat(_plus, _singletonList));
  }
  
  /**
   * Returns an {@link Iterable} which aligns values of iterable1 and iterable2 by their indices.
   * 
   * The size of the resulting iterable is the minimum of iterable1.size and iterable2.size.
   */
  public static <T1 extends Object, T2 extends Object> Iterable<Pair<T1, T2>> align(final Iterable<T1> iterable1, final Iterable<T2> iterable2) {
    final Iterable<Pair<T1, T2>> _function = () -> {
      final Iterator<T1> it1 = iterable1.iterator();
      final Iterator<T2> it2 = iterable2.iterator();
      return new Iterator<Pair<T1, T2>>() {
        @Override
        public boolean hasNext() {
          return (it1.hasNext() && it2.hasNext());
        }
        
        @Override
        public Pair<T1, T2> next() {
          T1 _next = it1.next();
          T2 _next_1 = it2.next();
          return Pair.<T1, T2>of(_next, _next_1);
        }
      };
    };
    return _function;
  }
}
