/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.orion.internal.server.search;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.common.params.SolrParams;
import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.orion.internal.server.core.IOUtilities;
import org.eclipse.orion.internal.server.search.SearchActivator;
import org.eclipse.orion.server.core.LogHelper;
import org.eclipse.orion.server.core.OrionConfiguration;
import org.eclipse.orion.server.core.metastore.IMetaStore;
import org.eclipse.orion.server.core.metastore.ProjectInfo;
import org.eclipse.orion.server.core.metastore.UserInfo;
import org.eclipse.orion.server.core.metastore.WorkspaceInfo;
import org.eclipse.osgi.util.NLS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Indexer
extends Job {
    private static final long DEFAULT_DELAY = 60000L;
    private static final long IDLE_DELAY = 300000L;
    private static final long MAX_SEARCH_SIZE = 300000L;
    private final List<String> INDEXED_FILE_TYPES;
    private final SolrServer server;

    public Indexer(SolrServer server) {
        super("Indexing");
        this.server = server;
        this.setSystem(true);
        this.INDEXED_FILE_TYPES = Arrays.asList("css", "js", "html", "txt", "xml", "java", "properties", "php", "htm", "project", "conf", "pl", "sh", "text", "xhtml", "mf", "manifest", "md", "yaml", "yml");
        Collections.sort(this.INDEXED_FILE_TYPES);
    }

    public boolean belongsTo(Object family) {
        return SearchActivator.JOB_FAMILY.equals(family);
    }

    private void collectFiles(IFileStore dir, List<IFileStore> files) {
        try {
            IFileStore[] children;
            IFileStore[] iFileStoreArray = children = dir.childStores(0, null);
            int n = children.length;
            int n2 = 0;
            while (n2 < n) {
                IFileStore child = iFileStoreArray[n2];
                if (!child.getName().startsWith(".")) {
                    IFileInfo info = child.fetchInfo();
                    if (info.isDirectory()) {
                        this.collectFiles(child, files);
                    } else {
                        files.add(child);
                    }
                }
                ++n2;
            }
        }
        catch (CoreException e) {
            this.handleIndexingFailure(e, dir);
        }
    }

    public void ensureUpdated() {
        this.schedule(60000L);
    }

    private String getContentsAsString(IFileStore file) {
        StringWriter writer = new StringWriter();
        try {
            IOUtilities.pipe((Reader)new InputStreamReader(file.openInputStream(0, null)), (Writer)writer, (boolean)true, (boolean)false);
        }
        catch (IOException e) {
            this.handleIndexingFailure(e, file);
        }
        catch (CoreException e) {
            this.handleIndexingFailure(e, file);
        }
        return writer.toString();
    }

    private void handleIndexingFailure(Throwable t, IFileStore file) {
        String message = file != null ? NLS.bind((String)"Error during searching indexing on file: {0}", (Object)file.toString()) : "Error during searching indexing";
        LogHelper.log((IStatus)new Status(4, "org.eclipse.orion.server.core.search", message, t));
    }

    private int indexUser(String userId, IProgressMonitor monitor, List<SolrInputDocument> documents) {
        int indexed = 0;
        try {
            IMetaStore store = OrionConfiguration.getMetaStore();
            UserInfo user = store.readUser(userId);
            List workspaceIds = user.getWorkspaceIds();
            SubMonitor progress = SubMonitor.convert((IProgressMonitor)monitor, (int)workspaceIds.size());
            for (String workspaceId : workspaceIds) {
                WorkspaceInfo workspace = store.readWorkspace(workspaceId);
                indexed += this.indexWorkspace(user, workspace, progress.newChild(1), documents);
            }
        }
        catch (CoreException e) {
            this.handleIndexingFailure(e, null);
        }
        return indexed;
    }

    private int indexWorkspace(UserInfo user, WorkspaceInfo workspace, SubMonitor monitor, List<SolrInputDocument> documents) {
        int indexed = 0;
        IMetaStore store = OrionConfiguration.getMetaStore();
        for (String projectName : workspace.getProjectNames()) {
            try {
                indexed += this.indexProject(user, workspace, store.readProject(workspace.getUniqueId(), projectName), monitor, documents);
            }
            catch (CoreException e) {
                this.handleIndexingFailure(e, null);
            }
        }
        return indexed;
    }

    private int indexProject(UserInfo user, WorkspaceInfo workspace, ProjectInfo project, SubMonitor monitor, List<SolrInputDocument> documents) {
        String encodedProjectName;
        IFileStore projectStore;
        Logger logger = LoggerFactory.getLogger(Indexer.class);
        if (logger.isDebugEnabled()) {
            logger.debug("Indexing project id: " + project.getUniqueId() + " name: " + project.getFullName());
        }
        this.checkCanceled((IProgressMonitor)monitor);
        try {
            projectStore = project.getProjectStore();
        }
        catch (CoreException e) {
            this.handleIndexingFailure(e, null);
            return 0;
        }
        if (!EFS.getLocalFileSystem().getScheme().equals(projectStore.getFileSystem().getScheme())) {
            return 0;
        }
        try {
            encodedProjectName = new URI(null, project.getFullName(), null).toString();
        }
        catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
        IPath projectLocation = new Path("/file").append(workspace.getUniqueId()).append(encodedProjectName).addTrailingSeparator();
        int projectLocationLength = projectStore.toURI().toString().length();
        ArrayList<IFileStore> toIndex = new ArrayList<IFileStore>();
        this.collectFiles(projectStore, toIndex);
        int unmodifiedCount = 0;
        int indexedCount = 0;
        for (IFileStore file : toIndex) {
            this.checkCanceled((IProgressMonitor)monitor);
            IFileInfo fileInfo = file.fetchInfo();
            if (!this.isModified(file, fileInfo)) {
                ++unmodifiedCount;
                continue;
            }
            ++indexedCount;
            SolrInputDocument doc = new SolrInputDocument();
            doc.addField("Id", (Object)file.toURI().toString());
            doc.addField("Name", (Object)fileInfo.getName());
            doc.addField("NameLower", (Object)fileInfo.getName());
            doc.addField("Length", (Object)Long.toString(fileInfo.getLength()));
            doc.addField("Directory", (Object)Boolean.toString(fileInfo.isDirectory()));
            doc.addField("LastModified", (Object)Long.toString(fileInfo.getLastModified()));
            String projectRelativePath = file.toURI().toString().substring(projectLocationLength);
            IPath fileLocation = projectLocation.append(projectRelativePath);
            doc.addField("Location", (Object)fileLocation.toString());
            String projectName = project.getFullName();
            if (projectName == null) continue;
            doc.addField("Path", (Object)new Path(projectName).append(projectRelativePath));
            if (!this.skip(fileInfo)) {
                doc.addField("Text", (Object)this.getContentsAsString(file));
            }
            doc.addField("UserName", (Object)user.getUniqueId());
            try {
                this.server.add(doc);
            }
            catch (Exception e) {
                this.handleIndexingFailure(e, file);
            }
        }
        try {
            this.server.commit();
        }
        catch (Exception e) {
            this.handleIndexingFailure(e, null);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("\tIndexed: " + indexedCount + " Unchanged:  " + unmodifiedCount);
        }
        return indexedCount;
    }

    private boolean skip(IFileInfo fileInfo) {
        if (fileInfo.getLength() > 300000L) {
            return true;
        }
        String extension = new Path(fileInfo.getName()).getFileExtension();
        return extension == null || Collections.binarySearch(this.INDEXED_FILE_TYPES, extension.toLowerCase()) < 0;
    }

    private boolean isModified(IFileStore file, IFileInfo fileInfo) {
        try {
            StringBuffer qString = new StringBuffer("Id");
            qString.append(':');
            qString.append(ClientUtils.escapeQueryChars((String)file.toURI().toString()));
            qString.append(" AND ");
            qString.append("LastModified");
            qString.append(':');
            qString.append(Long.toString(fileInfo.getLastModified()));
            SolrQuery query = new SolrQuery(qString.toString());
            query.setParam("fl", new String[]{"Id"});
            QueryResponse response = this.server.query((SolrParams)query);
            return response.getResults().getNumFound() == 0L;
        }
        catch (SolrServerException e) {
            this.handleIndexingFailure(e, file);
            return true;
        }
    }

    private void checkCanceled(IProgressMonitor monitor) {
        if (monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    protected IStatus run(IProgressMonitor monitor) {
        IMetaStore metaStore;
        long start = System.currentTimeMillis();
        Logger logger = LoggerFactory.getLogger(Indexer.class);
        try {
            metaStore = OrionConfiguration.getMetaStore();
        }
        catch (IllegalStateException illegalStateException) {
            if (logger.isDebugEnabled()) {
                logger.debug("Search indexer waiting for metadata service");
            }
            this.schedule(5000L);
            return Status.OK_STATUS;
        }
        int indexed = 0;
        try {
            List userIds = metaStore.readAllUsers();
            SubMonitor progress = SubMonitor.convert((IProgressMonitor)monitor, (int)userIds.size());
            ArrayList<SolrInputDocument> documents = new ArrayList<SolrInputDocument>();
            indexed = 0;
            for (String userId : userIds) {
                indexed += this.indexUser(userId, (IProgressMonitor)progress.newChild(1), documents);
            }
        }
        catch (CoreException e) {
            this.handleIndexingFailure(e, null);
        }
        long duration = System.currentTimeMillis() - start;
        if (logger.isDebugEnabled()) {
            logger.debug("Indexed " + indexed + " documents in " + duration + "ms");
        }
        long delay = Math.max(60000L, duration * 10L);
        if (indexed == 0) {
            delay = Math.max(delay, 300000L);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Rescheduling indexing in " + delay + "ms");
        }
        this.schedule(delay);
        return Status.OK_STATUS;
    }
}

