/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/Deferred',factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory();
    } else {
        root.orion = root.orion || {};
        root.orion.Deferred = factory();
    }
}(this, function() {
    var queue = [],
        running = false;

    function run() {
        var fn;
        while ((fn = queue.shift())) {
            fn();
        }
        running = false;
    }

	var runAsync = (function() {
		if (typeof process !== "undefined" && typeof process.nextTick === "function") {
			var nextTick = process.nextTick;
    		return function() {
    			nextTick(run);
    		};
		} else if (typeof MutationObserver === "function") {
			var div = document.createElement("div");
			var observer = new MutationObserver(run);
			observer.observe(div, {
            	attributes: true
        	});
        	return function() {
        		div.setAttribute("class", "_tick");
        	};
		}
		return function() {
			setTimeout(run, 0);
		};
	})();

    function enqueue(fn) {
        queue.push(fn);
        if (!running) {
            running = true;
            runAsync();
        }
    }

    function noReturn(fn) {
        return function(result) {
            fn(result);
        };
    }
    
    function settleDeferred(fn, result, deferred) {
    	try {
    		var listenerResult = fn(result);
    		var listenerThen = listenerResult && (typeof listenerResult === "object" || typeof listenerResult === "function") && listenerResult.then;
    		if (typeof listenerThen === "function") {
    			if (listenerResult === deferred.promise) {
    				deferred.reject(new TypeError());
    			} else {
    				var listenerResultCancel = listenerResult.cancel;
    				if (typeof listenerResultCancel === "function") {
    					deferred._parentCancel = listenerResultCancel.bind(listenerResult);
    				} else {
    					delete deferred._parentCancel;
    				}
    				listenerThen.call(listenerResult, noReturn(deferred.resolve), noReturn(deferred.reject), noReturn(deferred.progress));
    			}
    		} else {
    			deferred.resolve(listenerResult);
    		}
    	} catch (e) {
    		deferred.reject(e);
    	}
    }


    /**
     * @name orion.Promise
     * @class Interface representing an eventual value.
     * @description Promise is an interface that represents an eventual value returned from the single completion of an operation.
     *
     * <p>For a concrete class that implements Promise and provides additional API, see {@link orion.Deferred}.</p>
     * @see orion.Deferred
     * @see orion.Deferred#promise
     */
    /**
     * @name then
     * @function
     * @memberOf orion.Promise.prototype
     * @description Adds handlers to be called on fulfillment or progress of this promise.
     * @param {Function} [onResolve] Called when this promise is resolved.
     * @param {Function} [onReject] Called when this promise is rejected.
     * @param {Function} [onProgress] May be called to report progress events on this promise.
     * @returns {orion.Promise} A new promise that is fulfilled when the given <code>onResolve</code> or <code>onReject</code>
     * callback is finished. The callback's return value gives the fulfillment value of the returned promise.
     */
    /**
     * Cancels this promise.
     * @name cancel
     * @function
     * @memberOf orion.Promise.prototype
     * @param {Object} reason The reason for canceling this promise.
     * @param {Boolean} [strict]
     */

    /**
     * @name orion.Deferred
     * @borrows orion.Promise#then as #then
     * @borrows orion.Promise#cancel as #cancel
     * @class Provides abstraction over asynchronous operations.
     * @description Deferred provides abstraction over asynchronous operations.
     *
     * <p>Because Deferred implements the {@link orion.Promise} interface, a Deferred may be used anywhere a Promise is called for.
     * However, in most such cases it is recommended to use the Deferred's {@link #promise} field instead, which exposes a 
     * simplified, minimally <a href="https://github.com/promises-aplus/promises-spec">Promises/A+</a>-compliant interface to callers.</p>
     */
    function Deferred() {
        var result, state, listeners = [],
            _this = this;

        function notify() {
            var listener;
            while ((listener = listeners.shift())) {
                var deferred = listener.deferred;
                var methodName = state === "fulfilled" ? "resolve" : "reject"; //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
                var fn = listener[methodName];
                if (typeof fn === "function") { //$NON-NLS-0$
                	settleDeferred(fn, result, deferred);
                } else {
                    deferred[methodName](result);
                }
            }
        }

        function _reject(error) {
            delete _this._parentCancel;
            state = "rejected";
            result = error;
            if (listeners.length) {
                enqueue(notify);
            }
        }

        function _resolve(value) {
            function once(fn) {
                return function(result) {
                    if (!state || state === "assumed") {
                          fn(result);
                    }
                };
            }
            delete _this._parentCancel;
            try {
                var valueThen = value && (typeof value === "object" || typeof value === "function") && value.then;
                if (typeof valueThen === "function") {
                    if (value === _this) {
                        _reject(new TypeError());
                    } else {
                        state = "assumed";
                        var valueCancel = value && value.cancel;
                        if (typeof valueCancel !== "function") {
                            var deferred = new Deferred();
                            value = deferred.promise;
                            try {
                                valueThen(deferred.resolve, deferred.reject, deferred.progress);
                            } catch (thenError) {
                                deferred.reject(thenError);
                            }
                            valueCancel = value.cancel;
                            valueThen = value.then;
                        }
                        result = value;
                        valueThen.call(value, once(_resolve), once(_reject));
                        _this._parentCancel = valueCancel.bind(value);
                    }
                } else {
                    state = "fulfilled";
                    result = value;
                    if (listeners.length) {
                        enqueue(notify);
                    }
                }
            } catch (error) {
                once(_reject)(error);
            }
        }

        function cancel() {
            var parentCancel = _this._parentCancel;
            if (parentCancel) {
                delete _this._parentCancel;
                parentCancel();
            } else if (!state) {
                var cancelError = new Error("Cancel");
                cancelError.name = "Cancel";
                _reject(cancelError);
            }
        }


        /**
         * Resolves this Deferred.
         * @name resolve
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} value
         * @returns {orion.Promise}
         */
        this.resolve = function(value) {
            if (!state) {
                _resolve(value);
            }
            return _this;
        };

        /**
         * Rejects this Deferred.
         * @name reject
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} error
         * @param {Boolean} [strict]
         * @returns {orion.Promise}
         */
        this.reject = function(error) {
            if (!state) {
                _reject(error);
            }
            return _this;
        };

        /**
         * Notifies listeners of progress on this Deferred.
         * @name progress
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} update The progress update.
         * @returns {orion.Promise}
         */
        this.progress = function(update) {
            if (!state) {
                listeners.forEach(function(listener) {
                    if (listener.progress) {
                        try {
                            listener.progress(update);
                        } catch (ignore) {
                            // ignore
                        }
                    }
                });
            }
            return _this.promise;
        };

        this.cancel = function() {
            if (_this._parentCancel) {
                setTimeout(cancel, 0);
            } else {
                cancel();
            }
            return _this;
        };

        // Note: "then" ALWAYS returns before having onResolve or onReject called as per http://promises-aplus.github.com/promises-spec/
        this.then = function(onFulfill, onReject, onProgress) {
        	var deferred = new Deferred();
            deferred._parentCancel = _this.promise.cancel;
            listeners.push({
                resolve: onFulfill,
                reject: onReject,
                progress: onProgress,
                deferred: deferred
            });
            if (state === "fulfilled" || state === "rejected") {
                enqueue(notify);
            }
            return deferred.promise;
        };

        /**
         * The promise exposed by this Deferred.
         * @name promise
         * @field
         * @memberOf orion.Deferred.prototype
         * @type orion.Promise
         */
        this.promise = {
            then: _this.then,
            cancel: _this.cancel
        };
    }

    /**
     * Returns a promise that represents the outcome of all the input promises.
     * <p>When <code>all</code> is called with a single parameter, the returned promise has <dfn>eager</dfn> semantics,
     * meaning that if any input promise rejects, the returned promise immediately rejects, without waiting for the rest of the
     * input promises to fulfill.</p>
     *
     * To obtain <dfn>lazy</dfn> semantics (meaning the returned promise waits for every input promise to fulfill), pass the
     * optional parameter <code>optOnError</code>.
     * @name all
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {orion.Promise[]} promises The input promises.
     * @param {Function} [optOnError] Handles a rejected input promise. <code>optOnError</code> is invoked for every rejected
     * input promise, and is passed the reason the input promise was rejected. <p><code>optOnError</code> can return a value, which
     * allows it to act as a transformer: the return value serves as the final fulfillment value of the rejected promise in the 
     * results array generated by <code>all</code>.
     * @returns {orion.Promise} A new promise. The returned promise is generally fulfilled to an <code>Array</code> whose elements
     * give the fulfillment values of the input promises. <p>However, if an input promise rejects and eager semantics is used, the 
     * returned promise will instead be fulfilled to a single error value.</p>
     */
    Deferred.all = function(promises, optOnError) {
        var count = promises.length,
            result = [],
            rejected = false,
            deferred = new Deferred();

        deferred.then(undefined, function() {
            rejected = true;
            promises.forEach(function(promise) {
                if (promise.cancel) {
                    promise.cancel();
                }
            });
        });

        function onResolve(i, value) {
            if (!rejected) {
                result[i] = value;
                if (--count === 0) {
                    deferred.resolve(result);
                }
            }
        }

        function onReject(i, error) {
            if (!rejected) {
                if (optOnError) {
                    try {
                        onResolve(i, optOnError(error));
                        return;
                    } catch (e) {
                        error = e;
                    }
                }
                deferred.reject(error);
            }
        }

        if (count === 0) {
            deferred.resolve(result);
        } else {
            promises.forEach(function(promise, i) {
                promise.then(onResolve.bind(undefined, i), onReject.bind(undefined, i));
            });
        }
        return deferred.promise;
    };

    /**
     * Applies callbacks to a promise or to a regular object.
     * @name when
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {Object|orion.Promise} value Either a {@link orion.Promise}, or a normal value.
     * @param {Function} onResolve Called when the <code>value</code> promise is resolved. If <code>value</code> is not a promise,
     * this function is called immediately.
     * @param {Function} onReject Called when the <code>value</code> promise is rejected. If <code>value</code> is not a promise, 
     * this function is never called.
     * @param {Function} onProgress Called when the <code>value</code> promise provides a progress update. If <code>value</code> is
     * not a promise, this function is never called.
     * @returns {orion.Promise} A new promise.
     */
    Deferred.when = function(value, onResolve, onReject, onProgress) {
        var promise, deferred;
        if (value && typeof value.then === "function") { //$NON-NLS-0$
            promise = value;
        } else {
            deferred = new Deferred();
            deferred.resolve(value);
            promise = deferred.promise;
        }
        return promise.then(onResolve, onReject, onProgress);
    };

    return Deferred;
}));

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/plugin',["orion/Deferred"], factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory(require("orion/Deferred"));
    } else {
        root.orion = root.orion || {};
        root.orion.PluginProvider = factory(root.orion.Deferred);
    }
}(this, function(Deferred) {
    function ObjectReference(objectId, methods) {
        this.__objectId = objectId;
        this.__methods = methods;
    }
    
    function PluginProvider(headers) {
        var _headers = headers;
        var _connected = false;

        var _currentMessageId = 0;
        var _currentObjectId = 0;
        var _currentServiceId = 0;

        var _requestReferences = {};
        var _responseReferences = {};
        var _objectReferences = {};
        var _serviceReferences = {};
        
        var _target = null;
        if (typeof(window) === "undefined") { //$NON-NLS-0$
            _target = self;
        } else if (window !== window.parent) {
            _target = window.parent;
        } else if (window.opener !== null) {
            _target = window.opener;
        }        

        function _publish(message) {
            if (_target) {
                if (typeof(ArrayBuffer) === "undefined") { //$NON-NLS-0$
                    message = JSON.stringify(message);
                }
                if (_target === self) {
                    _target.postMessage(message);
                } else {
                    _target.postMessage(message, "*"); //$NON-NLS-0$
                }
            }
        }
        var _notify = _publish;
        
        var lastHeartbeat;
        var startTime = new Date().getTime();
        function log(state) {
            if (localStorage.pluginLogging) console.log(state + "(" + (new Date().getTime() - startTime) + "ms)=" + window.location); //$NON-NLS-1$ //$NON-NLS-0$
        }
        function heartbeat() {
            var time = new Date().getTime();
            // This timeout depends on the handshake timeout of the plugin registry. Update both accordingly.
            if (lastHeartbeat  && time - lastHeartbeat < 4000) return;
            lastHeartbeat = time;
            _publish({
                method: "loading", //$NON-NLS-0$
            });
            log("heartbeat"); //$NON-NLS-0$
        }
        heartbeat();

        function _getPluginData() {
            var services = [];
            // we filter out the service implementation from the data
            Object.keys(_serviceReferences).forEach(function(serviceId) {
                var serviceReference = _serviceReferences[serviceId];
                services.push({
                    serviceId: serviceId,
                    names: serviceReference.names,
                    methods: serviceReference.methods,
                    properties: serviceReference.properties
                });
            });
            return {
                headers: _headers || {},
                services: services
            };
        }

        function _jsonXMLHttpRequestReplacer(name, value) {
            if (value && value instanceof XMLHttpRequest) {
                var status, statusText;
                try {
                    status = value.status;
                    statusText = value.statusText;
                } catch (e) {
                    // https://bugs.webkit.org/show_bug.cgi?id=45994
                    status = 0;
                    statusText = ""; //$NON-NLS-0
                }
                return {
                    status: status || 0,
                    statusText: statusText
                };
            }
            return value;
        }

        function _serializeError(error) {
            var result = error ? JSON.parse(JSON.stringify(error, _jsonXMLHttpRequestReplacer)) : error; // sanitizing Error object
            if (error instanceof Error) {
                result.__isError = true;
                result.message = result.message || error.message;
                result.name = result.name || error.name;
            }
            return result;
        }

        function _request(message) {
            if (!_target) {
                return new Deferred().reject(new Error("plugin not connected"));
            }

            message.id = String(_currentMessageId++);
            var d = new Deferred();
            _responseReferences[message.id] = d;
            d.then(null, function(error) {
                if (_connected && error instanceof Error && error.name === "Cancel") {
                    _notify({
                        requestId: message.id,
                        method: "cancel",
                        params: error.message ? [error.message] : []
                    });
                }
            });

            var toString = Object.prototype.toString;
            message.params.forEach(function(param, i) {
                if (toString.call(param) === "[object Object]" && !(param instanceof ObjectReference)) {
                    var candidate, methods;
                    for (candidate in param) {
                        if (toString.call(param[candidate]) === "[object Function]") {
                            methods = methods || [];
                            methods.push(candidate);
                        }
                    }
                    if (methods) {
                        var objectId = _currentObjectId++;
                        _objectReferences[objectId] = param;
                        var removeReference = function() {
                            delete _objectReferences[objectId];
                        };
                        d.then(removeReference, removeReference);
                        message.params[i] = new ObjectReference(objectId, methods);
                    }
                }
            });
            _notify(message);
            return d.promise;
        }

        function _throwError(messageId, error) {
            if (messageId || messageId === 0) {
                _notify({
                    id: messageId,
                    result: null,
                    error: error
                });
            } else {
                console.log(error);
            }

        }

        function _callMethod(messageId, implementation, method, params) {
            params.forEach(function(param, i) {
                if (param && typeof param.__objectId !== "undefined") {
                    var obj = {};
                    param.__methods.forEach(function(method) {
                        obj[method] = function() {
                            return _request({
                                objectId: param.__objectId,
                                method: method,
                                params: Array.prototype.slice.call(arguments)
                            });
                        };
                    });
                    params[i] = obj;
                }
            });
            var response = typeof messageId === "undefined" ? null : {
                id: messageId,
                result: null,
                error: null
            };
            try {
                var promiseOrResult = method.apply(implementation, params);
                if (!response) {
                    return;
                }

                if (promiseOrResult && typeof promiseOrResult.then === "function") { //$NON-NLS-0$
                    _requestReferences[messageId] = promiseOrResult;
                    promiseOrResult.then(function(result) {
                        delete _requestReferences[messageId];
                        response.result = result;
                        _notify(response);
                    }, function(error) {
                        if (_requestReferences[messageId]) {
                            delete _requestReferences[messageId];
                            response.error = _serializeError(error);
                            _notify(response);
                        }
                    }, function() {
                        _notify({
                            responseId: messageId,
                            method: "progress",
                            params: Array.prototype.slice.call(arguments)
                        }); //$NON-NLS-0$
                    });
                } else {
                    response.result = promiseOrResult;
                    _notify(response);
                }
            } catch (error) {
                if (response) {
                    response.error = _serializeError(error);
                    _notify(response);
                }
            }
        }

        function _handleMessage(event) {
            if (event.source !== _target && typeof window !== "undefined") {
                return;
            }
            var message = (typeof event.data !== "string" ? event.data : JSON.parse(event.data)); //$NON-NLS-0$
            try {
                if (message.method) { // request
                    var method = message.method,
                        params = message.params || [];
                    if ("serviceId" in message) {
                        var service = _serviceReferences[message.serviceId];
                        if (!service) {
                            _throwError(message.id, "service not found");
                        }
                        service = service.implementation;
                        if (method in service) {
                            _callMethod(message.id, service, service[method], params);
                        } else {
                            _throwError(message.id, "method not found");
                        }
                    } else if ("objectId" in message) {
                        var object = _objectReferences[message.objectId];
                        if (!object) {
                            _throwError(message.id, "object not found");
                        }
                        if (!method in object) {
                            _callMethod(message.id, object, object[method], params);
                        } else {
                            _throwError(message.id, "method not found");
                        }
                    } else if ("requestId" in message) {
                        var request = _requestReferences[message.requestId];
                        if (request && method === "cancel" && request.cancel) {
                            request.cancel.apply(request, params);
                        }
                    } else if ("responseId" in message) {
                        var response = _responseReferences[message.responseId];
                        if (response && method === "progress" && response.progress) {
                            response.progress.apply(response, params);
                        }
                    } else {
                        throw new Error("Bad method: " + message.method);
                    }
                } else {
                    var deferred = _responseReferences[String(message.id)];
                    delete _responseReferences[String(message.id)];
                    if (message.error) {
                        deferred.reject(message.error);
                    } else {
                        deferred.resolve(message.result);
                    }
                }
            } catch (e) {
                console.log("Plugin._messageHandler " + e);
            }
        }

        this.updateHeaders = function(headers) {
            if (_connected) {
                throw new Error("Cannot update headers. Plugin Provider is connected");
            }
            _headers = headers;
        };

        this.registerService = function(names, implementation, properties) {
            if (_connected) {
                throw new Error("Cannot register service. Plugin Provider is connected");
            }

            if (typeof names === "string") {
                names = [names];
            } else if (!Array.isArray(names)) {
                names = [];
            }

            var method = null;
            var methods = [];
            for (method in implementation) {
                if (typeof implementation[method] === 'function') { //$NON-NLS-0$
                    methods.push(method);
                }
            }
            _serviceReferences[_currentServiceId++] = {
                names: names,
                methods: methods,
                implementation: implementation,
                properties: properties || {},
                listeners: {}
            };
            heartbeat();
        };
        this.registerServiceProvider = this.registerService;

        this.connect = function(callback, errback) {
            if (_connected) {
                if (callback) {
                    callback();
                }
                return;
            }
            if (!_target) {
            	if (errback) {
            		errback("No valid plugin target");
            	}
            	return;
            }           
            addEventListener("message", _handleMessage, false); //$NON-NLS-0$
            var message = {
                method: "plugin", //$NON-NLS-0$
                params: [_getPluginData()]
            };
            _publish(message);
            _connected = true;
            if (callback) {
                callback();
            }
        };

        this.disconnect = function() {
            if (_connected) {
                removeEventListener("message", _handleMessage); //$NON-NLS-0$
                _target = null;
                _connected = false;
            }
            // Note: re-connecting is not currently supported
        };            
    }
    
    return PluginProvider;
}));

/*******************************************************************************
 * Copyright (c) 2014 SAP AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAP AG - initial API and implementation
 *******************************************************************************/
define('orion/xsrfUtils',[],function(){
	var XSRF_TOKEN = "x-csrf-token";//$NON-NLS-0$

	/**
	 * extracts value of xsrf cookie if available
	 */
	function getCSRFToken() {
		var cookies = document.cookie.split(";");//$NON-NLS-0$

		var i,n,v;
		for(i = 0; i<cookies.length; i++) {
			n = cookies[i].substr(0, cookies[i].indexOf("=")).trim();//$NON-NLS-0$
			v = cookies[i].substr(cookies[i].indexOf("=") + 1).trim();//$NON-NLS-0$

			if(n == XSRF_TOKEN) {
				return v;
			}
		}
	}

	/**
	 * adds xsrf nonce to header if set in cookies
	 * @param {Object} request header
	 */
	function setNonceHeader(headers) {
		var token = getCSRFToken();
		if (token) {
			headers[XSRF_TOKEN] = token;
		}
	}

	/**
	 * adds xsrf nonce to an XMLHTTPRequest object if set in cookies
	 * @param {Object} XMLHttpRequest object
	 */
	function addCSRFNonce(request) {
		var token = getCSRFToken();
		if(token) {
			request.setRequestHeader(XSRF_TOKEN, token);
		}
	}

	return {
		XSRF_TOKEN: XSRF_TOKEN,
		getCSRFToken: getCSRFToken,
		setNonceHeader: setNonceHeader,
		addCSRFNonce: addCSRFNonce
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global StopIteration*/
// URL Shim -- see http://url.spec.whatwg.org/ and http://dvcs.w3.org/hg/url/raw-file/tip/Overview.html

(function() {
    try {
        var testURL;
        if (typeof window.URL === "function" && window.URL.length !== 0 &&
                (testURL = new window.URL("http://www.w3.org?q")).protocol === "http:" && testURL.query) {
            return;
        }
    } catch (e) {}

    //[1]scheme, [2]authority, [3]path, [4]query, [5]fragment
    var _URI_RE = /^(?:([^:\/?#]+):)?(?:\/\/([^\/?#]*))?([^?#]*)(?:\?([^#]*))?(?:#(.*))?$/;
    //[ userinfo "@" ] host [ ":" port ]
    var _AUTHORITY_RE = /^(?:(.*)@)?(\[[^\]]*\]|[^:]*)(?::(.*))?$/;

    var _NO_WS_RE = /^\S*$/;
    var _SCHEME_RE = /^([a-zA-Z](?:[a-zA-Z0-9+-.])*)$/;
    var _PORT_RE = /^\d*$/;
    var _HOST_RE = /^(\[[^\]\/?#\s]*\]|[^:\/?#\s]*)$/;
    var _HOSTPORT_RE = /^(\[[^\]\/?#\s]*\]|[^:\/?#\s]*)(?::(\d*))?$/;
    var _PATH_RE = /^([^?#\s]*)$/;
    var _QUERY_RE = /^([^\s]*)$/;
    var _FRAGMENT_RE = _NO_WS_RE;
    var _USERNAME_PASSWORD_RE = /([^:]*):?(.*)/;

    var STOP_ITERATION = typeof StopIteration !== "undefined" ? StopIteration : new Error("Stop Iteration");
    var DEFAULT_PORTS = {
        "ftp:": "21",
            "gopher:": "70",
            "http:": "80",
            "https:": "443",
            "ws:": "80",
            "wss:": "443"
    };

    function _checkString(txt) {
        if (typeof txt !== "string") {
            throw new TypeError();
        }
    }

    function _parseQuery(query) {
        return query ? query.split("&") : [];
    }

    function _stringifyQuery(pairs) {
        if (pairs.length === 0) {
            return "";
        }
        return pairs.join("&");
    }

    function _parsePair(pair) {
        var parsed = /([^=]*)(?:=?)(.*)/.exec(pair);
        var key = parsed[1] ? decodeURIComponent(parsed[1]) : "";
        var value = parsed[2] ? decodeURIComponent(parsed[2]) : "";
        return [key, value];
    }

    function _stringifyPair(entry) {
        var pair = encodeURIComponent(entry[0]);
        if (entry[1]) {
            pair += "=" + encodeURIComponent(entry[1]);
        }
        return pair;
    }

    function _createMapIterator(url, kind) {
        var query = "";
        var pairs = [];
        var index = 0;
        return {
            next: function() {
                if (query !== url.query) {
                    query = url.query;
                    pairs = _parseQuery(query);
                }
                if (index < pairs.length) {
                    var entry = _parsePair(pairs[index++]);
                    switch (kind) {
                        case "keys":
                            return entry[0];
                        case "values":
                            return entry[1];
                        case "keys+values":
                            return [entry[0], entry[1]];
                        default:
                            throw new TypeError();
                    }
                }
                throw STOP_ITERATION;
            }
        };
    }

    // See http://url.spec.whatwg.org/#interface-urlquery
    function URLQuery(url) {
        Object.defineProperty(this, "_url", {
            get: function() {
                return url._url;
            }
        });
    }

    Object.defineProperties(URLQuery.prototype, {
        get: {
            value: function(key) {
                _checkString(key);
                var result;
                var pairs = _parseQuery(this._url.query);
                pairs.some(function(pair) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        result = entry[1];
                        return true;
                    }
                });
                return result;
            },
            enumerable: true
        },
        set: {
            value: function(key, value) {
                _checkString(key);
                _checkString(value);
                var pairs = _parseQuery(this._url.query);
                var found = pairs.some(function(pair, i) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        entry[1] = value;
                        pairs[i] = _stringifyPair(entry);
                        return true;
                    }
                });
                if (!found) {
                    pairs.push(_stringifyPair([key, value]));
                }
                this._url.query = _stringifyQuery(pairs);
            },
            enumerable: true
        },
        has: {
            value: function(key) {
                _checkString(key);
                var pairs = _parseQuery(this._url.query);
                return pairs.some(function(pair) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        return true;
                    }
                });
            },
            enumerable: true
        },
            "delete": {
            value: function(key) {
                _checkString(key);
                var pairs = _parseQuery(this._url.query);
                var filtered = pairs.filter(function(pair) {
                    var entry = _parsePair(pair);
                    return entry[0] !== key;
                });
                if (filtered.length !== pairs.length) {
                    this._url.query = _stringifyQuery(filtered);
                    return true;
                }
                return false;
            },
            enumerable: true
        },
        clear: {
            value: function() {
                this._url.query = "";
            },
            enumerable: true
        },
        forEach: {
            value: function(callback, thisArg) {
                if (typeof callback !== "function") {
                    throw new TypeError();
                }
                var iterator = _createMapIterator(this._url, "keys+values");
                try {
                    while (true) {
                        var entry = iterator.next();
                        callback.call(thisArg, entry[1], entry[0], this);
                    }
                } catch (e) {
                    if (e !== STOP_ITERATION) {
                        throw e;
                    }
                }
            },
            enumerable: true
        },
        keys: {
            value: function() {
                return _createMapIterator(this._url, "keys");
            },
            enumerable: true
        },
        values: {
            value: function() {
                return _createMapIterator(this._url, "values");
            },
            enumerable: true
        },
        items: {
            value: function() {
                return _createMapIterator(this._url, "keys+values");
            }
        },
        size: {
            get: function() {
                return _parseQuery(this._url.query).length;
            },
            enumerable: true
        },
        getAll: {
            value: function(key) {
                _checkString(key);
                var result = [];
                var pairs = _parseQuery(this._url.query);
                pairs.forEach(function(pair) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        result.push(entry[1]);
                    }
                });
                return result;
            },
            enumerable: true
        },
        append: {
            value: function(key, value) {
                _checkString(key);
                _checkString(value);
                var pairs = _parseQuery(this._url.query);
                pairs.push(_stringifyPair([key, value]));
                this._url.query = _stringifyQuery(pairs);
            },
            enumerable: true
        }
    });

    function _makeAbsoluteURL(url, base) {
        if (!url.scheme && base) {
            url.scheme = base.scheme;
            if (!url.host && base.host) {
                url.userinfo = base.userinfo;
                url.host = base.host;
                url.port = base.port;
                url.pathRelative = true;
            }
        }
        if (url.pathRelative) {
            if (!url.path) {
                url.path = base.path;
            } else if (url.path[0] !== "/") {
                var basePath = /^(.*\/)[^\/]*$/.exec(base.path)[1] || "/";
                url.path = basePath + url.path;
            }
        }
    }

    function _normalizeScheme(scheme) {
        return scheme.toLowerCase();
    }

    function _normalizePort(port) {
        return port ? (/[1-9]\d*$/).exec(port)[0] : "";
    }

    function _normalizePath(path) {
        var result = [];
        path.split("/").forEach(function(segment) {
            if (segment === "..") {
                result.pop();
            } else if (segment !== ".") {
                result.push(segment);
            }
        });
        return result.join("/");
    }


    function _normalizeURL(url) {
        if (url.scheme) {
            url.scheme = _normalizeScheme(url.scheme);
        }
        if (url.port) {
            url.port = _normalizePort(url.port);
        }
        if (url.host && url.path) {
            url.path = _normalizePath(url.path);
        }
    }

    function _encodeWhitespace(text) {
        return text.replace(/\s/g, function(c) {
            return "%" + c.charCodeAt(0).toString(16);
        });
    }

    function _parseURL(input, base) {
        if (typeof input !== "string") {
            throw new TypeError();
        }

        input = _encodeWhitespace(input);

        var parsedURI = _URI_RE.exec(input);
        if (!parsedURI) {
            return null;
        }
        var url = {};
        url.scheme = parsedURI[1] || "";
        if (url.scheme && !_SCHEME_RE.test(url.scheme)) {
            return null;
        }
        var authority = parsedURI[2];
        if (authority) {
            var parsedAuthority = _AUTHORITY_RE.exec(authority);
            url.userinfo = parsedAuthority[1];
            url.host = parsedAuthority[2];
            url.port = parsedAuthority[3];
            if (url.port && !_PORT_RE.test(url.port)) {
                return null;
            }
        }
        url.path = parsedURI[3];
        url.query = parsedURI[4];
        url.fragment = parsedURI[5];

        _makeAbsoluteURL(url, base);
        _normalizeURL(url);
        return url;
    }

    function _serialize(url) {
        var result = (url.scheme ? url.scheme + ":" : "");
        if (url.host) {
            result += "//";
            if (url.userinfo) {
                result += url.userinfo + "@";
            }
            result += url.host;
            if (url.port) {
                result += ":" + url.port;
            }
        }
        result += url.path;
        if (url.query) {
            result += "?" + url.query;
        }
        if (url.fragment) {
            result += "#" + url.fragment;
        }
        return result;
    }

    // See http://url.spec.whatwg.org/#api
    function URL(input, base) {
        var baseURL;
        if (base) {
            base = base.href || base;
            baseURL = _parseURL(base);
            if (!baseURL || !baseURL.scheme) {
                throw new SyntaxError();
            }
            Object.defineProperty(this, "_baseURL", {
                value: baseURL
            });
        }

        var url = _parseURL(input, baseURL);
        if (!url) {
            throw new SyntaxError();
        }

        Object.defineProperty(this, "_input", {
            value: input,
            writable: true
        });

        Object.defineProperty(this, "_url", {
            value: url,
            writable: true
        });

        var query = new URLQuery(this);
        Object.defineProperty(this, "query", {
            get: function() {
                return this._url ? query : null;
            },
            enumerable: true
        });
    }

    Object.defineProperties(URL.prototype, {
        href: {
            get: function() {
                return this._url ? _serialize(this._url) : this._input;
            },
            set: function(value) {
                _checkString(value);
                this._input = value;
                this._url = _parseURL(this._input, this._baseURL);
            },
            enumerable: true
        },
        origin: {
            get: function() {
                return (this._url && this._url.host ? this.protocol + "//" + this.host : "");
            },
            enumerable: true
        },
        protocol: {
            get: function() {
                return this._url ? this._url.scheme + ":" : ":";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var scheme = (value.slice(-1) === ":") ? value.substring(0, value.length - 1) : value;
                if (scheme === "" || _SCHEME_RE.test(scheme)) {
                    this._url.scheme = _normalizeScheme(scheme);
                }

            },
            enumerable: true
        },
        _userinfo: { // note: not part of spec so not enumerable
            get: function() {
                return this._url ? this._url.userinfo : "";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                this._url.userinfo = value;
            }
        },
        username: {
            get: function() {
                if (!this._url) {
                    return "";
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var username = decodeURIComponent(parsed[1] || "");
                return username;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var userpass = [encodeURIComponent(value || "")];
                if (parsed[2]) {
                    userpass.push(parsed[2]);
                }
                this._userinfo = userpass.join(":");
            },
            enumerable: true
        },
        password: {
            get: function() {
                if (!this._url) {
                    return "";
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var password = decodeURIComponent(parsed[2] || "");
                return password;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var userpass = [parsed[1] || ""];
                if (value) {
                    userpass.push(encodeURIComponent(value));
                }
                this._userinfo = userpass.join(":");
            },
            enumerable: true
        },
        host: {
            get: function() {
                var result = "";
                if (this._url && this._url.host) {
                    result += this._url.host;
                    if (this._url.port) {
                        result += ":" + this._url.port;
                    }
                }
                return result;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _HOSTPORT_RE.exec(value);
                if (result) {
                    this._url.host = result[1];
                    this._url.port = _normalizePort(result[2]);
                }
            },
            enumerable: true
        },
        hostname: {
            get: function() {
                return this._url ? this._url.host : "";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _HOST_RE.exec(value);
                if (result) {
                    this._url.host = value;
                }
            },
            enumerable: true
        },
        port: {
            get: function() {
                var port = this._url ? this._url.port || "" : "";
                if (port && port === DEFAULT_PORTS[this.protocol]) {
                    port = "";
                }
                return port;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _PORT_RE.exec(value);
                if (result) {
                    this._url.port = _normalizePort(value);
                }
            },
            enumerable: true
        },
        pathname: {
            get: function() {
                return this._url ? this._url.path : "";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _PATH_RE.exec(value);
                if (result) {
                    if (this._url.host && value && value[0] !== "/") {
                        value = "/" + value;
                    }
                    this._url.path = value ? _normalizePath(value) : "";
                }
            },
            enumerable: true
        },
        search: {
            get: function() {
                return (this._url && this._url.query ? "?" + this._url.query : "");
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                if (value && value[0] === "?") {
                    value = value.substring(1);
                }
                var result = _QUERY_RE.exec(value);
                if (result) {
                    this._url.query = value;
                }
            },
            enumerable: true
        },
        hash: {
            get: function() {
                return (this._url && this._url.fragment ? "#" + this._url.fragment : "");
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                if (value && value[0] === "#") {
                    value = value.substring(1);
                }
                var result = _FRAGMENT_RE.exec(value);
                if (result) {
                    this._url.fragment = value;
                }
            },
            enumerable: true
        }
    });

	var _URL = window.URL || window.webkitURL;
    if (_URL && _URL.createObjectURL) {
        Object.defineProperty(URL, "createObjectURL", {
            value: _URL.createObjectURL.bind(_URL),
            enumerable: false
        });

        Object.defineProperty(URL, "revokeObjectURL", {
            value: _URL.revokeObjectURL.bind(_URL),
            enumerable: false
        });
    }
    window.URL = URL;
}());

define("orion/URL-shim", function(){});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global URL*/
/**
 * @name orion.xhr
 * @namespace Provides a promise-based API to {@link XMLHttpRequest}.
 */
define('orion/xhr',[
	'orion/Deferred',
	'orion/xsrfUtils',
	'orion/URL-shim', // no exports, must come last
], function(Deferred, xsrfUtils) {

	/**
	 * @name orion.xhr.Result
	 * @class Wraps an XHR response or failure.
	 * @property {Object} args Arguments passed to the {@link orion.xhr.xhr} call.
	 * @property {Object|ArrayBuffer|Blob|Document|String} response The <code>response</code> object returned by the XMLHttpRequest.
	 * It is typed according to the <code>responseType</code> passed to the XHR call (by default it is a {@link String}).
	 * @property {String} [responseText] The <code>response</code> returned by the XMLHttpRequest, if it is a {@link String}.
	 * If the <code>response</code> is not a String, this property is <code>null</code>.
	 * @property {Number} status The HTTP status code returned by the XMLHttpRequest.
	 * @property {String} url The URL that the XHR request was made to.
	 * @property {XMLHttpRequest} xhr The underlying XMLHttpRequest object.
	 * @property {String|Error} error <i>Optional</i>. If a timeout occurred or an error was thrown while performing the
	 * XMLHttpRequest, this field contains information about the error.
	 */

	/**
	 * @param {String} url
	 * @param {Object} options
	 * @param {XMLHttpRequest} xhr
	 * @param {String|Error} [error]
	 */
	function makeResult(url, options, xhr, error) {
		var response = typeof xhr.response !== 'undefined' ? xhr.response : xhr.responseText; //$NON-NLS-0$
		var responseText = typeof response === 'string' ? response : null; //$NON-NLS-0$
		var status;
		try {
			status = xhr.status;
		} catch (e) {
			status = 0;
		}
		var result = {
			args: options,
			response: response,
			responseText: responseText,
			status: status,
			url: url,
			xhr: xhr
		};
		if (typeof error !== 'undefined') { //$NON-NLS-0$
			result.error = error;
		}
		return result;
	}

	function isSameOrigin(url) {
		return new URL(location.href).origin === new URL(url, location.href).origin;
	}

	/**
	 * Wrapper for {@link XMLHttpRequest} that returns a promise.
	 * @name xhr
	 * @function
	 * @memberOf orion.xhr
	 * @param {String} method One of 'GET', 'POST', 'PUT', 'DELETE'.
	 * @param {String} url The URL to request.
	 * @param {Object} [options]
	 * @param {Object|ArrayBuffer|Blob|Document} [options.data] The raw data to send as the request body. (Only allowed for POST and PUT).
	 * @param {Object} [options.headers] A map of header names and values to set on the request.
	 * @param {Boolean} [options.log=false] If <code>true</code>, failed requests will be logged to the JavaScript console.
	 * @param {String} [options.responseType=''] Determines the type of the response object returned. Value must be one of the following:
	 * <ul><li><code>'arraybuffer'</code>
	 * <li><code>'blob'</code>
	 * <li><code>'document'</code>
	 * <li><code>'json'</code>
	 * <li><code>'text'</code>
	 * <li><code>''</code> (same as <code>'text'</code>)</ul>
	 * @param {Number} [options.timeout=0] Timeout in milliseconds. Defaults to 0 (no timeout).
	 * @returns {Deferred} A deferred for the result. The deferred will resolve on 2xx, 3xx status codes or reject on 4xx, 5xx status codes.
	 * In both cases a {@link orion.xhr.Result} is provided to the listener.
	 */
	// TODO: upload progress, user/password
	function _xhr(method, url, options/*, XMLHttpRequestImpl */) {
		options = options || {};
		var xhr = (arguments.length > 3 && arguments[3]) ? arguments[3] : new XMLHttpRequest(); //$NON-NLS-0$
		var d = new Deferred();
		var headers = options.headers || {};
		if (isSameOrigin(url)) {
			xsrfUtils.setNonceHeader(headers);
		}
		var log = options.log || false;
		var data;
		if (typeof headers['X-Requested-With'] === 'undefined') { //$NON-NLS-1$ //$NON-NLS-0$
			headers['X-Requested-With'] = 'XMLHttpRequest'; //$NON-NLS-1$ //$NON-NLS-0$
		}
		if (typeof options.data !== 'undefined' && (method === 'POST' || method === 'PUT')) { //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
			data = options.data;
		}
		
		var cancelled = false;
		var aborted = false;
		d.promise.then(undefined, function(error) {
			cancelled = true;
			if (!aborted && error instanceof Error && error.name === "Cancel") {
				xhr.abort();
			}
		});
		
		xhr.onabort = function() {
			aborted = true;
			if (!cancelled) {
				var cancelError = new Error("Cancel");
				cancelError.name = "Cancel";
				d.reject(cancelError);
			}
		};
		xhr.onload = function() {
			var result = makeResult(url, options, xhr);
			if(200 <= xhr.status && xhr.status < 400) {
				d.resolve(result);
			} else {
				d.reject(result);
				if(log && typeof console !== 'undefined') { //$NON-NLS-0$
					console.log(new Error(xhr.statusText));
				}
			}
		};
		xhr.onerror = function() {
			var result = makeResult(url, options, xhr);
			d.reject(result);
			if (log && typeof console !== 'undefined') { //$NON-NLS-0$
				console.log(new Error(xhr.statusText));
			}
		};
		xhr.onprogress = function(progressEvent) {
			progressEvent.xhr = xhr;
			d.progress(progressEvent);
		};
	
		try {
			xhr.open(method, url, true /* async */);
			if (typeof options.responseType === 'string') { //$NON-NLS-0$
				xhr.responseType = options.responseType;
			}
			if (typeof options.timeout === 'number') { //$NON-NLS-0$
				if (typeof xhr.timeout === 'number') { //$NON-NLS-0$
					// Browser supports XHR timeout
					xhr.timeout = options.timeout;
					xhr.addEventListener('timeout', function(e) { //$NON-NLS-0$
						d.reject(makeResult(url, options, xhr, 'Timeout exceeded')); //$NON-NLS-0$
					});
				} else {
					// Use our own timer
					var timeoutId = setTimeout(function() {
						d.reject(makeResult(url, options, xhr, 'Timeout exceeded')); //$NON-NLS-0$
					}, options.timeout);
					d.promise.then(clearTimeout.bind(null, timeoutId), clearTimeout.bind(null, timeoutId));
				}
			}
			Object.keys(headers).forEach(function(key) {
				xhr.setRequestHeader(key, headers[key]);
			});
			xhr.send(data || null);
		} catch (e) {
			d.reject(makeResult(url, options, xhr, e));
		}

		return d.promise;
	}
	return _xhr;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/EventTarget',[],function() {
	/**
	 * Creates an Event Target
	 *
	 * @name orion.EventTarget
	 * @class Base for creating an Orion event target
	 */
	function EventTarget() {
		this._namedListeners = {};
	}

	EventTarget.prototype = /** @lends orion.EventTarget.prototype */
	{
		/**
		 * Dispatches a named event along with an arbitrary set of arguments. Any arguments after <code>eventName</code>
		 * will be passed to the event listener(s).
		 * @param {Object} event The event to dispatch. The event object MUST have a type field
		 * @returns {boolean} false if the event has been canceled and any associated default action should not be performed
		 * listeners (if any) have resolved.
		 */
		dispatchEvent: function(event) {
			if (!event.type) {
				throw new Error("unspecified type");
			}
			var listeners = this._namedListeners[event.type];
			if (listeners) {
				listeners.forEach(function(listener) {
					try {
						if (typeof listener === "function") {
							listener(event);
						} else {
							listener.handleEvent(event);
						}
					} catch (e) {
						if (typeof console !== 'undefined') {
							console.log(e); // for now, probably should dispatch an ("error", e)
						}
					}			
				});
			}
			return !event.defaultPrevented;
		},

		/**
		 * Adds an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		addEventListener: function(eventName, listener) {
			if (typeof listener === "function" || listener.handleEvent) {
				this._namedListeners[eventName] = this._namedListeners[eventName] || [];
				this._namedListeners[eventName].push(listener);
			}
		},

		/**
		 * Removes an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		removeEventListener: function(eventName, listener) {
			var listeners = this._namedListeners[eventName];
			if (listeners) {
				for (var i = 0; i < listeners.length; i++) {
					if (listeners[i] === listener) {
						if (listeners.length === 1) {
							delete this._namedListeners[eventName];
						} else {
							listeners.splice(i, 1);
						}
						break;
					}
				}
			}
		}
	};
	EventTarget.prototype.constructor = EventTarget;
	
	EventTarget.attach = function(obj) {
		var eventTarget = new EventTarget();
		obj.dispatchEvent = eventTarget.dispatchEvent.bind(eventTarget);
		obj.addEventListener = eventTarget.addEventListener.bind(eventTarget);
		obj.removeEventListener = eventTarget.removeEventListener.bind(eventTarget);
	};
	
	return EventTarget;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/serviceregistry',["orion/Deferred", "orion/EventTarget"], function(Deferred, EventTarget) {

	/**
	 * @name orion.serviceregistry.ServiceReference
	 * @description Creates a new service reference.
	 * @class A reference to a service in the Orion service registry
	 * @param {String} serviceId The symbolic id of this service instance
	 * @param {String} name The service name
	 * @param {Object} properties A JSON object containing the service's declarative properties
	 */
	function ServiceReference(serviceId, objectClass, properties) {
		this._properties = properties || {};
		this._properties["service.id"] = serviceId;
		this._properties.objectClass = objectClass;
		this._properties["service.names"] = objectClass;
	}

	ServiceReference.prototype = /** @lends orion.serviceregistry.ServiceReference.prototype */
	{
		/**
		 * @name getPropertyKeys
		 * @description Returns the names of the declarative properties of this service.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceReference.prototype
		 * @returns the names of the declarative properties of this service
		 */
		getPropertyKeys: function() {
			var result = [];
			var name;
			for (name in this._properties) {
				if (this._properties.hasOwnProperty(name)) {
					result.push(name);
				}
			}
			return result;
		},
		/**
		 * @name getProperty
		 * @description Returns the declarative service property with the given name, or <code>undefined</code>
		 * if this service does not have such a property.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceReference.prototype
		 * @param {String} propertyName The name of the service property to return
		 * @returns The {String} property with the given name or <code>undefined</code>
		 */
		getProperty: function(propertyName) {
			return this._properties[propertyName];
		}
	};
	ServiceReference.prototype.constructor = ServiceReference;

	/**
	 * @name orion.serviceregistry.ServiceRegistration
	 * @description Creates a new service registration. This constructor is private and should only be called by the service registry.
	 * @class A reference to a registered service in the Orion service registry
	 * @param {String} serviceId The symbolic id of this service
	 * @param {String} serviceReference A reference to the service
	 * @param {Object} internalRegistry A JSON object containing the service's declarative properties
	 */
	function ServiceRegistration(serviceId, serviceReference, internalRegistry) {
		this._serviceId = serviceId;
		this._serviceReference = serviceReference;
		this._internalRegistry = internalRegistry;
	}

	ServiceRegistration.prototype = /** @lends orion.serviceregistry.ServiceRegistration.prototype */
	{
		/**
		 * @name unregister
		 * @description Unregister this service registration. Clients registered for <code>unregistering</code> service events
		 * will be notified of this change.
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 */
		unregister: function() {
			this._internalRegistry.unregisterService(this._serviceId);
		},

		/**
		 * @name getReference
		 * @description Returns the {@link orion.serviceregistry.ServiceReference} in this registration
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 * @param properties
		 * @returns the {@link orion.serviceregistry.ServiceReference} in this registration
		 * @throws An error is the service has been unregistered
		 */
		getReference: function() {
			if (!this._internalRegistry.isRegistered(this._serviceId)) {
				throw new Error("Service has already been unregistered: "+this._serviceId);
			}
			return this._serviceReference;
		},
		/**
		 * @name setProperties
		 * @description Sets the properties of this registration to the new given properties. Clients registered for <code>modified</code> service events
		 * will be notified of the change.
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 * @param {Object} properties
		 */
		setProperties: function(properties) {
			var oldProperties = this._serviceReference._properties;
			this._serviceReference._properties = properties || {};
			properties["service.id"] = this._serviceId;
			properties.objectClass = oldProperties.objectClass;
			properties["service.names"] = oldProperties["service.names"];
			this._internalRegistry.modifyService(this._serviceId);
		}
	};
	ServiceRegistration.prototype.constructor = ServiceRegistration;

	/**
	 * @name orion.serviceregistry.DeferredService
	 * @description Creates a new service promise to resolve the service at a later time.
	 * @class A service that is resolved later
	 * @private
	 * @param {orion.serviceregistry.ServiceReference} implementation The implementation of the service
	 * @param {Function} isRegistered A function to call to know if the service is already registered
	 */
	function DeferredService(implementation, isRegistered) {

		function _createServiceCall(methodName) {
			return function() {
					var d;
					try {
						if (!isRegistered()) {
							throw new Error("Service was unregistered");
						}
						var result = implementation[methodName].apply(implementation, Array.prototype.slice.call(arguments));
						if (result && typeof result.then === "function") {
							return result;
						} else {
							d = new Deferred();
							d.resolve(result);
						}
					} catch (e) {
							d = new Deferred();
							d.reject(e);
					}
					return d.promise;
			};
		}

		var method;
		for (method in implementation) {
			if (typeof implementation[method] === 'function') {
				this[method] = _createServiceCall(method);
			}
		}
	}

	/**
	 * @name orion.serviceregistry.ServiceEvent
	 * @description An event that is fired from the service registry. Clients must register to listen to events using 
	 * the {@link orion.serviceregistry.ServiceRegistry#addEventListener} function.
	 * <br> 
	 * There are three types of events that this registry will send:
	 * <ul>
	 * <li>modified - the service has been modified</li> 
	 * <li>registered - the service has been registered</li> 
	 * <li>unregistering - the service is unregistering</li>
	 * </ul> 
	 * @class A service event
	 * @param {String} type The type of the event, one of <code>modified</code>, <code>registered</code> or <code>unregistered</code>
	 * @param {orion.serviceregistry.ServiceReference} serviceReference the service reference the event is for
	 */
	function ServiceEvent(type, serviceReference) {
		this.type = type;
		this.serviceReference = serviceReference;
	}

	/**
	 * @name orion.serviceregistry.ServiceRegistry
	 * @description Creates a new service registry
	 * @class The Orion service registry
	 */
	function ServiceRegistry() {
		this._entries = [];
		this._namedReferences = {};
		this._serviceEventTarget = new EventTarget();
		var _this = this;
		this._internalRegistry = {
			/**
			 * @name isRegistered
			 * @description Returns if the service with the given identifier is registered or not.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @returns <code>true</code> if the service with the given identifier is registered, <code>false</code> otherwise
			 */
			isRegistered: function(serviceId) {
				return _this._entries[serviceId] ? true : false;
			},
			
			/**
			 * @name unregisterService
			 * @description Unregisters a service with the given identifier. This function will notify
			 * clients registered for <code>unregistering</code> service events.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @throws An error if the service has already been unregistered
			 * @see orion.serviceregistry.ServiceEvent
			 */
			unregisterService: function(serviceId) {
				var entry = _this._entries[serviceId];
				if (!entry) {
					throw new Error("Service has already been unregistered: "+serviceId);
				}
				var reference = entry.reference;
				_this._serviceEventTarget.dispatchEvent(new ServiceEvent("unregistering", reference));
				_this._entries[serviceId] = null;
				var objectClass = reference.getProperty("objectClass");
				objectClass.forEach(function(type) {
					var namedReferences = _this._namedReferences[type];
					for (var i = 0; i < namedReferences.length; i++) {
						if (namedReferences[i] === reference) {
							if (namedReferences.length === 1) {
								delete _this._namedReferences[type];
							} else {
								namedReferences.splice(i, 1);
							}
							break;
						}
					}
				});
			},
			/**
			 * @name modifyService
			 * @description Notifies that the service with the given identifier has been modified. This function will notify clients
			 * registered for <code>modified</code> service events.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @throws An error if the service for the given identifier does not exist
			 * @see orion.serviceregistry.ServiceEvent
			 */
			modifyService: function(serviceId) {
				var entry = _this._entries[serviceId];
				if (!entry) {
					throw new Error("Service not found while trying to send modified event: "+serviceId);
				}
				var reference = entry.reference;
				_this._serviceEventTarget.dispatchEvent(new ServiceEvent("modified", reference));
			}
		};
	}

	ServiceRegistry.prototype = /** @lends orion.serviceregistry.ServiceRegistry.prototype */
	{
		/**
		 * @name getService
		 * @description Returns the service with the given name or reference.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String|orion.serviceregistry.ServiceReference} nameOrServiceReference The service name or a service reference
		 * @returns {orion.serviceregistry.ServiceReference|null} The service implementation, or <code>null</code> if no such service was found.
		 */
		getService: function(typeOrServiceReference) {
			var service;
			if (typeof typeOrServiceReference === "string") {
				var references = this._namedReferences[typeOrServiceReference];
				if (references) {
					references.some(function(reference) {
						service = this._entries[reference.getProperty("service.id")].service;
						return !!service;
					}, this);
				}
			} else {
				var entry = this._entries[typeOrServiceReference.getProperty("service.id")];
				if (entry) {
					service = entry.service;
				}
			}
			return service || null;
		},

		/**
		 * @name getServiceReferences
		 * @description Returns all references to the service with the given name.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} name The name of the service to return
		 * @returns {orion.serviceregistry.ServiceReference[]} An array of service references
		 */
		getServiceReferences: function(name) {
			if (name) {
				return this._namedReferences[name] ? this._namedReferences[name] : [];
			}
			var result = [];
			this._entries.forEach(function(entry) {
				if (entry) {
					result.push(entry.reference);
				}
			});
			return result;
		},
		
		/**
		 * @name registerService
		 * @description Registers a service with this registry. This function will notify clients registered
		 * for <code>registered</code> service events.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String|String[]} names the name or names of the service being registered
		 * @param {Object} service The service implementation
		 * @param {Object} properties A JSON collection of declarative service properties
		 * @returns {orion.serviceregistry.ServiceRegistration} A service registration object for the service.
		 * @see orion.serviceregistry.ServiceEvent
		 */
		registerService: function(names, service, properties) {
			if (typeof service === "undefined" ||  service === null) {
				throw new Error("invalid service");
			}

			if (typeof names === "string") {
				names = [names];
			} else if (!Array.isArray(names)) {
				names = [];
			}

			var serviceId = this._entries.length;
			var reference = new ServiceReference(serviceId, names, properties);
			var namedReferences = this._namedReferences;
			names.forEach(function(name) {
				namedReferences[name] = namedReferences[name] || [];
				namedReferences[name].push(reference);
			});
			var deferredService = new DeferredService(service, this._internalRegistry.isRegistered.bind(null, serviceId));
			this._entries.push({
				reference: reference,
				service: deferredService
			});
			var internalRegistry = this._internalRegistry;
			this._serviceEventTarget.dispatchEvent(new ServiceEvent("registered", reference));
			return new ServiceRegistration(serviceId, reference, internalRegistry);
		},

		/**
		 * @name addEventListener
		 * @description Adds a listener for events on this registry.
		 * <br> 
		 * The events that this registry notifies about:
		 * <ul>
		 * <li>modified - the service has been modified</li> 
		 * <li>registered - the service has been registered</li> 
		 * <li>unregistering - the service is unregistering</li> 
		 * </ul> 
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} eventName The name of the event to be notified about.
		 * @param {Function} listener The listener to add
		 * @see orion.serviceregistry.ServiceEvent
		 */
		addEventListener: function(eventName, listener) {
			this._serviceEventTarget.addEventListener(eventName, listener);
		},

		/**
		 * @name removeEventListener
		 * @description Removes a listener for service events in this registry.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} eventName The name of the event to stop listening for
		 * @param {Function} listener The listener to remove
		 * @see orion.serviceregistry.ServiceEvent
		 */
		removeEventListener: function(eventName, listener) {
			this._serviceEventTarget.removeEventListener(eventName, listener);
		}
	};
	ServiceRegistry.prototype.constructor = ServiceRegistry;

	//return the module exports
	return {
		ServiceRegistry: ServiceRegistry
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/**
 * @name orion.operation
 * @namespace Provides an API for handling long running operations as promises.
 */
define('orion/operation',["orion/xhr", "orion/Deferred"], function(xhr, Deferred) {

	function _isRunning(operationType) {
		if (!operationType) {
			return true;
		}
		if (operationType === "loadstart" || operationType === "progress") {
			return true;
		}
		return false;
	}

	function _deleteTempOperation(operationLocation) {
		xhr("DELETE", operationLocation, {
			headers: {
				"Orion-Version": "1"
			},
			timeout: 15000
		});
	}

	function _cancelOperation(operationLocation) {
		xhr("PUT", operationLocation, {
			headers: {
				"Orion-Version": "1"
			},
			data: JSON.stringify({
				abort: true
			}),
			timeout: 15000
		});
	}

	function _getOperation(operation, deferred, onResolve, onReject) {
		xhr("GET", operation.location, {
			headers: {
				"Orion-Version": "1"
			},
			timeout: 15000
		}).then(function(result) {
			var operationJson = result.response ? JSON.parse(result.response) : null;
			deferred.progress(operationJson);
			if (_isRunning(operationJson.type)) {
				setTimeout(function() {
					_getOperation(operation	, deferred, onResolve, onReject);
				}, operation.timeout);
				operation.timeout = Math.min(operation.timeout * 2, 2000);
				return;
			}
			if (operationJson.type === "error" || operationJson.type === "abort") {
				deferred.reject(onReject ? onReject(operationJson) : operationJson.Result);
			} else {
				deferred.resolve(onResolve ? onResolve(operationJson) : operationJson.Result.JsonData);
			}
			if (!operationJson.Location) {
				_deleteTempOperation(operation.location); //This operation should not be kept 
			}
		}, function(error) {
			var errorMessage = error;
			if (error.responseText !== undefined) {
				errorMessage = error.responseText;
				try {
					errorMessage = JSON.parse(error.responseText);
				} catch (e) {
					//ignore
				}
			}
			if (errorMessage.Message !== undefined) {
				errorMessage.HttpCode = errorMessage.HttpCode === undefined ? error.status : errorMessage.HttpCode;
				errorMessage.Severity = errorMessage.Severity === undefined ? "Error" : errorMessage.Severity;
				deferred.reject(errorMessage);
			} else {
				deferred.reject({
					Severity: "Error",
					Message: errorMessage,
					HttpCode: error.status
				});
			}
		});
	}

	function _trackCancel(operationLocation, deferred) {
		deferred.then(null, function(error) {
			if (error instanceof Error && error.name === "Cancel") {
				_cancelOperation(operationLocation);
			}
		});
	}

	/**
	 * Handles a long-running operation as a promise.
	 * @name orion.operation.handle
	 * @function
	 * @param {String} operationLocation
	 * @param {Function} [onSuccess] If provided, will be called to transform a successful operation into the resolve value of the 
	 * returned promise.
	 * @param {Function} [onError] If provided, will be called to trasnform a failed operation into the reject value of the 
	 * returned promise.
	 * @returns {orion.Promise}
	 */
	function handle(operationLocation, onSuccess, onError) {
		var def = new Deferred();
		_trackCancel(operationLocation, def);
		_getOperation({location: operationLocation, timeout: 100}, def, onSuccess, onError);
		return def;
	}

	return {
		handle: handle
	};
});

/******************************************************************************* 
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

/** @namespace The global container for eclipse APIs. */

define('orion/git/gitClient',['require', "orion/xhr", "orion/Deferred", 'orion/operation'], function(require, xhr, Deferred, operation) {

var eclipse = eclipse || {};

eclipse.GitService = (function() {
	
	var contentType = "application/json; charset=UTF-8";
	var GIT_TIMEOUT = 60000;
	
	/**
	 * Creates a new Git service.
	 * @class Provides operations for browsing and manipulating Git repositories.
	 * @name orion.git.GitService
	 */
	function GitService(serviceRegistry) {
		if (serviceRegistry) {
			this._serviceRegistry = serviceRegistry;
			this._serviceRegistration = serviceRegistry.registerService(
					"orion.git.provider", this); //$NON-NLS-0$
		}
	}

	GitService.prototype = /** @lends eclipse.GitService.prototype */
	{
		checkGitService : function() {
		},
		
		cloneGitRepository : function(gitName, gitRepoUrl, targetPath, repoLocation, gitSshUsername, gitSshPassword, gitSshKnownHost, privateKey, passphrase, userInfo, initProject) {
			var service = this;
			var postData = {};
			if(gitName){
				postData.Name = gitName;
			}
			if(targetPath){
				postData.Path = targetPath;
			}
			if(gitRepoUrl){
				postData.GitUrl=gitRepoUrl;
			}
			postData.Location = repoLocation;
			if(gitSshUsername){
				postData.GitSshUsername = gitSshUsername;
			}
			if(gitSshPassword){
				postData.GitSshPassword = gitSshPassword;
			}
			if(gitSshKnownHost){
				postData.GitSshKnownHost = gitSshKnownHost;
			}
			if(privateKey) {
				postData.GitSshPrivateKey=privateKey;
			}
			if(passphrase) {
				postData.GitSshPassphrase=passphrase;
			}
			if(userInfo){
				if( userInfo.GitMail ){
					postData.GitMail = userInfo.GitMail;
				}
				if( userInfo.GitName ){
					postData.GitName = userInfo.GitName;
				}
			}
			if(initProject){
				postData.initProject = initProject;
			}
			//NOTE: require.toURL needs special logic here to handle "gitapi/clone"
			var gitapiCloneUrl = require.toUrl("gitapi/clone/._"); //$NON-NLS-0$
			gitapiCloneUrl = gitapiCloneUrl.substring(0,gitapiCloneUrl.length-2);
			
			var clientDeferred = new Deferred();
			xhr("POST", gitapiCloneUrl, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify(postData)
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		removeGitRepository : function(repositoryLocation){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("DELETE", repositoryLocation, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
	
		getGitStatus: function(url){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", url, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		stage: function(location){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		stageMultipleFiles: function(gitCloneURI, paths){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitCloneURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Path" : paths //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		unstageAll: function(location , resetParam){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Reset" : resetParam //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		unstage: function(location , paths){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$,
				data: JSON.stringify({
					"Path" : paths //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		checkoutPath: function(gitCloneURI, paths){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitCloneURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Path" : paths, //$NON-NLS-0$
					"RemoveUntracked" : "true" //$NON-NLS-1$ //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		ignorePath: function(gitIgnoreURI, paths){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitIgnoreURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Path" : paths, //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		commitAll: function(location , message , body){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: body
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		getGitClone : function(gitCloneURI) {
			var service = this;

			var clientDeferred = new Deferred();
			xhr("GET", gitCloneURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		getGitCloneConfig : function(gitCloneConfigURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", gitCloneConfigURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		getGitBranch : function(gitBranchURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", gitBranchURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		getGitRemote : function(gitRemoteURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", gitRemoteURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		checkoutBranch : function(gitCloneURI, branchName) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitCloneURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Branch" : branchName //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		resetIndex : function(gitIndexURI, refId, mode) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitIndexURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Commit" : refId, //$NON-NLS-0$
					"Reset" : mode ? mode : "HARD" //$NON-NLS-1$ //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		addBranch : function(gitBranchParentURI, branchName, startPoint) {
			var service = this;
			
			var postData = {};
			if (branchName) postData.Name = branchName;
			if (startPoint) postData.Branch = startPoint;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitBranchParentURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify(postData)
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		removeBranch : function(gitBranchURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("DELETE", gitBranchURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		addRemote : function(gitRemoteParentURI, remoteName, remoteURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitRemoteParentURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Remote" : remoteName, //$NON-NLS-0$
					"RemoteURI" : remoteURI //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		removeRemote : function(gitRemoteURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("DELETE", gitRemoteURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doGitLog : function(gitLogURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", gitLogURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		getDiff : function(gitDiffURI, commitName) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitDiffURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"New" : commitName //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doGitDiff : function(gitDiffURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", gitDiffURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});

			return clientDeferred;
		},
		
		doFetch : function(gitRemoteBranchURI, force, gitSshUsername, gitSshPassword, gitSshKnownHost, gitPrivateKey, gitPassphrase) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitRemoteBranchURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Fetch" : "true", //$NON-NLS-1$ //$NON-NLS-0$
					"Force" : force, //$NON-NLS-0$
					"GitSshUsername" : gitSshUsername, //$NON-NLS-0$
					"GitSshPassword" : gitSshPassword, //$NON-NLS-0$
					"GitSshKnownHost" : gitSshKnownHost, //$NON-NLS-0$
					"GitSshPrivateKey" : gitPrivateKey, //$NON-NLS-0$
					"GitSshPassphrase" : gitPassphrase //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doPull : function(gitCloneURI, force, gitSshUsername, gitSshPassword, gitSshKnownHost, gitPrivateKey, gitPassphrase) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitCloneURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Pull" : "true", //$NON-NLS-1$ //$NON-NLS-0$
					"Force" : force, //$NON-NLS-0$
					"GitSshUsername" : gitSshUsername, //$NON-NLS-0$
					"GitSshPassword" : gitSshPassword, //$NON-NLS-0$
					"GitSshKnownHost" : gitSshKnownHost, //$NON-NLS-0$
					"GitSshPrivateKey" : gitPrivateKey, //$NON-NLS-0$
					"GitSshPassphrase" : gitPassphrase //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doMerge : function(gitHeadURI, commitName, squash) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitHeadURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Merge" : commitName, //$NON-NLS-0$
					"Squash" : squash
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doCherryPick : function(gitHeadURI, commitName) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitHeadURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Cherry-Pick" : commitName //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doRevert : function(gitHeadURI, commitName) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitHeadURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Revert" : commitName //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doRebase : function(gitHeadURI, commitName, operation) {
			var service = this;
			var postData = {};
			postData.Rebase = commitName;
			if (operation) postData.Operation = operation;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitHeadURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify(postData)
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doPush : function(gitBranchURI, srcRef, tags, force, gitSshUsername, gitSshPassword, gitSshKnownHost, gitPrivateKey, gitPassphrase) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitBranchURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"PushSrcRef" : srcRef, //$NON-NLS-0$
					"PushTags" : tags, //$NON-NLS-0$
					"Force" : force, //$NON-NLS-0$
					"GitSshUsername" : gitSshUsername, //$NON-NLS-0$
					"GitSshPassword" : gitSshPassword, //$NON-NLS-0$
					"GitSshKnownHost" : gitSshKnownHost, //$NON-NLS-0$
					"GitSshPrivateKey" : gitPrivateKey, //$NON-NLS-0$
					"GitSshPassphrase" : gitPassphrase //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		getLog : function(gitCommitURI, commitName) {
			var service = this;
			var clientDeferred = new Deferred();
			var clientDeferred1 = new Deferred();
			
			xhr("POST", gitCommitURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"New" : commitName //$NON-NLS-0$
				})
			}).then(function(result) {
				var logLocation = null;
				//Bug 464972. We should use response instead of result.xhr.getResponseHeader. The result.xhr.getResponseHeader converts the non-english characters in the response, into "???"
				if(result.response && typeof result.response === "string") {
					var resonseObj = JSON.parse(result.response);
					if(resonseObj.Location) {
						logLocation = resonseObj.Location;	
					} 
				}
				if(!logLocation) {
					logLocation = result.xhr.getResponseHeader("Location");
				}
				clientDeferred1.resolve(logLocation);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			clientDeferred1.then(function(scopedGitCommitURI){
				xhr("GET", scopedGitCommitURI, { 
					headers : { 
						"Orion-Version" : "1",
						"Content-Type" : contentType
					},
					timeout : GIT_TIMEOUT,
					handleAs : "json" //$NON-NLS-0$
				}).then(function(result) {
					service._getGitServiceResponse(clientDeferred, result);
				}, function(error){
					service._handleGitServiceResponseError(clientDeferred, error);
				});
			});	
			
			return clientDeferred;			
		},
		
		getDefaultRemoteBranch : function(gitRemoteURI) {
			var service = this;
			var clientDeferred = new Deferred();
			var clientDeferred1 = new Deferred();
			
			xhr("GET", gitRemoteURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred1, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			clientDeferred1.then(function(remoteJsonData){
				xhr("GET", remoteJsonData.Children[0].Location, { 
					headers : { 
						"Orion-Version" : "1",
						"Content-Type" : contentType
					},
					timeout : GIT_TIMEOUT,
					handleAs : "json" //$NON-NLS-0$
				}).then(function(result) {
					service._getGitServiceResponse(clientDeferred, result);
				}, function(error){
					service._handleGitServiceResponseError(clientDeferred, error);
				});
			});	

			return clientDeferred;
		},
		
		doAddTag : function(gitCommitURI, tagName) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitCommitURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Name" : tagName  //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		doRemoveTag : function(gitTagURI) {
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("DELETE", gitTagURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		checkoutTag : function(gitCloneURI, tag, branchName){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitCloneURI, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Tag" : tag, //$NON-NLS-0$
					"Branch" : branchName //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		/**
		 * Performs git stash create
		 * @param gitStashLocation
		 * @param indexMessage [optional)
		 * @param workingDirectoryMessage [optional]
		 * @param includeUntracked [optional]
		 * @returns {Deferred}
		 */
		doStashCreate : function(gitStashLocation, indexMessage, workingDirectoryMessage, includeUntracked){
			var service = this;
			
			var payload = {};
			if(indexMessage != null) /* note that undefined == null */
				payload.IndexMessage = indexMessage;
			
			if(workingDirectoryMessage != null) /* note that undefined == null */
				payload.WorkingDirectoryMessage = workingDirectoryMessage;
			
			if(includeUntracked === true)
				payload.IncludeUntracked = true;
			
			var clientDeferred = new Deferred();
			xhr("POST", gitStashLocation, {
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify(payload)
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		/**
		 * Performs git stash pop
		 * @param gitStashLocation
		 * @returns {Deferred}
		 */
		doStashPop : function(gitStashLocation){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitStashLocation, {
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		/**
		 * Performs git stash apply on the given change
		 * @param gitStashApplyLocation /gitapi/stash/<changeRev>/(..)
		 * @returns {Deferred}
		 */
		doStashApply : function(gitStashApplyLocation){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", gitStashApplyLocation, {
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		/**
		 * Performs git stash list
		 * @param gitStashLocation
		 * @returns {Deferred}
		 */
		doStashList : function(gitStashLocation){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("GET", gitStashLocation, {
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		/**
		 * Performs git stash drop on the given change
		 * @param gitStashDropLocation /gitapi/stash/<changeRev>/(..)
		 * @returns {Deferred}
		 */
		doStashDrop : function(gitStashDropLocation){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("DELETE", gitStashDropLocation, {
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		addCloneConfigurationProperty: function(location, newKey, newValue){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Key" : newKey, //$NON-NLS-0$
					"Value" : newValue //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		editCloneConfigurationProperty: function(location, newValue){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("PUT", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"Value" : newValue //$NON-NLS-0$
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		deleteCloneConfigurationProperty: function(location){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("DELETE", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json" //$NON-NLS-0$
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},
		
		sendCommitReviewRequest: function(commit, location, login, url, authorName, message){
			var service = this;
			
			var clientDeferred = new Deferred();
			xhr("POST", location, { 
				headers : { 
					"Orion-Version" : "1",
					"Content-Type" : contentType
				},
				timeout : GIT_TIMEOUT,
				handleAs : "json", //$NON-NLS-0$
				data: JSON.stringify({
					"ReviewReqCommit": commit,
					"ReviewReqUrl" : url, //$NON-NLS-0$
					"ReviewReqNotifyLogin" : login, //$NON-NLS-0$	
					"ReviewReqAuthorName" : authorName,
					"ReviewMessage" : message
				})
			}).then(function(result) {
				service._getGitServiceResponse(clientDeferred, result);
			}, function(error){
				service._handleGitServiceResponseError(clientDeferred, error);
			});
			
			return clientDeferred;
		},

		_getGitServiceResponse : function(deferred, result) {
			var response =  result.response ? JSON.parse(result.response) : null;
			
			if (result.xhr && result.xhr.status === 202) {
				var def = operation.handle(response.Location);
				def.then(deferred.resolve, function(data) {
					data.failedOperation = response.Location;
					deferred.reject(data);
				}, deferred.progress);
				deferred.then(null, function(error){def.reject(error);});
				return;
			}
			deferred.resolve(response);
			return;
		},
		
		_handleGitServiceResponseError: function(deferred, error){
			deferred.reject(error);
		}
	};
	return GitService;
}());

return eclipse;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/ssh/sshTools',['orion/Deferred', 'orion/xhr'], function(Deferred, xhr){

	var eclipse = eclipse || {};
	eclipse.SshService = (function() {
		function SshService(serviceRegistry) {
			if (serviceRegistry) {
				this._serviceRegistry = serviceRegistry;
				this._authService = serviceRegistry.getService("orion.core.auth"); //$NON-NLS-0$
				this._serviceRegistration = serviceRegistry.registerService("orion.net.ssh", this); //$NON-NLS-0$
				
				/* plugins may require ssh service thus auth service may be absent at this point */
				if(!this._authService){
					
					/* Nolite iudicare et non iudicabimini. */
					var loginData;
					this._authService = {
						getUser : function(){
							if(loginData){
								var d = new Deferred();
								d.resolve(loginData);
								return d;
							}
							
							return xhr("POST", "../../login", {
								headers: {
									"Orion-Version": "1" //$NON-NLS-0$
								},
								timeout: 15000
							}).then(function(result) {
								loginData = result.response ? JSON.parse(result.response) : null;
								return loginData;
							}, function(error) {
								loginData = null;
								if (error instanceof Error && error.name === "Cancel") {
									return "_cancel";
								} 
								return error.response ? JSON.parse(error.response) : null;
							});
						}	
					};
				}
			}
		}
		
		/** @lends eclipse.SshService.prototype */
		SshService.prototype = {
			KNOWN_HOSTS: "knownHosts", //$NON-NLS-0$
			
			addKnownHost: function(options){
				options = options || {};
				var deferred = new Deferred();
				var self = this;
				
				var host = options.host,
					keyType = options.keyType,
					hostKey = options.hostKey,
					port = options.port || 22;
					
				if(!host || !keyType || !hostKey || !port){
					/* required parameters missing, fail */
					deferred.reject();
					return deferred;
				}
					
				this._authService.getUser().then(function(user){
					var currKnownHosts = localStorage.getItem(user.login + "/" + self.KNOWN_HOSTS); //$NON-NLS-0$
					currKnownHosts = currKnownHosts ? JSON.parse(currKnownHosts) : [];
					
					for(var i=0; i<currKnownHosts.length; ++i){
						var entry = currKnownHosts[i];
						if(entry.host === host && entry.port === port){
							/* already known host */
							deferred.resolve(entry.host + " " + entry.keyType + " " + entry.hostKey);
							return;
						}
					}
					
					/* new host entry */
					var entry = {
						host : host,
						port : port,
						keyType : keyType,
						hostKey : hostKey
					};
					
					/* flush to ls */
					currKnownHosts.push(entry);
					localStorage.setItem(user.login + "/" + self.KNOWN_HOSTS, JSON.stringify(currKnownHosts));
					deferred.resolve(entry.host + " " + entry.keyType + " " + entry.hostKey);
					
				}, function(error){
					deferred.reject(error);
				});
				
				return deferred;
			},
			
			clearKnownHosts : function(){
				var deferred = new Deferred();
				var self = this;
				
				this._authService.getUser().then(function(user){
					var currKnownHosts = localStorage.getItem(user.login + "/" + self.KNOWN_HOSTS); //$NON-NLS-0$
					currKnownHosts = currKnownHosts ? JSON.parse(currKnownHosts) : [];
					
					localStorage.removeItem(user.login + "/" + self.KNOWN_HOSTS); //$NON-NLS-0$
					deferred.resolve(currKnownHosts);
				}, function(error){
					deferred.reject(error);
				});
				
				return deferred;
			},
			
			getKnownHostCredentials : function(host, port){
				var deferred = new Deferred();
				var self = this;
				
				/* default ssh port */
				port = port || 22;
				
				if(!host || !port){
					/* required parameters missing, fail */
					deferred.reject();
					return deferred;
				}
				
				this._authService.getUser().then(function(user){
					var currKnownHosts = localStorage.getItem(user.login + "/" + self.KNOWN_HOSTS); //$NON-NLS-0$
					currKnownHosts = currKnownHosts ? JSON.parse(currKnownHosts) : [];
					
					for(var i=0; i<currKnownHosts.length; ++i){
						var entry = currKnownHosts[i];
						if(entry.host === host && entry.port === port){
							deferred.resolve(entry.host + " " + entry.keyType + " " + entry.hostKey);
							return;
						}
					}
					
					/* not found */
					deferred.resolve("");
				}, function(error){
					deferred.reject(error);
				});
				
				return deferred;
			}
		};
		
		return SshService;
	}());
	
	return eclipse;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global requirejs*/
define('orion/i18nUtil',['require', 'orion/Deferred'], function(require, Deferred) {
	/**
	 * Performs string substitution. Can be invoked in 2 ways:
	 *
	 * i) vargs giving numbered substition values:
	 *   formatMessage("${0} is ${1}", "foo", "bar")  // "foo is bar"
	 *
	 * ii) a map giving the substitutions:
	 *   formatMessage("${thing} is ${1}", {1: "bar", thing: "foo"})  // "foo is bar"
	 */
	function formatMessage(msg) {
		var pattern = /\$\{([^\}]+)\}/g, args = arguments;
		if (args.length === 2 && args[1] && typeof args[1] === "object") {
			return msg.replace(pattern, function(str, key) {
				return args[1][key];
			});
		}
		return msg.replace(pattern, function(str, index) {
			return args[(index << 0) + 1];
		});
	}
	return {
		formatMessage: formatMessage
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
// Encoding Shim -- see http://encoding.spec.whatwg.org/
(function(global) {
	function EncodingError() {
		Error.apply(this, arguments);
	}
	EncodingError.prototype = new Error();
	EncodingError.prototype.constructor = EncodingError;
	EncodingError.prototype.name = "EncodingError";

	function between(value, min, max) {
		return value >= min && value <= max;
	}

	function TextDecoder(label, options) {
		var encoding = label || "utf-8";
		if (encoding !== "utf-8" && encoding !== "utf8" && encoding !== "unicode-1-1-utf-8") {
			throw new TypeError("only utf-8 supported");
		}
		Object.defineProperties(this, {
			encoding: {
				value: encoding,
				enumerable: true
			},
			_fatal: {
				value: options && options.fatal
			},
			_saved: {
				value: [],
				writable: true
			},
			_checkBOM: {
				value: true,
				writable: true
			}
		});
	}
	TextDecoder.prototype.decode = function(input, options) {
		input = (input instanceof Uint8Array) ? input : new Uint8Array(input);
		var first, second, third, fourth, point;
		var stream = options && options.stream;
		var savedlen = this._saved.length;
		var inputlen = input.length;
		var offset = 0;
		var used = 0;
		var charCodes = new Uint16Array(inputlen + savedlen);
		var written = 0;

		function onError() {
			if (this._fatal) {
				this._saved.length = savedlen = 0;
				used = offset;
				this._checkBOM = this._checkBOM || !stream;
				throw new EncodingError();
			}
			charCodes[written++] = 0xFFFD;
		}
		
		if (this._checkBOM && inputlen) {
			if ((savedlen + inputlen) > 2) {
				for (var i = savedlen; i < 3; i++) {
					this._saved.push(input[offset++]);
				}
				if (this._saved[0] !== 0xEF || this._saved[1] !== 0xBB || this._saved[2] !== 0xBF) {
					offset = 0;
					this._saved.length = savedlen;
				} else {
					savedlen = this._saved.length -= 3;
				}
				this._checkBOM = false;
			} else if (stream) {
				while (offset < inputlen) {
					this._saved.push(input[offset++]);
				}
			}
		}
		while (offset < inputlen) {
			first = savedlen > 0 ? this._saved[0] : input[offset++];
			if (first < 0x80) {
				charCodes[written++] = first;
			} else if (between(first, 0xC2, 0xDF)) {
				if (offset === inputlen) break;
				second = savedlen > 1 ? this._saved[1] : input[offset++];
				if (!between(second, 0x80, 0xBF)) {
					onError();
					continue;
				}
				charCodes[written++] = ((first & 0x1F) << 6) | (second & 0x3F);
			} else if (between(first, 0xE0, 0xEF)) {
				if (offset === inputlen) break;
				second = savedlen > 1 ? this._saved[1] : input[offset++];
				if ((first === 0xE0 && !between(second, 0xA0, 0xBF)) || (first === 0xED && !between(second, 0x80, 0x9F)) || !between(second, 0x80, 0xBF)) {
					onError();
					continue;
				}
				if (offset === inputlen) break;
				third = savedlen > 2 ? this._saved[2] : input[offset++];
				if (!between(third, 0x80, 0xBF)) {
					onError();
					continue;
				}
				charCodes[written++] = ((first & 0x0F) << 12) | ((second & 0x3F) << 6) | (third & 0x3F);
			} else if (between(first, 0xF0, 0xF4)) {
				if (offset === inputlen) break;
				second = savedlen > 1 ? this._saved[1] : input[offset++];
				if ((first === 0xF0 && !between(second, 0x90, 0xBF)) || (first === 0xF4 && !between(second, 0x80, 0x8F)) || !between(second, 0x80, 0xBF)) {
					onError();
					continue;
				}
				if (offset === inputlen) break;
				third = savedlen > 2 ? this._saved[2] : input[offset++];
				if (!between(third, 0x80, 0xBF)) {
					onError();
					continue;
				}
				if (offset === inputlen) break;
				fourth = input[offset++];
				if (!between(fourth, 0x80, 0xBF)) {
					onError();
					continue;
				}
				point = (((first & 0x07) << 18) | ((second & 0x3F) << 12) | ((third & 0x3F) << 6) | (fourth & 0x3F)) & 0xFFFF;
				charCodes[written++] = (point >> 10) | 0xD800;
				charCodes[written++] = (point & 0x3FF) | 0xDC00;
			} else onError();
			used = offset;
			if (savedlen) {
				this._saved.length = savedlen = 0;
			}
		}
		while (used !== offset) {
			this._saved.push(input[used++]);
		}
		this._checkBOM = this._checkBOM || !stream;
		if (!stream && this._saved.length !== 0) {
			throw new EncodingError();
		}
		var result = [];
		for (var begin = 0; begin < written; begin += 0x10000) {
			result.push(String.fromCharCode.apply(null, charCodes.subarray(begin, Math.min(written, begin + 0x10000))));
		}
		return result.join("");
	};

	function TextEncoder(utfLabel) {
		var encoding = utfLabel || "utf-8";
		if (encoding !== "utf-8" && encoding !== "utf8" && encoding !== "unicode-1-1-utf-8") {
			throw new TypeError("only utf-8 supported");
		}
		Object.defineProperties(this, {
			encoding: {
				value: encoding,
				enumerable: true
			},
			_saved: {
				value: null,
				writable: true
			}
		});
	}
	TextEncoder.prototype.encode = function(input, options) {
		input = String(input !== undefined ? input : "");
		var first, second, point;
		var stream = options && options.stream;
		var inputlen = input.length;
		var offset = 0;
		var utf8 = new Uint8Array(3 * (inputlen + (this._saved === null ? 0 : 1)));
		var written = 0;

		while (offset < inputlen) {
			if (this._saved === null) {
				first = input.charCodeAt(offset++);
			} else {
				first = this._saved;
				this._saved = null;
			}
			if (first < 0x80) {
				utf8[written++] = first;
			} else if (first < 0x800) {
				utf8[written++] = 0xC0 | (first >> 6);
				utf8[written++] = 0x80 | (first & 0x3F);
			} else if (first < 0xD800 || first > 0xDBFF) {
				utf8[written++] = 0xE0 | (first >> 12);
				utf8[written++] = 0x80 | ((first >> 6) & 0x3F);
				utf8[written++] = 0x80 | (first & 0x3F);
			} else {
				if (offset < inputlen) {
					second = input.charCodeAt(offset++);
				} else if (stream) {
					this._saved = first;
					break;
				} else throw new EncodingError();
				if (!between(second, 0xDC00, 0xDFFF)) {
					throw new EncodingError();
				}
				point = 0x10000 | ((first & 0x03FF) << 10) | (second & 0x03FF);
				utf8[written++] = 0xF0 | (point >> 18);
				utf8[written++] = 0x80 | ((point >> 12) & 0x3F);
				utf8[written++] = 0x80 | ((point >> 6) & 0x3F);
				utf8[written++] = 0x80 | (point & 0x3F);
			}
		}
		if (!stream && this._saved !== null) {
			throw new EncodingError();
		}
		return utf8.buffer.slice ? new Uint8Array(utf8.buffer.slice(0, written)) : utf8.subarray(0, written);
	};

	global.TextDecoder = global.TextDecoder || TextDecoder;
	global.TextEncoder = global.TextEncoder || TextEncoder;
}((typeof global === "undefined") ? this || self : global));

define("orion/encoding-shim", function(){});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
/*global URL*/
define('orion/git/GitFileImpl',["orion/xhr", "orion/Deferred", "orion/encoding-shim", "orion/URL-shim"], function(xhr, Deferred) {
	
	function GitFileImpl(fileBase) {
		this.fileBase = fileBase;
	}
	
	var GIT_TIMEOUT = 60000;

	GitFileImpl.prototype = {
		fetchChildren: function(location) {
			var fetchLocation = location;
			if (fetchLocation===this.fileBase) {
				return new Deferred().resolve([]);
			}
			//If fetch location does not have ?depth=, then we need to add the depth parameter. Otherwise server will not return any children
			if (fetchLocation.indexOf("?depth=") === -1) { //$NON-NLS-0$
				fetchLocation += "?depth=1"; //$NON-NLS-0$
			}
			return xhr("GET", fetchLocation,{ //$NON-NLS-0$
				headers: {
					"Orion-Version": "1", //$NON-NLS-0$  //$NON-NLS-1$
					"Content-Type": "charset=UTF-8" //$NON-NLS-0$  //$NON-NLS-1$
				},
				timeout: GIT_TIMEOUT
			}).then(function(result) {
				var jsonData = result.response ? JSON.parse(result.response) : {};
				return jsonData.Children || [];
			});
		},
		loadWorkspaces: function() {
			return this.loadWorkspace(this._repoURL);
		},
		loadWorkspace: function(location) {
			if (location === "/gitapi/") {
				location += "tree/";
			}
			
			return xhr("GET", location,{ //$NON-NLS-0$
				headers: {
					"Orion-Version": "1", //$NON-NLS-0$  //$NON-NLS-1$
					"Content-Type": "charset=UTF-8" //$NON-NLS-0$  //$NON-NLS-1$
				},
				timeout: GIT_TIMEOUT
			}).then(function(result) {
				var jsonData = result.response ? JSON.parse(result.response) : {};
				return jsonData || {};
			});
		},
		createProject: function(url, projectName, serverPath, create) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		createFolder: function(parentLocation, folderName) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		createFile: function(parentLocation, fileName) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		deleteFile: function(location) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		moveFile: function(sourceLocation, targetLocation, name) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		copyFile: function(sourceLocation, targetLocation, name) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		read: function(location, isMetadata) {
			var url = new URL(location, window.location);
			if (isMetadata) {
				url.query.set("parts", "meta"); //$NON-NLS-0$  //$NON-NLS-1$
			}
			return xhr("GET", url.href, {
				timeout: GIT_TIMEOUT,
				headers: { "Orion-Version": "1" }, //$NON-NLS-0$  //$NON-NLS-1$
				log: false
			}).then(function(result) {
				if (isMetadata) {
					return result.response ? JSON.parse(result.response) : null;
				} else {
					return result.response;
				}
			});
		},
		write: function(location, contents, args) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		remoteImport: function(targetLocation, options) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		remoteExport: function(sourceLocation, options) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		},
		readBlob: function(location) {
			return xhr("GET", location, { //$NON-NLS-0$ 
				responseType: "arraybuffer", //$NON-NLS-0$ 
				timeout: GIT_TIMEOUT
			}).then(function(result) {
				return result.response;
			});
		},
		writeBlob: function(location, contents, args) {
			throw new Error("Not supported"); //$NON-NLS-0$ 
		}
	};
	GitFileImpl.prototype.constructor = GitFileImpl;

	return GitFileImpl;
});

/**
 * @license RequireJS i18n 2.0.4 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like "i18n!nls/colors".
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is "en-us",
 * then the plugin will ask for the "en-us", "en" and "root" bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec("foo/bar/baz/nls/en-ca/foo") gives:
    //["foo/bar/baz/nls/en-ca/foo", "foo/bar/baz/nls/", "/", "/", "en-ca", "foo"]
    //nlsRegExp.exec("foo/bar/baz/nls/foo") gives:
    //["foo/bar/baz/nls/foo", "foo/bar/baz/nls/", "/", "/", "foo", ""]
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};

        return {
            version: '2.0.4',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split("-"),
                    toLoad = [],
                    value = {},
                    i, part, current = "";

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === "undefined" ? "root" :
                            (navigator.language ||
                             navigator.userLanguage || "root").toLowerCase();
                    }
                    parts = locale.split("-");
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, "root", toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? "-" : "") + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }
                                        
                    if(config.locales) {
                    	var j, k; 
                    	for (j = 0; j < config.locales.length; j++) {
                    		locale = config.locales[j];
                    		parts = locale.split("-");
                    		current = "";
	                    	for (k = 0; k < parts.length; k++) {
		                        part = parts[k];
		                        current += (current ? "-" : "") + part;
		                        addIfExists(req, current, toLoad, prefix, suffix);
	                    	}
                    	}
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart("root", master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? "-" : "") + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/*eslint-env browser, amd*/
define('git/nls/gitmessages',{
	root:true
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/*eslint-env browser, amd*/
define('git/nls/root/gitmessages',{//Default message bundle
	"Compare": "Compare", //$NON-NLS-0$  //$NON-NLS-1$
	"View the side-by-side compare": "View the side-by-side compare", //$NON-NLS-0$  //$NON-NLS-1$
	"WorkingDirVer": "Open Working Directory", //$NON-NLS-0$  //$NON-NLS-1$
	"Working Directory": "Working Directory", //$NON-NLS-0$  //$NON-NLS-1$
	"ViewWorkingDirVer": "View the working directory version of the file", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading...": "Loading...", //$NON-NLS-0$  //$NON-NLS-1$
	"Repositories": "Git", //$NON-NLS-0$  //$NON-NLS-1$
	"Repo": "Repositories", //$NON-NLS-0$  //$NON-NLS-1$
	"0 on 1 - Git": "${0} on ${1} - Git", //$NON-NLS-0$  //$NON-NLS-1$
	"Git": "Git", //$NON-NLS-0$  //$NON-NLS-1$
	"Show in eclipse.org": "Show in eclipse.org", //$NON-NLS-0$  //$NON-NLS-1$
	"Show in GitHub": "Show in GitHub", //$NON-NLS-0$  //$NON-NLS-1$
	"Show this repository in GitHub": "Show this repository in GitHub", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit Details": "Commit Details", //$NON-NLS-0$  //$NON-NLS-1$
	"No Commits": "No Commits", //$NON-NLS-0$  //$NON-NLS-1$
	"commit: 0": "commit: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"parent: 0": "parent: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"authored by 0 (1) on 2": "authored by ${0} <${1}> on ${2}", //$NON-NLS-0$  //$NON-NLS-1$
	"committed by 0 (1)": "committed by ${0} <${1}>", //$NON-NLS-0$  //$NON-NLS-1$
	"committedby": "committed by ", //$NON-NLS-0$  //$NON-NLS-1$
	"on": " on ", //$NON-NLS-0$  //$NON-NLS-1$ //Used for the format as "authorName on date"
	"nameEmail": "${0} <${1}>", //$NON-NLS-0$  //$NON-NLS-1$
	"Tags:": "Tags:", //$NON-NLS-0$  //$NON-NLS-1$
	"No Tags": "No Tags", //$NON-NLS-0$  //$NON-NLS-1$
	"Diffs": "Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"WorkingDirChanges": "Working Directory Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitChanges": "Commit Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitChangesDialog": "Commit Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"more": "more ...", //$NON-NLS-0$  //$NON-NLS-1$
	"less": "less ...", //$NON-NLS-0$  //$NON-NLS-1$
	"More": "More", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreFiles" : "More Files", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreFilesProgress": "Loading more files...", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreCommits": "More commits for \"${0}\"", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreCommitsProgress": "Loading more commits for \"${0}\"...", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreBranches": "More branches for \"${0}\"", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreBranchesProgress": "Loading more branches for \"${0}\"...", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreTags": "More tags", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreTagsProgress": "Loading more tags...", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreStashes": "More stashes", //$NON-NLS-0$  //$NON-NLS-1$
	"MoreStashesProgress": "Loading more stashes...", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading git log...": "Loading git log...", //$NON-NLS-0$  //$NON-NLS-1$
	"local": "local", //$NON-NLS-0$  //$NON-NLS-1$
	"remote": "remote", //$NON-NLS-0$  //$NON-NLS-1$
	"View All": "View All", //$NON-NLS-0$  //$NON-NLS-1$
	"Error ${0}: ": "Error ${0}: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading ": "Loading ", //$NON-NLS-0$  //$NON-NLS-1$
	"Message": "Message", //$NON-NLS-0$  //$NON-NLS-1$
	"Author": "Author", //$NON-NLS-0$  //$NON-NLS-1$
	"Date": "Date", //$NON-NLS-0$  //$NON-NLS-1$
	"fromDate:": "Start Date:", //$NON-NLS-0$  //$NON-NLS-1$
	"toDate:": "End Date:", //$NON-NLS-0$  //$NON-NLS-1$
	"Actions": "Actions", //$NON-NLS-0$  //$NON-NLS-1$
	"Branches": "Branches", //$NON-NLS-0$  //$NON-NLS-1$
	"Tags": "Tags", //$NON-NLS-0$  //$NON-NLS-1$
	"Stage": "Stage", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstaged removal": "Unstaged removal", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstage": "Unstage", //$NON-NLS-0$  //$NON-NLS-1$
	"Staged removal": "Staged removal", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstaged change": "Unstaged change", //$NON-NLS-0$  //$NON-NLS-1$
	"Staged change": "Staged change", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstaged add": "Unstaged add", //$NON-NLS-0$  //$NON-NLS-1$
	"Staged add": "Staged add", //$NON-NLS-0$  //$NON-NLS-1$
	"Addition": "Addition", //$NON-NLS-0$  //$NON-NLS-1$
	"Deletion": "Deletion", //$NON-NLS-0$  //$NON-NLS-1$
	"Resolve Conflict": "Resolve Conflict", //$NON-NLS-0$  //$NON-NLS-1$
	"Conflicting": "Conflicting", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit message": "Commit message", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit": "Commit", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitTooltip": "Commit the selected files with the given message.", //$NON-NLS-0$  //$NON-NLS-1$
	"AuthMsgLink":"Authentication required for: ${0}. <a target=\"_blank\" href=\"${1}\">${2}</a> and re-try the request. </span>", //$NON-NLS-0$  //$NON-NLS-1$
	"SmartCommit": "Enter the commit message", //$NON-NLS-0$  //$NON-NLS-1$
	"SmartCountCommit": "Commit ${0} file(s)", //$NON-NLS-0$  //$NON-NLS-1$
	"Amend last commit": "Amend last commit", //$NON-NLS-0$  //$NON-NLS-1$
	" Amend": " Amend", //$NON-NLS-0$  //$NON-NLS-1$
	"Rebase in progress. Choose action:": "Rebase in progress. Choose action:", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseProgress": "Rebase in progress", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseTip": "Rebase your commits by removing them from the active branch, starting the active branch again based on the latest state of \"${0}\" and applying each commit again to the updated active branch.", //$NON-NLS-0$  //$NON-NLS-1$
	"RebasingRepo": "Rebasing git repository", //$NON-NLS-0$  //$NON-NLS-1$
	"AddingConfig": "Adding git configuration property: key=${0} value=${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"EditingConfig": "Editing git configuration property: key=${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"DeletingConfig": "Deleting git configuration property: key=${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"AddClone": "Cloning repository: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseProgressDetails": "Rebasing branch.\n\n\tUse Continue after merging the conflicts and selecting all files;\n\tSkip to bypass the current patch;\n\tAbort to end the rebase at any time.", //$NON-NLS-0$  //$NON-NLS-1$
	"Committer name:": "Committer name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Name:": "Name:", //$NON-NLS-0$  //$NON-NLS-1$
	"email:": "email:", //$NON-NLS-0$  //$NON-NLS-1$
	"Email:": "Email:", //$NON-NLS-0$  //$NON-NLS-1$
	"Author name: ": "Author name: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstaged": "Unstaged", //$NON-NLS-0$  //$NON-NLS-1$
	"Staged": "Staged", //$NON-NLS-0$  //$NON-NLS-1$
	"ChangedFiles": "Changed Files", //$NON-NLS-0$  //$NON-NLS-1$
	"Recent commits on": "Recent commits on", //$NON-NLS-0$  //$NON-NLS-1$
	"Git Status": "Git Status", //$NON-NLS-0$  //$NON-NLS-1$
	"Go to Git Status": "Open the Git Status page for the repository containing this file or folder.", //$NON-NLS-0$  //$NON-NLS-1$
	"GetGitIncomingMsg": "Getting git incoming changes...", //$NON-NLS-0$  //$NON-NLS-1$
	"Checkout": "Checkout", //$NON-NLS-0$  //$NON-NLS-1$
	"Checking out...": "Checking out...", //$NON-NLS-0$  //$NON-NLS-1$
	"Stage the change": "Stage the change", //$NON-NLS-0$  //$NON-NLS-1$
	"Staging...": "Staging...", //$NON-NLS-0$  //$NON-NLS-1$
	"CheckoutSelectedFiles": "Checkout all the selected files, discarding all changes", //$NON-NLS-0$  //$NON-NLS-1$
	"AddFilesToGitignore" : "Add all the selected files to .gitignore file(s)", //$NON-NLS-0$  //$NON-NLS-1$
	"Writing .gitignore rules" : "Writing .gitignore rules", //$NON-NLS-0$  //$NON-NLS-1$ 
	"Save Patch": "Save Patch", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstage the change": "Unstage the change", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstaging...": "Unstaging...", //$NON-NLS-0$  //$NON-NLS-1$
	"Undo": "Undo", //$NON-NLS-0$  //$NON-NLS-1$
	"UndoTooltip": "Revert this commit, keeping all changed files and not making any changes to the working directory.", //$NON-NLS-0$  //$NON-NLS-1$
	"UndoConfirm": "The content of your active branch will be replaced with commit \"${0}\". All changes in the commit and working directory will be kept. Are you sure?", //$NON-NLS-0$  //$NON-NLS-1$
	"Reset": "Reset", //$NON-NLS-0$  //$NON-NLS-1$
	"ResetConfirm": "All unstaged and staged changes in the working directory and index will be discarded and cannot be recovered.\n\nAre you sure you want to continue?", //$NON-NLS-0$  //$NON-NLS-1$
	"CheckoutConfirm" : "Your changes to the selected files will be discarded and cannot be recovered.\n\nAre you sure you want to continue?", //$NON-NLS-0$  //$NON-NLS-1$
	"ResetBranchDiscardChanges": "Reset the branch, discarding all staged and unstaged changes", //$NON-NLS-0$  //$NON-NLS-1$
	"ChangesIndexDiscardedMsg": "All unstaged and staged changes in the working directory and index will be discarded and cannot be recovered.", //$NON-NLS-0$  //$NON-NLS-1$
	"ContinueMsg": "Are you sure you want to continue?", //$NON-NLS-0$  //$NON-NLS-1$
	"KeepWorkDir" : "Keep Working Directory", //$NON-NLS-0$  //$NON-NLS-1$
	"Resetting local changes...": "Resetting local changes...", //$NON-NLS-0$  //$NON-NLS-1$
	"Continue rebase...": "Continue rebase...", //$NON-NLS-0$  //$NON-NLS-1$
	"Skipping patch...": "Skipping patch...", //$NON-NLS-0$  //$NON-NLS-1$
	"Aborting rebase...": "Aborting rebase...", //$NON-NLS-0$  //$NON-NLS-1$
	"Complete log": "Complete log", //$NON-NLS-0$  //$NON-NLS-1$
	"local VS index": "local VS index", //$NON-NLS-0$  //$NON-NLS-1$
	"index VS HEAD": "index VS HEAD", //$NON-NLS-0$  //$NON-NLS-1$
	"Compare(${0} : ${1})": "Compare(${0} : ${1})", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading status...": "Loading status...", //$NON-NLS-0$  //$NON-NLS-1$
	"Committing...": "Committing...", //$NON-NLS-0$  //$NON-NLS-1$
	"The author name is required.": "The author name is required.", //$NON-NLS-0$  //$NON-NLS-1$
	"The author mail is required.": "The author mail is required.", //$NON-NLS-0$  //$NON-NLS-1$
	"RepoConflict": ". Repository still contains conflicts.", //$NON-NLS-0$  //$NON-NLS-1$
	"RepoUnmergedPathResolveConflict": ". Repository contains unmerged paths. Resolve conflicts first.", //$NON-NLS-0$  //$NON-NLS-1$
	"Rendering ${0}": "Rendering ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Configuration": "Configuration", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting configuration of": "Getting configuration of ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting git repository details": "Getting git repository details", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting changes": "Getting changes", //$NON-NLS-0$  //$NON-NLS-1$
	" - Git": " - Git", //$NON-NLS-0$  //$NON-NLS-1$
	"Repositories - Git": "Repositories - Git", //$NON-NLS-0$  //$NON-NLS-1$
	"Repository": "Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Repository Not Found": "Repository Not Found", //$NON-NLS-0$  //$NON-NLS-1$
	"No Repositories": "No Repositories", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading repository": "Loading repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading repositories": "Loading repositories", //$NON-NLS-0$  //$NON-NLS-1$
	"(no remote)": "(no remote)", //$NON-NLS-0$  //$NON-NLS-1$
	"location: ": "location: ", //$NON-NLS-0$  //$NON-NLS-1$
	"NumFilesStageAndCommit": "${0} file(s) to stage and ${1} file(s) to commit.", //$NON-NLS-0$  //$NON-NLS-1$
 	"Nothing to commit.": "Nothing to commit.", //$NON-NLS-0$  //$NON-NLS-1$
	"Nothing to push.": "Nothing to push.", //$NON-NLS-0$  //$NON-NLS-1$
	"NCommitsToPush": "${0} commit(s) to push.", //$NON-NLS-0$  //$NON-NLS-1$
	"You have no changes to commit.": "You have no changes to commit.", //$NON-NLS-0$  //$NON-NLS-1$
	"Rebase in progress!": "Rebase in progress!", //$NON-NLS-0$  //$NON-NLS-1$
	"View all local and remote tracking branches": "View all local and remote tracking branches", //$NON-NLS-0$  //$NON-NLS-1$
	"tracksNoBranch": "tracks no branch", //$NON-NLS-0$  //$NON-NLS-1$
	"tracks": "tracks ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"last modified ${0} by ${1}": "last modified ${0} by ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"No Remote Branches": "No Remote Branches", //$NON-NLS-0$  //$NON-NLS-1$
	"Rendering branches": "Rendering branches", //$NON-NLS-0$  //$NON-NLS-1$
	"Commits": "Commits", //$NON-NLS-0$  //$NON-NLS-1$
	"GettingCurrentBranch": "Getting current branch for ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"See Full Log": "See Full Log", //$NON-NLS-0$  //$NON-NLS-1$
	"See the full log": "See the full log", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting commits for \"${0}\" branch": "Getting commits for \"${0}\" branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Rendering commits": "Rendering commits", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting outgoing commits": "Getting outgoing commits", //$NON-NLS-0$  //$NON-NLS-1$
	"The branch is up to date.": "The branch is up to date.", //$NON-NLS-0$  //$NON-NLS-1$
	"NoOutgoingIncomingCommits": "You have no outgoing or incoming commits.", //$NON-NLS-0$  //$NON-NLS-1$
 	") by ": ") by ", //$NON-NLS-0$  //$NON-NLS-1$
	" (SHA ": " (SHA ", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting tags": "Getting tags", //$NON-NLS-0$  //$NON-NLS-1$
	"View all tags": "View all tags", //$NON-NLS-0$  //$NON-NLS-1$
	" on ": " on ", //$NON-NLS-0$  //$NON-NLS-1$
	" by ": " by ", //$NON-NLS-0$  //$NON-NLS-1$
	"Remotes": "Remotes", //$NON-NLS-0$  //$NON-NLS-1$
	"Rendering remotes": "Rendering remotes", //$NON-NLS-0$  //$NON-NLS-1$
	"No Remotes": "No Remotes", //$NON-NLS-0$  //$NON-NLS-1$
	"Unstaged addition": "Unstaged addition", //$NON-NLS-0$  //$NON-NLS-1$
	"Staged addition": "Staged addition", //$NON-NLS-0$  //$NON-NLS-1$
	" (Rebase in Progress)": " (Rebase in Progress)", //$NON-NLS-0$  //$NON-NLS-1$
	"Status": "Status", //$NON-NLS-0$  //$NON-NLS-1$
	"Log (0)": "Log (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"Log (0) - 1": "Log (${0}) - ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"Status for ${0} - Git ": "Status for ${0} - Git ", //$NON-NLS-0$  //$NON-NLS-1$
	"No Unstaged Changes": "No Unstaged Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"No Staged Changes": "No Staged Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"Changes for \"${0}\" branch": "Changes for ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Commits for \"${0}\" branch": "Commits for ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Commits for \"${0}\" branch against": "Commits for \"${0}\" branch against", //$NON-NLS-0$  //$NON-NLS-1$
	"Add Remote": "Add Remote", //$NON-NLS-0$  //$NON-NLS-1$
	"Remote Name:": "Remote Name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Remote URI:": "Remote URI:", //$NON-NLS-0$  //$NON-NLS-1$
	"Apply Patch": "Apply Patch", //$NON-NLS-0$  //$NON-NLS-1$
	"ApplyPatchDialog": "Apply Patch", //$NON-NLS-0$  //$NON-NLS-1$
	"Git Repository": "Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Go to the git repository": "Open the Git Repository page for this file or folder.", //$NON-NLS-0$  //$NON-NLS-1$
	"Clone Git Repository": "Clone Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"CloneGitRepositoryDialog": "Clone Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Repository URL:": "Repository URL:", //$NON-NLS-0$  //$NON-NLS-1$
	"Existing directory:": "Existing folder:", //$NON-NLS-0$  //$NON-NLS-1$
	"New folder:": "New folder:", //$NON-NLS-0$  //$NON-NLS-1$
	"ChooseFolderDialog": "Choose a Folder", //$NON-NLS-0$  //$NON-NLS-1$
	"Message:": "Message:", //$NON-NLS-0$  //$NON-NLS-1$
	"Amend:": "Amend:", //$NON-NLS-0$  //$NON-NLS-1$
	"SmartAmend": "Amend previous commit", //$NON-NLS-0$  //$NON-NLS-1$
	"SmartSignedOffById": "Add Signed-off-by to message", //$NON-NLS-0$  //$NON-NLS-1$
	"ChangeId:": "ChangeId:", //$NON-NLS-0$  //$NON-NLS-1$
	"SmartChangeId": "Add Change-ID to message", //$NON-NLS-0$  //$NON-NLS-1$
	"Committer Name:": "Committer Name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Committer Email:": "Committer Email:", //$NON-NLS-0$  //$NON-NLS-1$
	"AuthorNamePlaceholder": "Enter author name", //$NON-NLS-0$  //$NON-NLS-1$
	"AuthorEmailPlaceholder": "Enter author email", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitterNamePlaceholder": "Enter committer name", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitterEmailPlaceholder": "Enter committer email", //$NON-NLS-0$  //$NON-NLS-1$
	"Author Name:": "Author Name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Author Email:": "Author Email:", //$NON-NLS-0$  //$NON-NLS-1$
	"The commit message is required.": "The commit message is required.", //$NON-NLS-0$  //$NON-NLS-1$
	"Git Credentials": "Git Credentials", //$NON-NLS-0$  //$NON-NLS-1$
	"Username:": "Username:", //$NON-NLS-0$  //$NON-NLS-1$
	"Private key:": "Private key:", //$NON-NLS-0$  //$NON-NLS-1$
	"Passphrase (optional):": "Passphrase (optional):", //$NON-NLS-0$  //$NON-NLS-1$
	"commit:": "commit: ", //$NON-NLS-0$  //$NON-NLS-1$
	"parent:": "parent: ", //$NON-NLS-0$  //$NON-NLS-1$
	"branches: ": "branches: ", //$NON-NLS-0$  //$NON-NLS-1$
	"tags: ": "tags: ", //$NON-NLS-0$  //$NON-NLS-1$
	"tags": "tags", //$NON-NLS-0$  //$NON-NLS-1$
	" authored by ${0} {${1}) on ${2}": " authored by ${0} (${1}) on ${2}", //$NON-NLS-0$  //$NON-NLS-1$
	"Content": "Content", //$NON-NLS-0$  //$NON-NLS-1$
	"Go to ${0} section": "Go to ${0} section", //$NON-NLS-0$  //$NON-NLS-1$
	"Type the commit name (sha1):": "Type the commit name (sha1):", //$NON-NLS-0$  //$NON-NLS-1$
	"Search": "Search", //$NON-NLS-0$  //$NON-NLS-1$
	"Searching...": "Searching...", //$NON-NLS-0$  //$NON-NLS-1$
	"SelectAll": "Select All", //$NON-NLS-0$  //$NON-NLS-1$
	"Looking for the commit": "Looking for the commit", //$NON-NLS-0$  //$NON-NLS-1$
	"New Branch:": "New Branch:", //$NON-NLS-0$  //$NON-NLS-1$
	"No remote selected": "No remote selected", //$NON-NLS-0$  //$NON-NLS-1$
	"Enter a name...": "Enter a name...", //$NON-NLS-0$  //$NON-NLS-1$
	"OK": "OK", //$NON-NLS-0$  //$NON-NLS-1$
	"Cancel": "Cancel", //$NON-NLS-0$  //$NON-NLS-1$
	"Clear": "Clear", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter": "Filter", //$NON-NLS-0$  //$NON-NLS-1$
	"FilterCommits": "Filter Commits", //$NON-NLS-0$  //$NON-NLS-1$
	"FilterCommitsTip": "Toggles the filter commits panel", //$NON-NLS-0$  //$NON-NLS-1$
	"MaximizeCmd": "Maximize", //$NON-NLS-0$  //$NON-NLS-1$
	"MaximizeTip": "Toggles the maximize state of the editor", //$NON-NLS-0$  //$NON-NLS-1$
	" [New branch]": " [New branch]", //$NON-NLS-0$  //$NON-NLS-1$
	"AddKeyToHostContinueOp": "Would you like to add ${0} key for host ${1} to continue operation? Key fingerpt is ${2}.", //$NON-NLS-0$  //$NON-NLS-1$
 	"Link Repository": "Link Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Folder name:": "Folder name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Repository was linked to ": "Repository was linked to ", //$NON-NLS-0$  //$NON-NLS-1$
	"CheckoutCommitTooltip": "Checkout this commit, creating a local branch based on its contents.", //$NON-NLS-0$  //$NON-NLS-1$
	"CheckoutTagTooltip": "Checkout this tag, creating a local branch based on its contents.", //$NON-NLS-0$  //$NON-NLS-1$
	"Checking out ${0}": "Checking out ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"CheckoutBranchMsg": "Checkout the branch or corresponding local branch and make it active. If the remote tracking branch does not have a corresponding local branch, the local branch will be created first.", //$NON-NLS-0$  //$NON-NLS-1$
 	"Checking out branch...": "Checking out branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Adding branch ${0}...": "Adding branch ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Removing branch ${0}...": "Removing branch ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Adding remote ${0}...": "Adding remote ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Removing remote ${0}...": "Removing remote ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Removing repository ${0}": "Removing repository ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Adding tag {$0}": "Adding tag {$0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Removing tag {$0}": "Removing tag {$0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Merging ${0}": "Merging ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Merging${0}Succeeded" : "Merging ${0} succeeded", //$NON-NLS-0$  //$NON-NLS-1$
	'Unstaging changes' : 'Unstaging changes', //$NON-NLS-0$  //$NON-NLS-1$
	"Checking out branch ${0}...": "Checking out branch ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Branch checked out.": "Branch checked out.", //$NON-NLS-0$  //$NON-NLS-1$
	"New Branch": "New Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Add a new local branch to the repository": "Add a new local branch to the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Branch name": "Branch name", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete": "Delete", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete the local branch from the repository": "Delete the local branch from the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"DelBrConfirm": "Are you sure you want to delete branch ${0}?", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete the remote tracking branch from the repository": "Delete the remote tracking branch from the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Are you sure?": "Are you sure?", //$NON-NLS-0$  //$NON-NLS-1$
	"RemoveRemoteBranchConfirm": "You're going to delete remote branch \"${0}\" and push the change.\n\nAre you sure?", //$NON-NLS-0$  //$NON-NLS-1$
	"Removing remote branch: ": "Removing remote branch: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete Remote Branch": "Delete Remote Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"New Remote": "New Remote", //$NON-NLS-0$  //$NON-NLS-1$
	"Git Remote": "Git Remote", //$NON-NLS-0$  //$NON-NLS-1$
	"Go to Git Remote": "Open the remote Git Log page for this file or folder.", //$NON-NLS-0$  //$NON-NLS-1$
	"Add a new remote to the repository": "Add a new remote to the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete the remote from the repository": "Delete the remote from the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Are you sure you want to delete remote ${0}?": "Are you sure you want to delete remote ${0}?", //$NON-NLS-0$  //$NON-NLS-1$
	"Pull": "Pull", //$NON-NLS-0$  //$NON-NLS-1$
	"Pull from the repository": "Pull from the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Pulling: ": "Pulling: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Pull Git Repository": "Pull Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Git Log": "Git Log", //$NON-NLS-0$  //$NON-NLS-1$
	"Go to Git Log": "Open the local Git Log page for this file or folder.", //$NON-NLS-0$  //$NON-NLS-1$
	"Open the log for the branch": "Open the log for the branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Open the log for the repository": "Open the log for the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Open the status for the repository": "Open the status for the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowInEditor": "Show in Editor", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowInEditorTooltip": "Show the repository folder in the editor", //$NON-NLS-0$  //$NON-NLS-1$
	"CompareEach": "Compare With Each Other", //$NON-NLS-0$  //$NON-NLS-1$
 	"Compare With Working Tree": "Compare With Working Tree", //$NON-NLS-0$  //$NON-NLS-1$
	"Open": "Open", //$NON-NLS-0$  //$NON-NLS-1$
	"OpenGitCommitTip": "View the tree for this commit", //$NON-NLS-0$  //$NON-NLS-1$
	"OpenCommitVersion": "Open Commit", //$NON-NLS-0$  //$NON-NLS-1$
	"ViewCommitVersionTip": "View the committed version of the file", //$NON-NLS-0$  //$NON-NLS-1$
	"Fetch": "Fetch", //$NON-NLS-0$  //$NON-NLS-1$
	"Fetch from the remote": "Fetch from the remote", //$NON-NLS-0$  //$NON-NLS-1$
	"Password:": "Password:", //$NON-NLS-0$  //$NON-NLS-1$
	"User Name:": "User Name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Fetching remote: ": "Fetching remote: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Force Fetch": "Force Fetch", //$NON-NLS-0$  //$NON-NLS-1$
	"FetchRemoteBranch": "Fetch from the remote branch into your remote tracking branch overriding its current content", //$NON-NLS-0$  //$NON-NLS-1$
	"OverrideContentRemoteTrackingBr": "You're going to override content of the remote tracking branch. This can cause the branch to lose commits.", //$NON-NLS-0$  //$NON-NLS-1$
	"Merge": "Merge", //$NON-NLS-0$  //$NON-NLS-1$
	"MergeContentFrmBr": "Merge the content from the branch to your active branch", //$NON-NLS-0$  //$NON-NLS-1$
 	". Go to ${0}.": ". Go to ${0}.", //$NON-NLS-0$  //$NON-NLS-1$
	"Git Status page": "Git Status page", //$NON-NLS-0$  //$NON-NLS-1$
	"Rebase": "Rebase", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseOK": "Rebase succeeded", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseAborted": "Rebase aborted", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseSucceeded": "${0} succeeded", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseCommitsMsg": "Rebase your commits by removing them from the active branch, starting the active branch again based on the latest state of the selected branch ", //$NON-NLS-0$  //$NON-NLS-1$
 	"Rebase on top of ": "Rebase on top of ", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseSTOPPED": "Some conflicts occurred. Please resolve them and continue, skip patch or abort rebasing.", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseFAILED_WRONG_REPOSITORY_STATE": "Repository state is invalid (i.e. already during rebasing).", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseFAILED_UNMERGED_PATHS": "Repository contains unmerged paths.", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseFAILED_PENDING_CHANGES": "Repository contains pending changes. Please commit or stash them before performing the rebase again.", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseUNCOMMITTED_CHANGES": "There are uncommitted changes. Please commit or stash them before performing the rebase again.", //$NON-NLS-0$  //$NON-NLS-1$
	"RebaseCommitsByRmvingThem": "Rebase your commits by removing them from the active branch, ", //$NON-NLS-0$  //$NON-NLS-1$
	"StartActiveBranch": "starting the active branch again based on the latest state of '", //$NON-NLS-0$  //$NON-NLS-1$
	"ApplyEachCommitAgain": "and applying each commit again to the updated active branch.", //$NON-NLS-0$  //$NON-NLS-1$
	"Push All": "Push All", //$NON-NLS-0$  //$NON-NLS-1$
	"PushCommitsTagsFrmLocal": "Push commits and tags from your local branch into the remote branch", //$NON-NLS-0$  //$NON-NLS-1$
 	"Push Branch": "Push Branch", //$NON-NLS-0$  //$NON-NLS-1$
 	"PushResult": "Push Result:", //$NON-NLS-0$  //$NON-NLS-1$
 	"PushCommitsWithoutTags": "Push commits without tags from your local branch into the remote branch", //$NON-NLS-0$  //$NON-NLS-1$
 	"Push for Review": "Push for Review", //$NON-NLS-0$  //$NON-NLS-1$
	"Push commits to Gerrit Code Review": "Push commits to Gerrit Code Review", //$NON-NLS-0$  //$NON-NLS-1$
	"Force Push Branch": "Force Push Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"PushCommitsWithoutTagsOverridingCurrentContent": "Push commits without tags from your local branch into the remote branch overriding its current content", //$NON-NLS-0$  //$NON-NLS-1$
 	"Pushing remote: ": "Pushing remote: ", //$NON-NLS-0$  //$NON-NLS-1$
 	"PushingRemoteSucceeded": "Pushing to remote: ${0} succeeded", //$NON-NLS-0$  //$NON-NLS-1$
 	"PushingGerritChangeSucceeded": "Pushing to gerrit ${0} succeeded: ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"ChooseBranchDialog": "Choose Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Choose the remote branch.": "Choose the remote branch.", //$NON-NLS-0$  //$NON-NLS-1$
	"Force Push All": "Force Push All", //$NON-NLS-0$  //$NON-NLS-1$
	"PushCommitsTagsFrmLocalBr": "Push commits and tags from your local branch into the remote branch overriding its current content", //$NON-NLS-0$  //$NON-NLS-1$
	"OverrideContentOfRemoteBr": "You're going to override content of the remote branch. This can cause the remote repository to lose commits.", //$NON-NLS-0$  //$NON-NLS-1$
	"< Previous Page": "< Previous Page", //$NON-NLS-0$  //$NON-NLS-1$
	"Show previous page of git log": "Show previous page of git log", //$NON-NLS-0$  //$NON-NLS-1$
	"Show previous page of git tags" : "Show previous page of git tags", //$NON-NLS-0$  //$NON-NLS-1$
	"Next Page >": "Next Page >", //$NON-NLS-0$  //$NON-NLS-1$
	"Show next page of git log": "Show next page of git log", //$NON-NLS-0$  //$NON-NLS-1$
	"Show next page of git tags" : "Show next page of git tags", //$NON-NLS-0$  //$NON-NLS-1$
	"Push from your local branch into the selected remote branch": "Push from your local branch into the selected remote branch", //$NON-NLS-0$  //$NON-NLS-1$
	"ResetActiveBr": "Reset your active branch to the state of this reference. Discard all staged and unstaged changes.", //$NON-NLS-0$  //$NON-NLS-1$
 	"GitResetIndexConfirm": "The content of your active branch will be replaced with commit \"${0}\". All unstaged and staged changes will be discarded and cannot be recovered if \"${1}\" is not checked. Are you sure?", //$NON-NLS-0$  //$NON-NLS-1$
	"Resetting index...": "Resetting index...", //$NON-NLS-0$  //$NON-NLS-1$
	"Resetting git index for ${0}" : "Resetting git index for ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Tag": "Tag", //$NON-NLS-0$  //$NON-NLS-1$
	"Create a tag for the commit": "Create a tag for the commit", //$NON-NLS-0$  //$NON-NLS-1$
	"ProjectSetup": "Your project is being set up. This might take a minute...", //$NON-NLS-0$  //$NON-NLS-1$
	"LookingForProject": "Looking for project: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Tag name": "Tag name", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete the tag from the repository": "Delete the tag from the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Are you sure you want to delete tag ${0}?": "Are you sure you want to delete tag ${0}?", //$NON-NLS-0$  //$NON-NLS-1$
	"Cherry-Pick": "Cherry-Pick", //$NON-NLS-0$  //$NON-NLS-1$
	"CherryPicking": "Cherry Picking commit: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"RevertingCommit": "Reverting commit: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Apply the change introduced by the commit to your active branch": "Apply the change introduced by the commit to your active branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Nothing changed.": "Nothing changed.", //$NON-NLS-0$  //$NON-NLS-1$
	". Some conflicts occurred": ". Some conflicts occurred.", //$NON-NLS-0$  //$NON-NLS-1$
	"Fetch from the remote branch into your remote tracking branch": "Fetch from the remote branch into your remote tracking branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Fetch Git Repository": "Fetch Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Push": "Push", //$NON-NLS-0$  //$NON-NLS-1$
	"Push from your local branch into the remote branch": "Push from your local branch into the remote branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Push Git Repository": "Push Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Key:": "Key:", //$NON-NLS-0$  //$NON-NLS-1$
	"Value:": "Value:", //$NON-NLS-0$  //$NON-NLS-1$
	"New Configuration Entry": "New Configuration Entry", //$NON-NLS-0$  //$NON-NLS-1$
	"Edit": "Edit", //$NON-NLS-0$  //$NON-NLS-1$
	"Edit the configuration entry": "Edit the configuration entry", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete the configuration entry": "Delete the configuration entry", //$NON-NLS-0$  //$NON-NLS-1$
	"Are you sure you want to delete ${0}?": "Are you sure you want to delete ${0}?", //$NON-NLS-0$  //$NON-NLS-1$
	"Clone Repository": "Clone Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Clone an existing Git repository to a folder": "Clone an existing Git repository to a folder", //$NON-NLS-0$  //$NON-NLS-1$
	"Cloning repository: ": "Cloning repository: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Init Repository": "Init Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Create a new Git repository in a new folder": "Create a new Git repository in a new folder", //$NON-NLS-0$  //$NON-NLS-1$
	"Initializing repository: ": "Initializing repository: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Init Git Repository": "Init Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Delete the repository": "Delete the repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Are you sure you want do delete ${0} repositories?": "Are you sure you want do delete ${0} repositories?", //$NON-NLS-0$  //$NON-NLS-1$
	"Apply a patch on the selected repository": "Apply a patch on the selected repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Show content": "Show content", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit name:": "Commit name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Open Commit": "Open Commit", //$NON-NLS-0$  //$NON-NLS-1$
	"OpenCommitDialog": "Open Commit", //$NON-NLS-0$  //$NON-NLS-1$
	"Open the commit with the given name": "Open the commit with the given name", //$NON-NLS-0$  //$NON-NLS-1$
	"No commits found": "No commits found", //$NON-NLS-0$  //$NON-NLS-1$
	"Staging changes": "Staging changes", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit message:": "Commit message:", //$NON-NLS-0$  //$NON-NLS-1$
	"Committing changes": "Committing changes", //$NON-NLS-0$  //$NON-NLS-1$
	"Fetching previous commit message": "Fetching previous commit message", //$NON-NLS-0$  //$NON-NLS-1$
	"Resetting local changes": "Resetting local changes", //$NON-NLS-0$  //$NON-NLS-1$
	"Checkout files, discarding all changes": "Checkout files, discarding all changes", //$NON-NLS-0$  //$NON-NLS-1$
	"Show Patch": "Show Patch", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading default workspace": "Loading default workspace", //$NON-NLS-0$  //$NON-NLS-1$
	"Show workspace changes as a patch": "Show workspace changes as a patch", //$NON-NLS-0$  //$NON-NLS-1$
	"Show checked changes as a patch": "Show checked changes as a patch", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowCommitPatchTip": "Show patch for changes in this commit", //$NON-NLS-0$  //$NON-NLS-1$
	"Continue": "Continue", //$NON-NLS-0$  //$NON-NLS-1$
	"Contibue Rebase": "Continue Rebase", //$NON-NLS-0$  //$NON-NLS-1$
	"Skip Patch": "Skip Patch", //$NON-NLS-0$  //$NON-NLS-1$
	"Abort": "Abort", //$NON-NLS-0$  //$NON-NLS-1$
	"Abort Rebase": "Abort Rebase", //$NON-NLS-0$  //$NON-NLS-1$
	"Discard": "Discard", //$NON-NLS-0$  //$NON-NLS-1$
	"Ignore": "Ignore", //$NON-NLS-0$  //$NON-NLS-1$
	"ChangesSelectedFilesDiscard": "Your changes to the selected files will be discarded and cannot be recovered.", //$NON-NLS-0$  //$NON-NLS-1$
 	"Getting git log": "Getting git log", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting stashed changes...": "Getting stashed changes...", //$NON-NLS-0$  //$NON-NLS-1$
	"Active Branch (${0})": "Active Branch (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"Branch (${0})": "Branch (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"Tag (${0})": "Tag (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit (${0})": "Commit (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"StashCommit (${0})": "Stash (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"WIPStash": "WIP on ${0}: ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"IndexStash": "index on ${0}: ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"RemoteTrackingBranch (${0})": "Remote Branch (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"Active Branch Log": "Git Log (Active Branch)", //$NON-NLS-0$  //$NON-NLS-1$
	"Show the log for the active local branch": "Show the log for the active local branch", //$NON-NLS-0$  //$NON-NLS-1$
	"Remote Branch Log": "Git Log (Remote Branch)", //$NON-NLS-0$  //$NON-NLS-1$
	"Show the log for the corresponding remote tracking branch": "Show the log for the corresponding remote tracking branch", //$NON-NLS-0$  //$NON-NLS-1$
	"See Full Status" : "See Full Status", //$NON-NLS-0$  //$NON-NLS-1$
	"See the status" : "See the status", //$NON-NLS-0$  //$NON-NLS-1$
	"Choose target location" : "Choose target location", //$NON-NLS-0$  //$NON-NLS-1$
	"Default target location" : "Default target location", //$NON-NLS-0$  //$NON-NLS-1$
	"Change..." : "Change...", //$NON-NLS-0$  //$NON-NLS-1$
	"Merge Squash": "Merge Squash", //$NON-NLS-0$  //$NON-NLS-1$
	"Squash the content of the branch to the index" : "Squash the content of the branch to the index", //$NON-NLS-0$  //$NON-NLS-1$
	"Local Branch Name:" : "Local Branch Name:", //$NON-NLS-0$  //$NON-NLS-1$
	"Local": "local", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter items" : "Filter items", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter filter" : "Filter message", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter author" : "Filter author", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter committer" : "Filter commiter", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter sha1" : "Filter sha1", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter fromDate" : "Filter from date YYYY-MM-DD or 1(h d w m y)", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter toDate" : "Filter to date YYYY-MM-DD or 1(h d w m y)", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter path" : "Filter path", //$NON-NLS-0$  //$NON-NLS-1$
	"Filter remote branches" : "Filter remote branches", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting remote branches" : "Getting remote branches ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting remote details": "Getting remote details: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"PatchApplied": "Patch applied successfully", //$NON-NLS-0$  //$NON-NLS-1$
	"PatchFailed": "Apply patch failed. ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Getting branches" : "Getting branches ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Paste link in email or IM" : "Paste link in email or IM", //$NON-NLS-0$  //$NON-NLS-1$
	"Show Commit in GitHub" : "Show Commit in GitHub", //$NON-NLS-0$  //$NON-NLS-1$
	"Show Repository in GitHub" : "Show Repository in GitHub", //$NON-NLS-0$  //$NON-NLS-1$
	"Show this commit in GitHub": "Show this commit in GitHub", //$NON-NLS-0$  //$NON-NLS-1$
	"Show Commit in eclipse.org": "Show Commit in eclipse.org", //$NON-NLS-0$  //$NON-NLS-1$
	"Show this commit in eclipse.org" : "Show this commit in eclipse.org", //$NON-NLS-0$  //$NON-NLS-1$
	"Show Repository in eclipse.org":"Show Repository in eclipse.org", //$NON-NLS-0$  //$NON-NLS-1$
	"Show this repository in eclipse.org":"Show this repository in eclipse.org", //$NON-NLS-0$  //$NON-NLS-1$
	"Ask for review" : "Ask for review", //$NON-NLS-0$  //$NON-NLS-1$
	"Ask for review tooltip" : "Send email with request for commit review", //$NON-NLS-0$  //$NON-NLS-1$
	"Reviewer name" : "Reviewer name", //$NON-NLS-0$  //$NON-NLS-1$
	"Contribution Review Request" : "Contribution Review Request", //$NON-NLS-0$  //$NON-NLS-1$
	"Send the link to the reviewer" : "Send the link to the reviewer", //$NON-NLS-0$  //$NON-NLS-1$
	"Private key file (optional):" : "Private key file (optional):", //$NON-NLS-0$  //$NON-NLS-1$
	"Don't prompt me again:" : "Don't prompt me again:", //$NON-NLS-0$  //$NON-NLS-1$
	"Your private key will be saved in the browser for further use" : "Your private key will be saved in the browser for further use", //$NON-NLS-0$  //$NON-NLS-1$
	"Loading Contribution Review Request..." : "Loading Contribution Review Request...", //$NON-NLS-0$  //$NON-NLS-1$
	"The commit can be found in the following repositories" : "The commit can be found in the following repositories", //$NON-NLS-0$  //$NON-NLS-1$
	"Try to update your repositories" : "Try to update your repositories", //$NON-NLS-0$  //$NON-NLS-1$
	"Create new repository" : "Create new repository", //$NON-NLS-0$  //$NON-NLS-1$
	"Attach the remote to one of your existing repositories" : "Attach the remote to one of your existing repositories", //$NON-NLS-0$  //$NON-NLS-1$
	"You are reviewing contribution ${0} from ${1}" : "You are reviewing contribution ${0} from ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitNotFoundInWorkspace" : "Unfortunately the commit can not be found in your workspace. To see it try one of the following: ", //$NON-NLS-0$  //$NON-NLS-1$
 	"To review the commit you can also:" : "To review the commit you can also:", //$NON-NLS-0$  //$NON-NLS-1$
	"Contribution Review Request for ${0} on ${1}" : "Contribution Review Request for ${0} on ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"Failing paths: ${0}": "Failing paths: ${0}", //$NON-NLS-0$  //$NON-NLS-1$
	"Problem while performing the action": "Problem while performing the action", //$NON-NLS-0$  //$NON-NLS-1$
	"Go to the Orion repositories page to provide a git repository URL. Once the repository is created, it will appear in the Navigator.": "Go to the Orion repositories page to provide a git repository URL. Once the repository is created, it will appear in the Navigator.", //$NON-NLS-0$  //$NON-NLS-1$
	"URL:": "URL:", //$NON-NLS-0$  //$NON-NLS-1$
	"File:": "File:", //$NON-NLS-0$  //$NON-NLS-1$
	"Submit": "Submit", //$NON-NLS-0$  //$NON-NLS-1$
	"git url:": "git url: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Revert": "Revert", //$NON-NLS-0$  //$NON-NLS-1$
	"Revert changes introduced by the commit into your active branch": "Revert changes introduced by the commit into your active branch", //$NON-NLS-0$  //$NON-NLS-1$
	". Could not revert into active branch": ". Could not revert into active branch.", //$NON-NLS-0$  //$NON-NLS-1$
	"Login": "Login", //$NON-NLS-0$  //$NON-NLS-1$
	"Authentication required for: ${0}. ${1} and re-try the request.": "Authentication required for: ${0}. ${1} and re-try the request.", //$NON-NLS-0$  //$NON-NLS-1$
	"Save":"Save", //$NON-NLS-0$  //$NON-NLS-1$
	"Remember my committer name and email:":"Remember my committer name and email:", //$NON-NLS-0$  //$NON-NLS-1$
	"Successfully edited ${0} to have value ${1}":"Successfully edited ${0} to have value ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"Successfully added ${0} with value ${1}":"Successfully added ${0} with value ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"Signed-off-by: ":"Signed-off-by: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Change-Id: ":"Change-Id: ", //$NON-NLS-0$  //$NON-NLS-1$
	"Push_REJECTED_NONFASTFORWARD":"Push is non-fastforward and was rejected. Use Fetch to see new commits that must be merged.", //$NON-NLS-0$  //$NON-NLS-1$
	"Commit and Push" : "Commit and Push", //$NON-NLS-0$  //$NON-NLS-1$
	"Sync" : "Sync", //$NON-NLS-0$  //$NON-NLS-1$
	"SyncTooltip" : "Fetch from the remote branch. Rebase your commits by removing them from the local branch, starting the local branch again based on the latest state of the remote branch and applying each commit to the updated local branch. Push commits and tags from your local branch into the remote branch.", //$NON-NLS-0$  //$NON-NLS-1$
	"NoCommits" : "No Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"NoContent" : "No Content", //$NON-NLS-0$  //$NON-NLS-1$
	"Incoming" : "Incoming", //$NON-NLS-0$  //$NON-NLS-1$
	"Outgoing" : "Outgoing", //$NON-NLS-0$  //$NON-NLS-1$
	"IncomingWithCount" : "Incoming (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"OutgoingWithCount" : "Outgoing (${0})", //$NON-NLS-0$  //$NON-NLS-1$
	"Synchronized" : "History", //$NON-NLS-0$  //$NON-NLS-1$
	"Uncommited" : "Uncommited", //$NON-NLS-0$  //$NON-NLS-1$
	"Repository:" : "Repository:", //$NON-NLS-0$  //$NON-NLS-1$
	"Reference:" : "Reference:", //$NON-NLS-0$  //$NON-NLS-1$
	"Author:" : "Author:", //$NON-NLS-0$  //$NON-NLS-1$
	"Committer:" : "Committer:", //$NON-NLS-0$  //$NON-NLS-1$
	"SHA1:" : "SHA1:", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowActiveBranchCmd" : "Show Active Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowReferenceCmd": "Show Reference", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowReferenceTip": "View the history of ${1} \"${2}\"", //$NON-NLS-0$  //$NON-NLS-1$
	"ShowActiveBranchTip": "View the history of \"${0}\" relative to ${1} \"${2}\"", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitType": "commit", //$NON-NLS-0$  //$NON-NLS-1$
	"BranchType": "branch", //$NON-NLS-0$  //$NON-NLS-1$
	"RemoteTrackingBranchType": "remote branch", //$NON-NLS-0$  //$NON-NLS-1$
	"TagType": "tag", //$NON-NLS-0$  //$NON-NLS-1$
	"StashCommitType": "stash", //$NON-NLS-0$  //$NON-NLS-1$
	"Path:" : "Path:", //$NON-NLS-0$  //$NON-NLS-1$
	"LocalChanges" : "Working Directory Changes", //$NON-NLS-0$  //$NON-NLS-1$
	"LocalChangesDetails" : "Working Directory Details", //$NON-NLS-0$  //$NON-NLS-1$
	"CompareChanges" : "Compare (${0} => ${1})", //$NON-NLS-0$  //$NON-NLS-1$
	"NoBranch" : "No Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"NoActiveBranch" : "No Active Branch", //$NON-NLS-0$  //$NON-NLS-1$
	"NoRef" : "No Selected Reference", //$NON-NLS-0$  //$NON-NLS-1$
	"None": "None", //$NON-NLS-0$  //$NON-NLS-1$
	"FileSelected": "${0} file selected", //$NON-NLS-0$  //$NON-NLS-1$
	"FilesSelected": "${0} files selected", //$NON-NLS-0$  //$NON-NLS-1$
	"FileChanged": "${0} file changed", //$NON-NLS-0$  //$NON-NLS-1$
	"FilesChanged": "${0} files changed", //$NON-NLS-0$  //$NON-NLS-1$
	"file": "file", //$NON-NLS-0$  //$NON-NLS-1$
	"files": "files", //$NON-NLS-0$  //$NON-NLS-1$
	"EmptyCommitConfirm": "You have no files selected. If you continue, you will end up with an empty commit. Do you want to continue?", //$NON-NLS-0$  //$NON-NLS-1$
	"AlwaysSelectFiles": "Always select changed files (see Git Settings)", //$NON-NLS-0$  //$NON-NLS-1$
	"EmptyCommitWarning": "The commit is empty", //$NON-NLS-0$  //$NON-NLS-1$
	"FilesChangedVsReadyToCommit": "${0} ${1} changed. ${2} ${3} ready to commit.", //$NON-NLS-0$  //$NON-NLS-1$
	"CommitPush": "Commit and Push", //$NON-NLS-0$  //$NON-NLS-1$
	"Commits and pushes files to the default remote": "Commits and pushes files to the default remote", //$NON-NLS-0$  //$NON-NLS-1$
	"Stash" : "Stash", //$NON-NLS-0$  //$NON-NLS-1$
	"stashIndex" : "stash@{${0}}: ${1}", //$NON-NLS-0$  //$NON-NLS-1$
	"Stash all current changes away" : "Stash all current changes away", //$NON-NLS-0$  //$NON-NLS-1$
	"Drop" : "Drop", //$NON-NLS-0$  //$NON-NLS-1$
	"Drop the commit from the stash list" : "Drop the commit from the stash list", //$NON-NLS-0$  //$NON-NLS-1$
	"Apply" : "Apply", //$NON-NLS-0$  //$NON-NLS-1$
	"Pop Stash" : "Pop Stash", //$NON-NLS-0$  //$NON-NLS-1$
	"Apply the most recently stashed change to your active branch and drop it from the stashes" : "Apply the most recently stashed change to your active branch and drop it from the stashes", //$NON-NLS-0$  //$NON-NLS-1$
	"stashes" : "stashes", //$NON-NLS-0$  //$NON-NLS-1$
	'addDependencyName': "Git Repository", //$NON-NLS-0$  //$NON-NLS-1$
	'addDependencyTooltip': "Associate a git repository with this project.",  //$NON-NLS-0$  //$NON-NLS-1$
	'addProjectName': "Git Repository",  //$NON-NLS-0$  //$NON-NLS-1$
	'addProjectTooltip': "Create a project from a git repository.",  //$NON-NLS-0$  //$NON-NLS-1$
	'fetchGroup': 'Fetch',  //$NON-NLS-0$  //$NON-NLS-1$
	'pushGroup' : 'Push',  //$NON-NLS-0$  //$NON-NLS-1$
	'Url:' : 'Url:', //$NON-NLS-0$  //$NON-NLS-1$
	'Ssh Private Key:' : 'Ssh Private Key:', //$NON-NLS-0$  //$NON-NLS-1$
	'Ssh Passphrase:' : 'Ssh Passphrase:', //$NON-NLS-0$  //$NON-NLS-1$
	'confirmUnsavedChanges': 'There are unsaved changes. Do you want to save them?', //$NON-NLS-1$ //$NON-NLS-0$
	'NewConfigurationEntryTooltip': "Add a new entry to the repository configuration" //$NON-NLS-1$ //$NON-NLS-0$
});


/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/

/**
 * Utility methods that do not have UI dependencies.
 */
/*globals URL */
define('orion/git/util',[
	'i18n!git/nls/gitmessages',
	'orion/i18nUtil',
	"orion/URL-shim"
], function(messages, i18nUtil) {

	var interestedUnstagedGroup = ["Missing", "Modified", "Untracked", "Conflicting"]; //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
	var interestedStagedGroup = ["Added", "Changed", "Removed"]; //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
	
	function isStaged(change) {
		for (var i = 0; i < interestedStagedGroup.length; i++) {
			if (change.type === interestedStagedGroup[i]) {
				return true;
			}
		}
		return false;
	}
	
	function isUnstaged(change) {
		for (var i = 0; i < interestedUnstagedGroup.length; i++) {
			if (change.type === interestedUnstagedGroup[i]) {
				return true;
			}
		}
		return false;
	}
	
	function isChange(change) {
		return isStaged(change) || isUnstaged(change);
	}
	
	function hasStagedChanges(status) {
		for (var i = 0; i < interestedStagedGroup.length; i++) {
			if (status[interestedStagedGroup[i]].length > 0) {
				return true;
			}
		}
		return false;
	}
	
	function hasUnstagedChanges(status) {
		for (var i = 0; i < interestedUnstagedGroup.length; i++) {
			if (status[interestedUnstagedGroup[i]].length > 0) {
				return true;
			}
		}
		return false;
	}

	/* parses ssh gitUrl to get hostname and port */
	function parseSshGitUrl(gitUrl){
		try {
			/* try ssh:// protocol */
			var url = new URL(gitUrl);
			return {
				host : url.hostname,
				port : url.port
			};
					
		} catch(e){
			/* try scp-like uri */
			try {
				/* [user@]host.xz:path/to/repo.git/ */
				var scp = gitUrl.split(":"); //$NON-NLS-0$
				var hostPart = scp[0].split("@"); //$NON-NLS-0$
				var host = hostPart.length > 1 ? hostPart[1] : hostPart[0];
				return {
					host : host,
					port : 22
				};
				
			} catch(ex){
				/* admit failure */
				return {
					host : "",
					port : ""
				};
			}
		}
	}
	/**
	 * Trims messages, skips empty lines until non-empty one is found
	 */
	function trimCommitMessage(message) {
		var splitted = message.split(/\r\n|\n/);
		var iterator = 0;
		
		while(splitted.length > 0 && /^\s*$/.test(splitted[iterator])) {
			iterator++;
		}
		var maxMessageLength = 100;
		if (splitted[iterator].length > maxMessageLength) return splitted[iterator].substring(0,maxMessageLength);
		return splitted[iterator];
	}
	
	function changeSignedOffByCommitMessage(name, email, message, isIncluded) {
		var sob = _generateSignedOffBy(name, email);
		if (sob) {
			if (isIncluded) {
				return message + sob;
			} else {
				return message.replace(sob, "");
			}
		}
		return message;
	}
	
	function _generateSignedOffBy(name, email) {
		if (name && email) {
			return "\n\nSigned-off-by: " + name + " <" + email + ">";
		}
	}
	
	/**
	 * Returns Change-Id and Signed-off-by of a commit if present
	 */
	function getGerritFooter(message) {
		
		var splitted = message.split(/\r\n|\n/);
		var footer = {};
		var changeIdCount = 0, 
			signedOffByPresent = false;
		for (var i = splitted.length-1; i >= 0; --i) {
			var changeId = "Change-Id: ";	//$NON-NLS-0$
			var signedOffBy = "Signed-off-by: ";	//$NON-NLS-0$
			if (splitted[i].indexOf(changeId) === 0) {
				footer.changeId = splitted[i].substring(changeId.length,splitted[i].length);
				if (++changeIdCount > 1) {
					footer = {};
					break;
				}
			} else if (!signedOffByPresent && splitted[i].indexOf(signedOffBy) === 0) {
				footer.signedOffBy = splitted[i].substring(signedOffBy.length,splitted[i].length);
				signedOffBy = true;
			}
		}
		
		return footer;
	}
	
	function shortenRefName(ref) {
		var refName = ref.Name;
		if (ref.Type === "StashCommit") { //$NON-NLS-0$
			refName = i18nUtil.formatMessage(messages["stashIndex"], ref.parent.children.indexOf(ref), refName.substring(0, 6)); //$NON-NLS-0$
		}
		if (ref.Type === "Commit") { //$NON-NLS-0$
			refName = refName.substring(0, 6);
		}
		if (ref.Type === "RemoteTrackingBranch" && !ref.Id) { //$NON-NLS-0$
			refName += messages[" [New branch]"];
		}
		return refName;
	}
	
	function shortenPath(path) {
		var result = path.split('/').slice(-3); //$NON-NLS-0$
		result = result.join("/"); //$NON-NLS-0$
		return result.length < path.length ? "..." + result : path; //$NON-NLS-0$
	}
	
	function relativePath(treePath) {
		var path = "";
		if (typeof treePath === "string") { //$NON-NLS-0$
			path = treePath;
		} else if (treePath) {
			var parents = treePath.Parents;
			if (parents.length > 1) {
				path = treePath.Location.substring(parents[parents.length -2].Location.length);
			}
		}
		return path;
	}
	
	function generateQuery(queries) {
		var result = queries.filter(function(q) { return q; }).join("&");  //$NON-NLS-0$
		if (result.length) {
			result = "?" + result;  //$NON-NLS-0$
		}
		return result;
	}
	
	function isNewBranch(branch) {
		return branch && branch.Type === "RemoteTrackingBranch" && !branch.Id; //$NON-NLS-0$
	}
	
	function tracksRemoteBranch(branch) {
		return branch && branch.RemoteLocation && branch.RemoteLocation[0] && !isNewBranch(branch.RemoteLocation[0].Children[0]);
	}
	
	return {
		isStaged: isStaged,
		isUnstaged: isUnstaged,
		isChange: isChange,
		isNewBranch: isNewBranch,
		tracksRemoteBranch: tracksRemoteBranch,
		generateQuery: generateQuery,
		hasStagedChanges: hasStagedChanges,
		hasUnstagedChanges: hasUnstagedChanges,
		parseSshGitUrl: parseSshGitUrl,
		trimCommitMessage: trimCommitMessage,
		changeSignedOffByCommitMessage: changeSignedOffByCommitMessage,
		shortenRefName: shortenRefName,
		shortenPath: shortenPath,
		relativePath: relativePath,
		getGerritFooter: getGerritFooter
	};
});

/******************************************************************************* 
 * @license
 * Copyright (c) 2013, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env amd, browser*/
/*global URL confirm*/
define('git/plugins/gitPlugin',[
	"orion/plugin",
	"orion/xhr",
	"orion/serviceregistry",
	"orion/git/gitClient",
	"orion/ssh/sshTools",
	"orion/i18nUtil",
	"orion/Deferred",
	"orion/git/GitFileImpl",
	"orion/git/util",
	"i18n!git/nls/gitmessages",
	"orion/URL-shim", // no exports
], function(PluginProvider, xhr, mServiceregistry, mGitClient, mSshTools, i18nUtil, Deferred, GitFileImpl, mGitUtil, gitmessages) {
	var serviceRegistry = new mServiceregistry.ServiceRegistry();
	var gitClient = new mGitClient.GitService(serviceRegistry);
	var sshService = new mSshTools.SshService(serviceRegistry);
	var login = new URL("../../mixloginstatic/LoginWindow.html", window.location.href).href;
	var headers = {
		name: "Orion Git Support",
		version: "1.0",
		description: "This plugin provides Git Support to the Orion File Service.",
		login: login
	};
	var GIT_TIMEOUT = 60000;

	var provider = new PluginProvider(headers);

	// Git category for contributed links

	provider.registerService("orion.edit.diff", {
		computeDiff: function(editorContext, context) {
			var diffTarget = new URL(context.metadata.Git.DiffLocation, window.location);
			diffTarget.query.set("parts", "diff");
			return xhr("GET", diffTarget.href, {
				headers: {
					"Orion-Version": "1"
				},
				timeout: 10000
			}).then(function(result) {
				return result.responseText;
			});
		}
	}, {
		name: "Git Diff",
		validationProperties: [{
			source: "Git",
			variableName: "Git"
		} //$NON-NLS-1$ //$NON-NLS-0$
		]
	}); //$NON-NLS-0$

	provider.registerService("orion.page.link.category", null, {
		id: "git",
		name: gitmessages["Git"],
		nls: "git/nls/gitmessages",
		imageClass: "core-sprite-git-logo",
		order: 20
	});

	provider.registerService("orion.page.link", {}, {
		name: gitmessages["Repositories"],
		id: "orion.git.repositories",
		nls: "git/nls/gitmessages",
		category: "git",
		order: 1000, // low priority
		uriTemplate: "{+OrionHome}/git/git-repository.html#"
	});
	
	provider.registerService("orion.navigate.command", {}, {
		name: gitmessages["Git Log"],
		id: "eclipse.git.log",
		tooltip: gitmessages["Go to Git Log"],
		nls: "git/nls/gitmessages",
		validationProperties: [
			{source: "Git:CommitLocation", variableName: "GitLogLocation"}
		],
		uriTemplate: "{+OrionHome}/git/git-repository.html#{,GitLogLocation}?page=1",
		forceSingleItem: true
	});
	
	
	// orion.navigate.command for Git Repository -- applies to File objects
	provider.registerService("orion.navigate.command", null, {
		id: "eclipse.git.repository",
		name: gitmessages["Git Repository"],
		tooltip: gitmessages["Go to the git repository"],
		nls: "git/nls/gitmessages",
		category: "git",
		validationProperties: [{
			source: "Git:CloneLocation",
			variableName: "GitRepoLocation"
		}],
		uriTemplate: "{+OrionHome}/git/git-repository.html#{,GitRepoLocation}"
	});

	provider.registerService("orion.core.content", null, {
		id: "orion.content.gitClone",
		nls: "git/nls/gitmessages",
		name: gitmessages["Clone Git Repository"],
		description: gitmessages["Go to the Orion repositories page to provide a git repository URL. Once the repository is created, it will appear in the Navigator."],
		uriTemplate: "{+OrionHome}/git/git-repository.html#,cloneGitRepository=URL"
	});

	provider.registerService("orion.page.link.related", null, {
		id: "eclipse.git.repository",
		category: "git",
		order: 10, // Git Repository should be first in the Git category
	});
	
	provider.registerService("orion.page.link.related", null, {
		id: "eclipse.git.log",
		category: "git"
	});
	/*
	provider.registerService("orion.page.link.related", null, {
		id: "eclipse.git.remote"
	});
	*/
	provider.registerService("orion.page.link.related", null, {
		name: gitmessages["Active Branch Log"],
		id: "eclipse.orion.git.switchToCurrentLocal",
		tooltip: gitmessages["Show the log for the active local branch"],
		nls: "git/nls/gitmessages",
		category: "git",
		validationProperties: [
			{source: "Clone:ActiveBranch", variableName: "GitBranchLocation"},
			{source: "toRef:Type", match: "RemoteTrackingBranch"}
		],
		uriTemplate: "{+OrionHome}/git/git-repository.html#{,GitBranchLocation}?page=1",
		forceSingleItem: true
	});
	
	provider.registerService("orion.page.link.related", null, {
		name: gitmessages["Remote Branch Log"],
		id: "eclipse.orion.git.switchToRemote2",
		tooltip: gitmessages["Show the log for the corresponding remote tracking branch"],
		nls: "git/nls/gitmessages",
		category: "git",
		validationProperties: [
			{source: "toRef:RemoteLocation:0:Children:0:CommitLocation", variableName: "GitRemoteLocation"}
		],
		uriTemplate: "{+OrionHome}/git/git-repository.html#{,GitRemoteLocation}?page=1",
		forceSingleItem: true
	});

	// Applies to Git commit objects
	provider.registerService("orion.page.link.related", null, {
		id: "eclipse.git.repository2",
		category: "git",
		order: 10,
		name: gitmessages["Git Repository"],
		tooltip: gitmessages["Go to the git repository"],
		nls: "git/nls/gitmessages",
		validationProperties: [
			{source: "CloneLocation", variableName: "GitCloneLocation"},
			{source: "Type", match: "Commit"}
		],
		uriTemplate: "{+OrionHome}/git/git-repository.html#{,GitCloneLocation}"
	});
	
	provider.registerService("orion.page.link.related", null, {
		id: "eclipse.git.repository3",
		category: "git",
		order: 10,
		name: gitmessages["Git Repository"],
		tooltip: gitmessages["Go to the git repository"],
		nls: "git/nls/gitmessages",
		validationProperties: [
			{source: "Location", variableName: "GitCloneLocation"},
			{source: "Type", match: "Clone"}
		],
		uriTemplate: "{+OrionHome}/git/git-repository.html#{,GitCloneLocation}"
	});
	
	// Applies to File objects
	provider.registerService("orion.page.link.related", null, {
		id: "eclipse.git.repository", // ref existing orion.navigate.command
		category: "git"
	});

	provider.registerService("orion.page.link.related", null, {
		id: "orion.git.gotoEclipseGit",
		name: gitmessages["Show Repository in eclipse.org"],
		tooltip: gitmessages["Show this repository in eclipse.org"],
		nls: "git/nls/gitmessages",
		category: "git",
		validationProperties: [{
			source: "GitUrl|Clone:GitUrl", 
			match: "git.eclipse.org/gitroot", 
			variableName: "EclipseGitLocation", 
			variableMatchPosition: "after"
		}],
		uriTemplate: "http://git.eclipse.org/c{+EclipseGitLocation}"
	});
	
	provider.registerService("orion.page.link.related", null, {
		id: "orion.git.gotoGithub",
		name: gitmessages["Show Repository in GitHub"],
		nls: "git/nls/gitmessages",
		tooltip: gitmessages["Show this repository in GitHub"],
		category: "git",
		validationProperties: [{
			source: "GitUrl|Clone:GitUrl", 
			match: "github\.com.*\.git", 
			variableName: "GitHubLocation", 
			variableMatchPosition: "only",
			replacements: [{pattern: ":", replacement: "/"}, {pattern: ".git$", replacement: ""}]
		}],
		uriTemplate: "https://{+GitHubLocation}"
	});
	
	provider.registerServiceProvider("orion.page.link.related", null, {
		id: "orion.git.gotoGithubCommit",
		name: gitmessages["Show Commit in GitHub"],
		nls: "git/nls/gitmessages",
		tooltip: gitmessages["Show this commit in GitHub"],
		category: "git",
		validationProperties: [{
			source: "GitUrl", 
			match: "github\.com.*\.git", 
			variableName: "GitHubLocation", 
			variableMatchPosition: "only",
			replacements: [{pattern: ":", replacement: "/"}, {pattern: ".git$", replacement: ""}]
		},
		{source: "Type", match: "Commit"},
		{source: "Name", variableName: "commitName"}
		],
		uriTemplate: "https://{+GitHubLocation}/commit/{+commitName}"
	});
	
	provider.registerServiceProvider("orion.page.link.related", null, {
		id: "orion.git.gotoEclipseGitCommit",
		name: gitmessages["Show Commit in eclipse.org"],
		nls: "git/nls/gitmessages",
		tooltip: gitmessages["Show this commit in eclipse.org"],
		category: "git",
		validationProperties: [{
			source: "GitUrl", 
			match: "git.eclipse.org/gitroot", 
			variableName: "EclipseGitLocation", 
			variableMatchPosition: "after"
		},
		{source: "Type", match: "Commit"},
		{source: "Name", variableName: "commitName"}
		],
		uriTemplate: "http://git.eclipse.org/c{+EclipseGitLocation}/commit/?id={+commitName}"
	});
	
	var tryParentRelative = true;
	function makeParentRelative(location) {
		if (tryParentRelative) {
			try {
				if (window.location.host === parent.location.host && window.location.protocol === parent.location.protocol) {
					return location.substring(parent.location.href.indexOf(parent.location.host) + parent.location.host.length);
				} else {
					tryParentRelative = false;
				}
			} catch (e) {
				tryParentRelative = false;
			}
		}
		return location;
	}
	var queries = new URL(window.location.href).query;
	var gitFSPattern = queries.get("gitFSPattern");
	var gitBase = gitFSPattern ? gitFSPattern : makeParentRelative(new URL("../../gitapi/", window.location.href).href);
	var service = new GitFileImpl(gitBase);

	provider.registerService("orion.core.file", service, {
		Name: 'Git File System',
		top: gitBase,
		pattern: gitBase
	});

	var gitDiffPattern = queries.get("gitDiffPattern");
	var base = gitDiffPattern ? gitDiffPattern : new URL("../../gitapi/diff/", window.location.href).href;
	provider.registerService("orion.core.diff", {
		getDiffContent: function(diffURI, options){	
			var url = new URL(diffURI, window.location);
			url.query.set("parts", "diff");
			if (options && typeof options === "object") {
				Object.keys(options).forEach(function(param) {
					url.query.set(param, options[param]);
				});
			}
			return xhr("GET", url.href, {
				headers: {
					"Orion-Version": "1"
				},
				timeout: GIT_TIMEOUT
			}).then(function(xhrResult) {
				return xhrResult.responseText;
			});
		},			
		getDiffFileURI: function(diffURI){
			var url = new URL(diffURI, window.location);
			url.query.set("parts", "uris");
			return xhr("GET", url.href, {
				headers: {
					"Orion-Version": "1"
				},
				timeout: GIT_TIMEOUT
			}).then(function(xhrResult) {
				return JSON.parse(xhrResult.responseText);
			});
		}
	}, {
		pattern: base
	});
	
	function parseGitUrl(gitUrl){
		var gitPath = gitUrl;
		var gitInfo = {};
		if(gitUrl.indexOf("://")>0){
			gitPath = gitUrl.substring(gitUrl.indexOf("://")+3);
		}
		var segments = gitPath.split("/");
		gitInfo.serverName = segments[0];
		if(gitInfo.serverName.indexOf("@")){
			gitInfo.serverName = gitInfo.serverName.substring(gitInfo.serverName.indexOf("@")+1);
		}
		gitInfo.repoName = segments[segments.length-1];
		if(gitInfo.repoName.indexOf(".git")>0){
			gitInfo.repoName = gitInfo.repoName.substring(0, gitInfo.repoName.lastIndexOf(".git"));
		}
		return gitInfo;
	}
	
	function removeUserInformation(gitUrl){
		if(gitUrl.indexOf("@")>0 && gitUrl.indexOf("ssh://")>=0){
			return gitUrl.substring(0, gitUrl.indexOf("ssh://") + 6) + gitUrl.substring(gitUrl.indexOf("@")+1);
		}
		return gitUrl;
	}
	
	provider.registerService("orion.project.handler", {
		paramsToDependencyDescription: function(params){
			return {Type: "git", Location: removeUserInformation(params.url)};
		},
		_cloneRepository: function(gitUrl, params, workspaceLocation, isProject){
			var self = this;
			var deferred = new Deferred();
			
			/* parse gitURL */
			var repositoryURL = mGitUtil.parseSshGitUrl(gitUrl);
			sshService.getKnownHostCredentials(repositoryURL.host, repositoryURL.port).then(function(knownHosts){
				gitClient.cloneGitRepository(null, gitUrl, null, workspaceLocation, params.sshuser, params.sshpassword, knownHosts, params.sshprivateKey, params.sshpassphrase, null, isProject).then(function(cloneResp){
					gitClient.getGitClone(cloneResp.Location).then(function(clone){
						if(clone.Children){
							clone = clone.Children[0];
						}
						var gitInfo = parseGitUrl(clone.GitUrl);
						if(isProject){
							deferred.resolve({ContentLocation: clone.ContentLocation});					
						} else {
							deferred.resolve({Type: "git", Location: removeUserInformation(clone.GitUrl), Name: (gitInfo.repoName || clone.Name) + " at " + gitInfo.serverName});					
						}
					}, deferred.reject, deferred.progress);
				}.bind(this), function(error){
					try{
						if (error && error.status !== undefined) {
							try {
								error = JSON.parse(error.responseText);
							} catch (e) {
								error = { 
									Message : "Problem while performing the action"
								};
							}
						}
					}catch(e){
						deferred.reject(error);
						return;
					}
					if(error.JsonData){
						if(error.JsonData.HostKey){
							if(confirm(i18nUtil.formatMessage('Would you like to add ${0} key for host ${1} to continue operation? Key fingerpt is ${2}.',
								error.JsonData.KeyType, error.JsonData.Host, error.JsonData.HostFingerprint))){
									
									var hostURL = mGitUtil.parseSshGitUrl(error.JsonData.Url);
									var hostCredentials = {
											host : error.JsonData.Host,
											keyType : error.JsonData.KeyType,
											hostKey : error.JsonData.HostKey,
											port : hostURL.port
										};
									
									sshService.addKnownHost(hostCredentials).then(function(){
										self._cloneRepository(gitUrl, params, workspaceLocation).then(deferred.resolve, deferred.reject, deferred.progress);
									});
									
							} else {
								deferred.reject(error);
							}
							return;
						} 
						if(error.JsonData.Host){
							error.Retry = {
								addParameters : [{id: "sshuser", type: "text", name:  gitmessages["User Name:"]}, {id: "sshpassword", type: "password", name:  gitmessages["Password:"]}],
								optionalParameters: [{id: "sshprivateKey", type: "textarea", name:  gitmessages["Ssh Private Key:"]}, {id: "sshpassphrase", type: "password", name:  gitmessages["Ssh Passphrase:"]}]
							};
							deferred.reject(error);
							return;
						}
					}
					deferred.reject(error);
				}.bind(this), deferred.progress);	
			});
			
			return deferred;
		},
		initDependency: function(dependency, params, projectMetadata){
			var gitUrl = removeUserInformation(dependency.Location || params.url);
			return this._cloneRepository(gitUrl, params, projectMetadata.WorkspaceLocation);
		},
		initProject: function(params, projectMetadata){
			var gitUrl = removeUserInformation(params.url);
			return this._cloneRepository(gitUrl, params, projectMetadata.WorkspaceLocation, true);
		},
		getDependencyDescription: function(item){
			if(!item.Git){
				return null;
			}
			var deferred = new Deferred();
			gitClient.getGitClone(item.Git.CloneLocation).then(
				function(clone){
					if(clone.Children){
						clone = clone.Children[0];
					}
					if(clone.GitUrl){
						var gitInfo = parseGitUrl(clone.GitUrl);
						deferred.resolve({Type: "git", Location: removeUserInformation(clone.GitUrl), Name: (gitInfo.repoName || clone.Name) + " at " + gitInfo.serverName});
					}
				},deferred.reject, deferred.progress
			);
			return deferred;
		},
		getAdditionalProjectProperties: function(item, projectMetadata){
			if(!item.Git){
				return null;
			}
			var deferred = new Deferred();
			gitClient.getGitClone(item.Git.CloneLocation).then(
			function(clone){
				if(clone.Children){
					clone = clone.Children[0];
				}
				deferred.resolve([
					{
						Name: "Git",
						Children: [
							{
								Name: "Git Url",
								Value: clone.GitUrl
							},
							{
								Name: "Git Repository",
								Value: "Git Repository",
								Href: "{+OrionHome}/git/git-repository.html#" + item.Git.CloneLocation
							}
						]
					}
				]);
			},deferred.reject, deferred.progress
			);
			return deferred;
		}
	}, {
		id: "orion.git.projecthandler",
		type: "git",
		addParameters: [{id: "url", type: "url", name: gitmessages["Url:"]}],
		optionalParameters: [{id: "sshuser", type: "text", name:  gitmessages["User Name:"]}, {id: "sshpassword", type: "password", name:  gitmessages["Password:"]},{id: "sshprivateKey", type: "textarea", name:  gitmessages["Ssh Private Key:"]}, {id: "sshpassphrase", type: "password", name:  gitmessages["Ssh Passphrase:"]}],
		nls: "git/nls/gitmessages",
		addDependencyName: gitmessages["addDependencyName"],
		addDependencyTooltip: gitmessages["addDependencyTooltip"],
		addProjectName: gitmessages["addProjectName"],
		addProjectTooltip: gitmessages["addProjectTooltip"],
		actionComment: "Cloning ${url}",
		validationProperties: [
			{source: "Git"} // alternate {soruce: "Children:[Name]", match: ".git"}
		]
	});
	
	provider.connect();
});


//# sourceMappingURL=gitPlugin.js.src.js.map
