/*****************************************************************************
 * Copyright (c) 2014 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.types.advice;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gef.commands.UnexecutableCommand;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.common.ui.util.DisplayUtils;
import org.eclipse.gmf.runtime.diagram.ui.commands.CommandProxy;
import org.eclipse.gmf.runtime.emf.type.core.commands.ConfigureElementCommand;
import org.eclipse.gmf.runtime.emf.type.core.commands.SetValueCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.ConfigureRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateRelationshipRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.papyrusrt.umlrt.core.Messages;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsuleUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.CapsulePart;
import org.eclipse.uml2.uml.AggregationKind;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * Edit Helper Advice for {@link CapsulePart}
 */
public class CapsulePartEditHelperAdvice extends AbstractEditHelperAdvice {

	/** Lower bound of multiplicity */
	protected int lower = -1;
	/** Upper Bound of Multiplicity */
	protected int upper = -1;
	/** Multiplicity Dialog */
	protected static MessageDialog multiplicityDialog;
	/** Aggregation Dialog */
	protected static MessageDialog aggregationDialog;

	public CapsulePartEditHelperAdvice() {
		super();
		// Create the Dialog at the Advice construction to be created only once.
		multiplicityDialog = new MessageDialog(DisplayUtils.getDefaultShell(), Messages.WarningMessageDialog, null, Messages.NotAllowedMultiplicity, MessageDialog.WARNING,
				new String[] { Messages.OK_Button_Label }, 0);
		aggregationDialog = new MessageDialog(DisplayUtils.getDefaultShell(), Messages.WarningMessageDialog, null, Messages.ReadOnlyAggregationForCapsulePart, MessageDialog.WARNING,
				new String[] { Messages.OK_Button_Label }, 0);

	}

	@Override
	protected ICommand getBeforeCreateRelationshipCommand(CreateRelationshipRequest request) {
		return super.getBeforeCreateRelationshipCommand(request);
	}

	@Override
	public boolean approveRequest(IEditCommandRequest request) {
		// check that the type of the RTPort is a protocol, and nothing else
		if (request instanceof SetRequest) {
			SetRequest setRequest = (SetRequest) request;
			EStructuralFeature feature = setRequest.getFeature();
			if (UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature)) {
				// new value should be a protocol or null
				Object newValue = ((SetRequest) request).getValue();
				if (newValue instanceof Classifier) {
					if (CapsuleUtils.isCapsule((Classifier) newValue)) {
						return true;
					} else {
						return false;
					}
				} else {
					return false;
				}
			}
			return super.approveRequest(setRequest);

		}
		return super.approveRequest(request);
	}

	@Override
	protected ICommand getAfterConfigureCommand(ConfigureRequest request) {
		final Property capsulePart = (Property) request.getElementToConfigure();

		return new ConfigureElementCommand(request) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor progressMonitor, IAdaptable info) throws ExecutionException {
				capsulePart.setAggregation(AggregationKind.COMPOSITE_LITERAL);
				capsulePart.setLower(1);
				capsulePart.setUpper(1);
				capsulePart.setName(null);// set to null to let it be named when type is set
				return CommandResult.newOKCommandResult(capsulePart);
			}

		};
	}

	@Override
	protected ICommand getAfterSetCommand(SetRequest request) {
		ICommand command = null;
		EStructuralFeature feature = request.getFeature();
		if (UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature)) {
			// if element is unnamed, set a name according to the new Type name
			Object newValue = request.getValue();
			EObject elementToEdit = request.getElementToEdit();
			if (newValue instanceof Type && elementToEdit instanceof Property && ((Property) elementToEdit).getName() == null) {
				String name = ((Type) newValue).getName();
				if (name != null && !name.isEmpty()) {
					String newName = Character.toLowerCase(name.charAt(0)) + name.substring(1);
					command = new SetValueCommand(new SetRequest(request.getElementToEdit(), UMLPackage.eINSTANCE.getNamedElement_Name(), newName));
				}
			}
		} else if (UMLPackage.eINSTANCE.getProperty_Aggregation().equals(feature)) { // Aggregation is in Read Only for Capsule part
			aggregationDialog.open();
			command = new CommandProxy(UnexecutableCommand.INSTANCE);
		} else if (UMLPackage.eINSTANCE.getMultiplicityElement_Lower().equals(feature)) { // Prepare Lower Value for multiplicity check
			lower = (Integer) request.getValue();
		} else if (UMLPackage.eINSTANCE.getMultiplicityElement_Upper().equals(feature)) { // Check the multiplicity in UML.
			upper = (Integer) request.getValue();
			if (!((lower > 0 && lower == upper) || (lower == 0 && upper > 0))) {// Allow only n..n or 0..n

				// Display message if multiplicity is not allowed
				multiplicityDialog.open();
				command = new CommandProxy(UnexecutableCommand.INSTANCE);
			}
		} else {

			command = super.getAfterSetCommand(request);
		}

		return command;
	}




}
