/*****************************************************************************
 * Copyright (c) 2015 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.modelexplorer.provider;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.emf.type.core.IElementMatcher;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.ISpecializationType;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrus.infra.services.labelprovider.service.IFilteredLabelProvider;
import org.eclipse.papyrusrt.umlrt.core.types.UMLRTElementTypesEnumerator;
import org.eclipse.papyrusrt.umlrt.core.utils.CallEventUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.InterfaceRealizationUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.MessageSetUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.ProtocolContainerUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.UsageUtils;
import org.eclipse.papyrusrt.umlrt.tooling.modelexplorer.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.modelexplorer.utils.LabelUtils;
import org.eclipse.swt.graphics.Image;
import org.eclipse.uml2.uml.CallEvent;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.InterfaceRealization;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Usage;

/**
 * Label provider used by the label provider service
 */
public class UMLRTFilteredLabelProvider extends UMLRTLabelProvider implements IFilteredLabelProvider {

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean accept(Object element) {
		EObject semanticObject = EMFHelper.getEObject(element);

		// element should be an UML element at least. UML-RT profile should be present.
		if (!(semanticObject instanceof Element)) {
			return false;
		}

		for (IElementType type : UMLRTElementTypesEnumerator.getAllRTTypes()) {
			if (type instanceof ISpecializationType) {
				IElementMatcher matcher = ((ISpecializationType) type).getMatcher();
				if (matcher != null) {
					if (matcher.matches(semanticObject)) {
						return true;
					}
				} else {
					Activator.log.debug("no matcher for this element type: " + type);
				}

			}
		}

		if (semanticObject instanceof CallEvent) {
			if (CallEventUtils.isProtocolMessageCallEvent((CallEvent) semanticObject)) {
				return true;
			}
		}

		if (semanticObject instanceof Interface) {
			if (MessageSetUtils.isRTMessageSet((Interface) semanticObject)) {
				return true;
			}
		}

		if (semanticObject instanceof InterfaceRealization) {
			if (InterfaceRealizationUtils.isInterfaceRealizationFromProtocol((InterfaceRealization) semanticObject)) {
				return true;
			}
		}

		if (semanticObject instanceof Usage) {
			if (UsageUtils.isUsageFromProtocol((Usage) semanticObject)) {
				return true;
			}
		}

		
		return false;
	}

	/**
	 * @see org.eclipse.papyrus.uml.tools.providers.UMLLabelProvider#getImage(org.eclipse.emf.ecore.EObject)
	 *
	 * @param element
	 * @return
	 */
	@Override
	public Image getImage(Object element) {
		EObject semanticObject = EMFHelper.getEObject(element);

		if (!(semanticObject instanceof Element)) {
			Activator.log.debug("Trying to display an UMLRT image for a non UML-RT element");
			return null;
		}

		// depending on the element type that matches, return a different icon
		String matchingTypeMatcher = getMatchingType(semanticObject);

		if (matchingTypeMatcher == null) {
			// check for UML related elements, like CallEvents
			if (semanticObject instanceof Element) {
				return baseEMFLabelProvider.getImage(semanticObject);
			}
			return null;
		}
		return super.getImage(element);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getText(Object element) {
		EObject semanticObject = EMFHelper.getEObject(element);
		if (semanticObject == null) {
			return super.getText(element);
		}

		// specific for RT Message label
		ISpecializationType inMessageType = (ISpecializationType) UMLRTElementTypesEnumerator.PROTOCOL_MESSAGE_IN;
		if (inMessageType != null && inMessageType.getMatcher() != null && inMessageType.getMatcher().matches((EObject) semanticObject)) {
			return "in " + super.getText(element);
		}
		ISpecializationType outMessageType = (ISpecializationType) UMLRTElementTypesEnumerator.PROTOCOL_MESSAGE_OUT;
		if (outMessageType != null && outMessageType.getMatcher() != null && outMessageType.getMatcher().matches((EObject) semanticObject)) {
			return "out " + super.getText(element);
		}
		ISpecializationType inOutMessageType = (ISpecializationType) UMLRTElementTypesEnumerator.PROTOCOL_MESSAGE_INOUT;
		if (inOutMessageType != null && inOutMessageType.getMatcher() != null && inOutMessageType.getMatcher().matches((EObject) semanticObject)) {
			return "inout " + super.getText(element);
		}

		/* @noname */
		// for unnamed elements (protocol container, callevents for ProtocolMessages), deduce the name from attached element
		if (semanticObject instanceof CallEvent) {
			if (CallEventUtils.isProtocolMessageCallEvent((CallEvent) semanticObject)) {
				return LabelUtils.getCallEventForProtocolMessageLabel((CallEvent) semanticObject, labelProvider);
			}
		}

		if (semanticObject instanceof Package) {
			if (ProtocolContainerUtils.isProtocolContainer(semanticObject)) {
				return LabelUtils.getProtocolContainerLabel((Package) semanticObject, labelProvider);
			}
		}

		if (semanticObject instanceof Interface) {
			if (MessageSetUtils.isRTMessageSet((Interface) semanticObject)) {
				return LabelUtils.getMessageSetLabel((Interface) semanticObject, labelProvider);
			}
		}

		if (semanticObject instanceof InterfaceRealization) {
			if (InterfaceRealizationUtils.isInterfaceRealizationFromProtocol((InterfaceRealization) semanticObject)) {
				return LabelUtils.getInterfaceRealizationLabel((InterfaceRealization) semanticObject, labelProvider);
			}
		}

		if (semanticObject instanceof Usage) {
			if (UsageUtils.isUsageFromProtocol((Usage) semanticObject)) {
				return LabelUtils.getUsageLabel((Usage) semanticObject, labelProvider);
			}
		}


		return super.getText(element);
	}

}
