/*****************************************************************************
 * Copyright (c) 2016, 2017 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.core.utils;

import static java.util.Spliterators.spliteratorUnknownSize;

import java.util.Iterator;
import java.util.Spliterator;
import java.util.stream.StreamSupport;

import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.transaction.Transaction;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.impl.InternalTransaction;
import org.eclipse.emf.transaction.impl.InternalTransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.common.core.command.ICompositeCommand;
import org.eclipse.gmf.runtime.common.core.command.IdentityCommand;
import org.eclipse.gmf.runtime.common.core.command.UnexecutableCommand;

/**
 * Various static utilities for working with commands.
 */
public class UMLRTCommandUtils {

	// Not instantiable by clients
	private UMLRTCommandUtils() {
		super();
	}

	/**
	 * An alternative to the {@link ICommand#compose(org.eclipse.core.commands.operations.IUndoableOperation)}
	 * API that compose commands into a single, flat composite, instead of adding nested composites
	 * to the result.
	 * 
	 * @param c1
	 *            a possible composite command
	 * @param c2
	 *            another possibly composite command
	 * 
	 * @return a command that composes {@code c1} with {@code c2} or its components,
	 *         according to whethr {@code c2} is composite
	 */
	public static ICommand flatCompose(ICommand c1, ICommand c2) {
		ICommand result;

		if (c2 instanceof UnexecutableCommand) {
			// That's enough
			result = c2;
		} else if (c1 instanceof UnexecutableCommand) {
			// Or that
			result = c1;
		} else if (c2 instanceof ICompositeCommand) {
			ICompositeCommand cc2 = (ICompositeCommand) c2;
			@SuppressWarnings("unchecked")
			Iterator<ICommand> commands = cc2.iterator();
			result = StreamSupport.stream(
					spliteratorUnknownSize(commands, Spliterator.NONNULL | Spliterator.ORDERED),
					false).reduce(c1, UMLRTCommandUtils::flatCompose);
		} else if (c1 != null) {
			result = ((c2 == null) || (c2 instanceof IdentityCommand))
					? c1
					: (c1 instanceof IdentityCommand)
							? c2
							: c1.compose(c2);
		} else {
			result = c2; // Whether it's null or not
		}

		return result;
	}

	/**
	 * Queries whether changes reported by a {@code notifier} are happening in the
	 * context of an undo or a redo of a command execution, or roll-back of a
	 * transaction, which is a kind of undo.
	 * 
	 * @param notifier
	 *            some EMF run-time object
	 * @return whether it is being changed by undo/redo or roll-back
	 */
	public static boolean isUndoRedoInProgress(Notifier notifier) {
		boolean result = false;

		TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(notifier);
		if (domain instanceof InternalTransactionalEditingDomain) {
			InternalTransaction active = ((InternalTransactionalEditingDomain) domain).getActiveTransaction();
			result = (active != null)
					&& (Boolean.TRUE.equals(active.getOptions().get(Transaction.OPTION_IS_UNDO_REDO_TRANSACTION))
							|| active.isRollingBack());
		}

		return result;
	}
}
