/*****************************************************************************
 * Copyright (c) 2016, 2017 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.types.advice;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.DestroyDependentsRequest;
import org.eclipse.gmf.runtime.notation.NotationPackage;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrusrt.umlrt.core.utils.UMLRTCommandUtils;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.utils.EditPartInheritanceUtils;
import org.eclipse.papyrusrt.umlrt.uml.util.UMLRTExtensionUtil;
import org.eclipse.uml2.uml.Element;

/**
 * Advice that handles the special case of notation dependents deletion for
 * inherited views: these are views that reference the inherited element
 * that is not being deleted but that actually need to be deleted because
 * a virtual element redefining it that the view actually visualizes was deleted.
 */
public class DeleteInheritedViewsAdvice extends AbstractEditHelperAdvice {

	/**
	 * Initializes me.
	 */
	public DeleteInheritedViewsAdvice() {
		super();
	}

	@Override
	protected ICommand getBeforeDestroyDependentsCommand(DestroyDependentsRequest request) {
		ICommand result = null;

		EObject destructee = request.getElementToDestroy();
		if (destructee instanceof Element) {
			result = getBeforeDestroyDependentsCommand(request, (Element) destructee);
		}

		return result;
	}

	protected ICommand getBeforeDestroyDependentsCommand(DestroyDependentsRequest request, Element destructee) {
		ICommand result;

		Optional<Element> inherited = Optional.of(destructee)
				.map(UMLRTExtensionUtil::getRootDefinition);

		// Look for views that visualize this element but which in the notation
		// model actually reference the root definition
		Stream<View> views = inherited.map(this::getViews)
				.orElseGet(Stream::empty)
				.filter(v -> {
					EObject semantic = EditPartInheritanceUtils.resolveSemanticElement(v);
					return (semantic != destructee)
							&& (semantic instanceof Element)
							&& UMLRTExtensionUtil.redefines((Element) semantic, destructee);
				});

		List<View> views_ = views.collect(Collectors.toList());
		views = views_.stream();

		result = views.map(request::getDestroyDependentCommand)
				.filter(Objects::nonNull)
				.reduce(UMLRTCommandUtils::flatCompose)
				.orElse(null);

		return result;
	}

	Stream<View> getViews(Element element) {
		return EMFHelper.getUsages(element).stream()
				.filter(s -> s.getEStructuralFeature() == NotationPackage.Literals.VIEW__ELEMENT)
				.map(EStructuralFeature.Setting::getEObject)
				.map(View.class::cast)
				.filter(v -> v.eResource() != null); // Only views not already deleted
	}
}
