/*****************************************************************************
 * Copyright (c) 2016, 2017 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.databinding.facade;

import static java.util.stream.Collectors.toList;
import static org.eclipse.core.databinding.observable.Diffs.createListDiff;
import static org.eclipse.core.databinding.observable.Diffs.createListDiffEntry;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.Realm;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.list.ListDiff;
import org.eclipse.core.databinding.observable.list.ListDiffVisitor;
import org.eclipse.core.databinding.property.INativePropertyListener;
import org.eclipse.core.databinding.property.ISimplePropertyListener;
import org.eclipse.core.databinding.property.SimplePropertyEvent;
import org.eclipse.core.databinding.property.list.SimpleListProperty;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.emf.commands.core.command.CompositeTransactionalCommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.DestroyElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.infra.tools.util.TypeUtils;
import org.eclipse.papyrusrt.umlrt.core.commands.ExclusionCommand;
import org.eclipse.papyrusrt.umlrt.tooling.ui.databinding.IFilteredObservableList;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTFactory;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTInheritanceKind;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTNamedElement;
import org.eclipse.papyrusrt.umlrt.uml.provider.UMLRTEditPlugin;
import org.eclipse.uml2.uml.NamedElement;

/**
 * A property for a {@link UMLRTNamedElement}'s inheritable list reference, supporting
 * filtering.
 */
class FacadeListProperty<S extends UMLRTNamedElement, T extends UMLRTNamedElement> extends SimpleListProperty<S, T> implements IFilteredListProperty<S, T>, IFacadeProperty {
	private final Class<T> elementType;
	private final EStructuralFeature facadeFeature;
	private final EStructuralFeature umlFeature;

	private List<T> listView;

	FacadeListProperty(Class<T> elementType, EStructuralFeature facadeFeature, EStructuralFeature umlFeature) {
		super();

		this.elementType = elementType;
		this.facadeFeature = facadeFeature;
		this.umlFeature = umlFeature;
	}

	@Override
	public Object getElementType() {
		return elementType;
	}

	@Override
	public String getPropertyName() {
		return UMLRTEditPlugin.INSTANCE.getString(String.format("_UI_%s_%s_feature",
				facadeFeature.getEContainingClass().getName(), facadeFeature.getName()));
	}

	@Override
	public IObservableList<T> observe(Realm realm, S source) {
		return IFilteredObservableList.wrap(super.observe(realm, source));
	}

	@Override
	protected List<T> doGetList(S source) {
		if (listView == null) {
			Stream<T> elements = this.<T> eList(source).stream();
			// Also excluded elements of the reference type
			Stream<T> excluded = source.getExcludedElements().stream()
					.filter(elementType::isInstance).map(elementType::cast);

			listView = Stream.concat(elements, excluded)
					.sorted(UMLRTInheritanceKind.facadeComparator())
					.collect(toList());
		}
		return listView;
	}

	protected <E> EList<E> eList(S source) {
		return eList((EObject) source, facadeFeature);
	}

	protected <E> EList<E> eList(NamedElement uml) {
		return eList(umlOwner(uml), umlFeature);
	}

	protected EObject umlOwner(NamedElement uml) {
		return uml;
	}

	@SuppressWarnings("unchecked")
	private <E> EList<E> eList(EObject source, EStructuralFeature feature) {
		return (EList<E>) source.eGet(feature);
	}

	@Override
	protected void doSetList(S source, List<T> list, ListDiff<T> diff) {
		List<? extends ICommand> commands = getCommands(source, diff);
		if (!commands.isEmpty()) {
			TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(source.toUML());
			CompositeTransactionalCommand command = new CompositeTransactionalCommand(
					domain, "Edit " + getPropertyName(), commands);
			if (command.canExecute()) {
				domain.getCommandStack().execute(GMFtoEMFCommandWrapper.wrap(command.reduce()));

				// Recalculate the content
				listView = null;
				doGetList(source);
			}
		}
	}

	@Override
	public INativePropertyListener<S> adaptListener(ISimplePropertyListener<S, ListDiff<T>> listener) {
		class NativeListener extends AdapterImpl implements INativePropertyListener<S> {

			private S source;

			@Override
			public void addTo(S source) {
				((Notifier) source).eAdapters().add(this);
				this.source = source;

				addToMembers(source);
			}

			@Override
			public void removeFrom(S source) {
				if (this.source == source) {
					removeFromMembers(source);
					((Notifier) source).eAdapters().remove(this);
					this.source = null;
				}
			}

			private void addToMembers(S source) {
				FacadeListProperty.this.<T> eList(source).forEach(this::addToElement);
			}

			private void removeFromMembers(S source) {
				FacadeListProperty.this.<T> eList(source).forEach(this::removeFromElement);
			}

			private void addToElement(T facade) {
				List<Adapter> adapters = facade.toUML().eAdapters();
				if (!adapters.contains(this)) {
					adapters.add(this);
				}
			}

			private void removeFromElement(T facade) {
				facade.toUML().eAdapters().remove(this);
			}

			@Override
			public void notifyChanged(Notification msg) {
				// Process removals by UML representation because after they have been removed
				// from the model, it may not be possible to obtain their façade
				List<T> added = Collections.emptyList();
				List<T> removed = Collections.emptyList();
				List<T> changed = Collections.emptyList();

				// If our list has never been accessed, then there's no point in any of it
				if ((listView != null) && !msg.isTouch()) {
					if ((msg.getNotifier() == source) && (msg.getFeature() == facadeFeature)) {
						switch (msg.getEventType()) {
						case Notification.ADD: {
							T newElement = elementType.cast(msg.getNewValue());
							// We need to listen to this new element for refresh
							addToElement(newElement);

							added = Collections.singletonList(newElement);
							break;
						}
						case Notification.ADD_MANY: {
							@SuppressWarnings("unchecked")
							List<T> newElements = (List<T>) msg.getNewValue();
							// We need to listen to these new elements for refresh
							newElements.forEach(this::addToElement);

							added = newElements;
							break;
						}
						case Notification.REMOVE: {
							T oldElement = elementType.cast(msg.getOldValue());
							// We need to stop listening to this old element for refresh
							removeFromElement(oldElement);

							removed = Collections.singletonList(oldElement);
							break;
						}
						case Notification.REMOVE_MANY: {
							@SuppressWarnings("unchecked")
							List<T> oldElements = (List<T>) msg.getOldValue();
							// We need to stop to listening to these old elements for refresh
							oldElements.forEach(this::removeFromElement);

							removed = oldElements;
							break;
						}
						case Notification.MOVE: {
							T movedElement = elementType.cast(msg.getNewValue());
							removed = Collections.singletonList(movedElement);
							added = Collections.singletonList(movedElement);
							break;
						}
						case Notification.SET:
						case Notification.RESOLVE: {
							T newElement = elementType.cast(msg.getNewValue());
							// We need to listen to this new element for refresh
							addToElement(newElement);
							T oldElement = elementType.cast(msg.getOldValue());
							// We need to stop listening to this old element for refresh
							removeFromElement(oldElement);

							removed = Collections.singletonList(oldElement);
							added = Collections.singletonList(newElement);
							break;
						}
						}
					} else if (msg.getNotifier() instanceof NamedElement) {
						// It came from an object in this list
						T notifier = TypeUtils.as(UMLRTFactory.create((NamedElement) msg.getNotifier()), elementType);

						if (notifier != null) {
							// UI presentation of the list may need to be refreshed
							changed = Collections.singletonList(notifier);
						}
					}
				}

				if (!added.isEmpty() || !removed.isEmpty()) {
					List<T> oldListView = new ArrayList<>(listView);

					// Rebuild the list view from the model
					listView = null;

					ListDiff<T> diff = Diffs.computeListDiff(oldListView, doGetList(source));

					if (!diff.isEmpty()) {
						listener.handleEvent(new SimplePropertyEvent<>(SimplePropertyEvent.CHANGE,
								source, FacadeListProperty.this, diff));
					}
				} else if (!changed.isEmpty()) {
					T changed_ = changed.get(0);
					int position = listView.indexOf(changed_);
					listener.handleEvent(new SimplePropertyEvent<>(SimplePropertyEvent.CHANGE,
							source, FacadeListProperty.this,
							createListDiff(createListDiffEntry(position, false, changed_),
									createListDiffEntry(position, true, changed_))));
				}
			}
		}

		return new NativeListener();
	}

	protected List<? extends ICommand> getCommands(S source, ListDiff<T> diff) {
		TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(source.toUML());
		List<ICommand> result = new ArrayList<>();

		// We cannot use GMF edit-helpers for these complex properties
		diff.accept(new ListDiffVisitor<T>() {

			@Override
			public void handleMove(int oldIndex, int newIndex, T element) {
				// Support for moves is very restrictive: the elements must be in the
				// same EMF containment list. That means that whatever the old index
				// is, the element that's there now would have to be in the same
				// containment list, so we can use it as a reference
				EObject elementAtOldPosition = listView.get(oldIndex).toUML();
				EObject elementAtNewPosition = listView.get(newIndex).toUML();
				EList<?> containment = (EList<?>) elementAtOldPosition.eContainer().eGet(elementAtOldPosition.eContainmentFeature());
				EList<?> newContainment = (EList<?>) elementAtNewPosition.eContainer().eGet(elementAtNewPosition.eContainmentFeature());
				if (containment == newContainment) {
					// Convert to this list's coordinates
					oldIndex = containment.indexOf(elementAtOldPosition);
					newIndex = containment.indexOf(elementAtNewPosition);
					if ((oldIndex >= 0) && (newIndex >= 0)) {
						class MoveCommand extends AbstractTransactionalCommand {
							private EList<?> list;
							private int oldIndex;
							private int newIndex;

							MoveCommand(TransactionalEditingDomain domain, String label, EList<?> list, int oldIndex, int newIndex) {
								super(domain, label, getWorkspaceFiles(source.toUML()));

								this.list = list;
								this.oldIndex = oldIndex;
								this.newIndex = newIndex;
							}

							@Override
							protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
								list.move(newIndex, oldIndex);
								return CommandResult.newOKCommandResult(element);
							}
						}

						result.add(new MoveCommand(domain, "Move " + getPropertyName(), containment, oldIndex, newIndex));
					}
				}
			}

			@Override
			public void handleRemove(int index, T element) {
				switch (element.getInheritanceKind()) {
				case INHERITED:
				case REDEFINED:
					// Exclude, don't destroy
					result.add(ExclusionCommand.getExclusionCommand(domain, element.toUML(), true));
					break;
				case NONE:
					// Actually destroy it
					IEditCommandRequest request = new DestroyElementRequest(domain, element.toUML(), false);
					IElementEditService edit = ElementEditServiceUtils.getCommandProvider(element.toUML());
					ICommand destroy = edit.getEditCommand(request);
					result.add(destroy);
					break;
				default:
					// Pass
					break;
				}
			}

			@Override
			public void handleAdd(int index, T element) {
				EList<T> list = eList(source);

				// We can't really honour the index. And we can't use an AddCommand because it doesn't
				// work with the ownedPort derived list implementation for undo/redo
				class AddCommand extends AbstractTransactionalCommand {
					private T element;

					AddCommand(TransactionalEditingDomain domain, String label, T element) {
						super(domain, label, getWorkspaceFiles(source.toUML()));

						this.element = element;
					}

					@Override
					protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
						if (!list.contains(element)) {
							eList(source.toUML()).add(element.toUML());
						}
						return CommandResult.newOKCommandResult(element);
					}
				}

				result.add(new AddCommand(domain, "Add " + getPropertyName(), element));
			}
		});

		return result;
	}
}
