/*****************************************************************************
 * Copyright (c) 2017 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.internal.editparts;

import static org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.editpolicies.AbstractInheritanceEditPolicy.INHERITANCE_ROLE;

import java.util.Optional;
import java.util.function.Predicate;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.diagram.ui.editpolicies.EditPolicyRoles;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.uml.diagram.statemachine.custom.edit.part.CustomRegionEditPart;
import org.eclipse.papyrusrt.umlrt.core.utils.Either;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.editpolicies.InheritanceEditPolicy;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.editpolicies.RTSemanticEditPolicy;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.utils.EditPartInheritanceUtils;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTFactory;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTNamedElement;
import org.eclipse.papyrusrt.umlrt.uml.util.UMLRTExtensionUtil;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Region;

/**
 * UML-RT implementation of the edit-part for inheritable {@link Region}s in state machines.
 */
public class RTRegionEditPart extends CustomRegionEditPart implements IStateMachineInheritableEditPart {

	/**
	 * Initializes me with my notation {@code view}.
	 *
	 * @param view
	 *            my notation view
	 */
	public RTRegionEditPart(View view) {
		super(view);
	}

	@Override
	protected void createDefaultEditPolicies() {
		// This needs to be ahead of the edit-policies that process user edit gestures
		installEditPolicy(INHERITANCE_ROLE, new InheritanceEditPolicy());

		super.createDefaultEditPolicies();

		installEditPolicy(EditPolicyRoles.SEMANTIC_ROLE, new RTSemanticEditPolicy());
	}

	@Override
	public EObject resolveSemanticElement() {
		return EditPartInheritanceUtils.resolveSemanticElement(this, super.resolveSemanticElement());
	}

	@Override
	public Object getAdapter(@SuppressWarnings("rawtypes") java.lang.Class key) {
		return EditPartInheritanceUtils.getAdapter(this, key, super.getAdapter(key));
	}

	@Override
	public boolean isSemanticInherited() {
		// Regions have no realization in the façade model apart from
		// their containing state or state machine
		EObject semantic = resolveSemanticElement();
		return (semantic instanceof Region) && UMLRTExtensionUtil.isInherited((Region) semantic);
	}

	/**
	 * As a region has no specific realization in the façade model apart from
	 * its containing state or state machine, use that as its UML-RT element.
	 *
	 * @return the façade of my region's containing state or state machine
	 */
	@Override
	public Optional<UMLRTNamedElement> getUMLRTElement() {
		Optional<UMLRTNamedElement> result = Optional.empty();

		EObject semantic = resolveSemanticElement();
		if (semantic instanceof Region) {
			Region region = (Region) semantic;
			result = Optional.ofNullable(Either.or(region.getStateMachine(), region.getState())
					.orElse(NamedElement.class, null))
					.map(UMLRTFactory::create);
		}

		return result;
	}

	@Override
	public UMLRTNamedElement getRedefinitionContext(UMLRTNamedElement element) {
		// My containing state machine or state is the redefinition context
		// for my diagram because it is the diagram's root element. So, likewise
		// for any ancestor definition that it (recursively) redefines
		Predicate<UMLRTNamedElement> same = element::equals;
		Predicate<UMLRTNamedElement> redefinition = e -> e.redefines(element);

		return getUMLRTElement() // See note on getUMLRTElement() method
				.filter(same.or(redefinition))
				.map(__ -> element)
				.orElseGet(() -> EditPartInheritanceUtils.getStateMachineContext(element).getEither(UMLRTNamedElement.class));
	}
}
