/*****************************************************************************
 * Copyright (c) 2015, 2017 CEA LIST, Christian W. Damus, and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   Christian W. Damus - bugs 490859, 492368, 496649, 467545, 510188
 *   
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.utils;

import static org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes.EXTERNAL_BEHAVIOR_PORT_ID;
import static org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes.INTERNAL_BEHAVIOR_PORT_ID;
import static org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes.RELAY_PORT_ID;
import static org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID;
import static org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.Map;

import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.IHintedType;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.core.types.UMLRTElementTypesEnumerator;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTPortKind;

/**
 * @deprecated Use the {@link UMLRTPortKind} enumeration type, instead.
 */
public enum RTPortKindEnum implements Comparable<RTPortKindEnum> {
	/**
	 * 
	 */
	EXTERNAL(0, "External Behavior", UMLRTPortKind.EXTERNAL_BEHAVIOR),

	/**
	 * 
	 */
	INTERNAL(1, "Internal Behavior", UMLRTPortKind.INTERNAL_BEHAVIOR),

	/**
	 *
	 */
	RELAY(2, "Relay", UMLRTPortKind.RELAY),

	/**
	 * 
	 */
	SAP(3, "SAP", UMLRTPortKind.SAP),

	/**
	 * 
	 */
	SPP(4, "SPP", UMLRTPortKind.SPP);

	private static final Map<UMLRTPortKind, RTPortKindEnum> FACADE_MAP = new EnumMap<>(UMLRTPortKind.class);

	private final String label;
	private final int index;
	private final UMLRTPortKind facadeKind;

	static {
		FACADE_MAP.put(UMLRTPortKind.NULL, null);
		Arrays.asList(values()).forEach(k -> FACADE_MAP.put(k.toFacade(), k));
	}

	public String getLabel() {

		return label;
	}

	public int getIndex() {

		return index;
	}



	private RTPortKindEnum(int index, String label, UMLRTPortKind facadeKind) {
		this.label = label;
		this.index = index;
		this.facadeKind = facadeKind;
	}

	@Override
	public String toString() {
		return label;
	}

	/**
	 * Queries the kind of <tt>RTPort</tt> indicated by an element-type, if it is some kind
	 * of <tt>RTPort</tt> type.
	 * 
	 * @param elementType
	 *            an element-type
	 * @return the corresponding port kind, or {@code null} if the element-type is not a
	 *         kind of <tt>RTPort</tt>
	 * 
	 * @deprecated Use the {@link RTPortUtils#getPortKind(IElementType)} API, instead.
	 */
	@Deprecated
	public static RTPortKindEnum of(IElementType elementType) {
		RTPortKindEnum result = null;

		if ((elementType != null) && ElementTypeUtils.isTypeCompatible(elementType, UMLRTElementTypesEnumerator.RT_PORT)) {
			// Which specific kind is it?
			if (ElementTypeUtils.isTypeCompatible(elementType, ElementTypeRegistry.getInstance().getType(INTERNAL_BEHAVIOR_PORT_ID))) {
				result = RTPortKindEnum.INTERNAL;
			} else if (ElementTypeUtils.isTypeCompatible(elementType, ElementTypeRegistry.getInstance().getType(EXTERNAL_BEHAVIOR_PORT_ID))) {
				result = RTPortKindEnum.EXTERNAL;
			} else if (ElementTypeUtils.isTypeCompatible(elementType, ElementTypeRegistry.getInstance().getType(SERVICE_PROVISION_POINT_ID))) {
				result = RTPortKindEnum.SPP;
			} else if (ElementTypeUtils.isTypeCompatible(elementType, ElementTypeRegistry.getInstance().getType(SERVICE_ACCESS_POINT_ID))) {
				result = RTPortKindEnum.SAP;
			} else if (ElementTypeUtils.isTypeCompatible(elementType, ElementTypeRegistry.getInstance().getType(RELAY_PORT_ID))) {
				result = RTPortKindEnum.RELAY;
			}
		}

		return result;
	}

	public static RTPortKindEnum fromFacade(UMLRTPortKind facadeKind) {
		return FACADE_MAP.get(facadeKind);
	}

	public UMLRTPortKind toFacade() {
		return facadeKind;
	}

	public static UMLRTPortKind toFacade(RTPortKindEnum kind) {
		return (kind == null) ? UMLRTPortKind.NULL : kind.toFacade();
	}

	/**
	 * Converts me to my corresponding {@linkplain IElementType element-type}.
	 * 
	 * @return my element-type
	 * 
	 * @deprecated Use the {@link RTPortUtils#getElementType(UMLRTPortKind)} API, instead.
	 */
	@Deprecated
	public IHintedType toElementType() {
		IElementType result;

		switch (this) {
		case EXTERNAL:
			result = ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.EXTERNAL_BEHAVIOR_PORT_ID);
			break;
		case INTERNAL:
			result = ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.INTERNAL_BEHAVIOR_PORT_ID);
			break;
		case RELAY:
			result = ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.RELAY_PORT_ID);
			break;
		case SAP:
			result = ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.SERVICE_ACCESS_POINT_ID);
			break;
		case SPP:
			result = ElementTypeRegistry.getInstance().getType(IUMLRTElementTypes.SERVICE_PROVISION_POINT_ID);
			break;
		default:
			result = UMLRTElementTypesEnumerator.RT_PORT;
			break;
		}

		// The registered port types are all hinted
		return (IHintedType) result;
	}

	/**
	 * Are ports of my kind external facing? These are visualized on the border of
	 * the capsule frame in its structure diagram.
	 * 
	 * @return whether ports of my kind are external
	 */
	public boolean isExternal() {
		boolean result;

		switch (this) {
		case EXTERNAL:
		case RELAY:
		case SPP:
			result = true;
			break;
		default:
			result = false;
			break;
		}
		return result;
	}


	/**
	 * Are ports of my kind internal facing? These are visualized inside
	 * the capsule frame in its structure diagram.
	 * 
	 * @return whether ports of my kind are internal
	 */
	public boolean isInternal() {
		boolean result;

		switch (this) {
		case INTERNAL:
		case SAP:
			result = true;
			break;
		default:
			result = false;
			break;
		}
		return result;
	}
}
