/*****************************************************************************************
 * Copyright (c) 2015, 2017 CEA LIST, Christian W. Damus, Zeligsoft (2009),  and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine Janssens (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *  Christian W. Damus - bugs 476984, 495823, 510315, 507282
 *  Young-Soo Roh - Refactored to common multi-reference table property editor. 
 *                  bugs 494291
 *  
 ****************************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.properties.widget;

import static java.lang.Math.abs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.coordinate.PositionCoordinate;
import org.eclipse.nebula.widgets.nattable.hideshow.RowHideShowLayer;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer.MoveDirectionEnum;
import org.eclipse.nebula.widgets.nattable.selection.command.MoveSelectionCommand;
import org.eclipse.papyrus.infra.emf.nattable.selection.EObjectSelectionExtractor;
import org.eclipse.papyrus.infra.nattable.layerstack.BodyLayerStack;
import org.eclipse.papyrus.infra.nattable.manager.axis.IAxisManager;
import org.eclipse.papyrus.infra.nattable.manager.axis.ICompositeAxisManager;
import org.eclipse.papyrus.infra.nattable.manager.table.INattableModelManager;
import org.eclipse.papyrus.infra.nattable.model.nattable.Table;
import org.eclipse.papyrus.infra.nattable.model.nattable.nattableaxis.IAxis;
import org.eclipse.papyrus.infra.nattable.model.nattable.nattableaxis.NattableaxisFactory;
import org.eclipse.papyrus.infra.nattable.provider.TableStructuredSelection;
import org.eclipse.papyrus.infra.nattable.utils.AxisUtils;
import org.eclipse.papyrus.infra.nattable.utils.NattableModelManagerFactory;
import org.eclipse.papyrus.infra.nattable.utils.TableSelectionWrapper;
import org.eclipse.papyrusrt.umlrt.tooling.properties.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.properties.editors.MultiReferenceControlEditor;
import org.eclipse.papyrusrt.umlrt.tooling.properties.providers.IRTTableSelectionProvider;
import org.eclipse.papyrusrt.umlrt.tooling.properties.providers.RTNattableSelectionService;
import org.eclipse.papyrusrt.umlrt.tooling.properties.providers.RTTableSelectionProvider;
import org.eclipse.papyrusrt.umlrt.tooling.tables.manager.axis.RTSynchronizedOnFeatureAxisManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;

/**
 * 
 * A PropertyEditor for editing multiple references in a List for UMLRT.
 *
 */
public class RTNatTableMultiReferencePropertyEditor extends RTNatTablePropertyEditor implements ISelectionChangedListener {

	/**
	 * Minimum Height of a row
	 */
	private static final int ROW_HEIGHT = 40;

	/**
	 * Selection provider
	 */
	private IRTTableSelectionProvider selectionProvider;

	/**
	 * Control Button Editor
	 */
	private MultiReferenceControlEditor editor;

	/**
	 * Style of the Table
	 */
	protected int style;

	protected String createElementTypeId;

	protected Class<? extends IControlManager> controlManagerClass;

	protected List<EObject> selectedElements = new ArrayList<>();

	/**
	 * @return the createElementTypeId
	 */
	public String getCreateElementTypeID() {
		return createElementTypeId;
	}

	/**
	 * @param createElementTypeId
	 *            the createElementTypeId to set
	 */
	public void setCreateElementTypeID(String createElementTypeId) {
		this.createElementTypeId = createElementTypeId;
		if (editor != null) {
			editor.setCreateElementTypeID(createElementTypeId);
		}
	}



	/**
	 * Constructor.
	 *
	 * @param parent
	 *            The parent composite.
	 * @param style
	 *            The style of the composite.
	 */
	public RTNatTableMultiReferencePropertyEditor(final Composite parent, final int style) {
		super(parent, style);

		this.style = style;
	}



	/**
	 * @see org.eclipse.papyrus.uml.properties.widgets.NattablePropertyEditor#createPreviousWidgets(org.eclipse.emf.ecore.EObject, org.eclipse.emf.ecore.EStructuralFeature)
	 *
	 * @param sourceElement
	 * @param feature
	 */
	@Override
	protected void createPreviousWidgets(EObject sourceElement, EStructuralFeature feature) {
		super.createPreviousWidgets(sourceElement, feature);

		// Create the Button Widget
		disposeControlEditor();
		editor = createControlEditor(sourceElement, feature);
		setEditor(editor);

	}

	@Override
	protected NatTable createNatTableWidget(INattableModelManager manager, Composite parent, int style, Collection<?> rows) {
		NatTable result = super.createNatTableWidget(manager, parent, style, rows);

		// Hook the selection listener to the table for updating of buttons etc.
		setSelectionProvider();

		self.addDisposeListener(e -> {
			disposeControlEditor();
			RTNattableSelectionService.getInstance().removeSelectionProvider(selectionProvider);
			selectionProvider.removeSelectionChangedListener(this);
			selectedElements.clear();
		});

		return result;
	}

	protected MultiReferenceControlEditor createControlEditor(EObject sourceElement, EStructuralFeature feature) {
		MultiReferenceControlEditor editor = new MultiReferenceControlEditor(self, style, sourceElement, feature, nattableManager);
		editor.setModelObservable(getInputObservableList());
		editor.setFactory(input.getValueFactory(propertyPath));
		editor.setCreateElementTypeID(createElementTypeId);
		editor.onElementMoved(this::refreshTable);
		editor.onElementRemoved(this::handleExclusion); // Tables don't know how to detect exclusion

		if (getControlManagerClass() != null) {
			hookControlManager(editor);
		}

		return editor;
	}

	/**
	 * Configure layout of the different Composite
	 * 
	 * @param sourceElement
	 *            the
	 */
	@Override
	protected void configureLayout(EObject sourceElement) {
		// parent composite
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.horizontalIndent = 0;
		setLayoutData(data);

		// table group layout
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		self.setLayout(layout);

		// button editor layout data
		GridData editorData = new GridData(SWT.RIGHT, SWT.TOP, false, false);
		editorData.horizontalIndent = 0;
		editor.setLayoutData(editorData);

		// Table Composite
		GridData tableData = new GridData(SWT.FILL, SWT.FILL, true, true);
		tableData.minimumHeight = getInputObservableList().size() * ROW_HEIGHT;
		tableData.horizontalIndent = 0;
		natTableWidget.setLayoutData(tableData);

		// Layout the different Widget
		self.layout();
		editor.layout();
		super.configureLayout(sourceElement);
	}


	/**
	 * Set the Selection Provider
	 */
	protected void setSelectionProvider() {
		selectionProvider = new RTTableSelectionProvider(nattableManager, nattableManager.getBodyLayerStack().getSelectionLayer());
		selectionProvider.addSelectionChangedListener(this);
		RTNattableSelectionService.getInstance().addSelectionProvider(selectionProvider);
	}

	/**
	 * Dispose the existing button
	 */
	protected void disposeControlEditor() {
		if (null != editor) {
			editor.dispose();
		}
	}

	/**
	 * Get the Table Manager
	 * 
	 * @param table
	 * @return
	 */
	public INattableModelManager getTableManager(final Table table) {
		return NattableModelManagerFactory.INSTANCE.createNatTableModelManager(table, new EObjectSelectionExtractor());
	}

	/**
	 * When the selection changed set the selected elements to the Button Editor.
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		if (self == null || self.isDisposed()) {
			// table is being disposed.
			return;
		}
		// List of the selected elements
		List elements = new ArrayList<>(1);

		// Get the selected cells coordinates
		TableStructuredSelection selection = (TableStructuredSelection) event.getSelection();
		TableSelectionWrapper wrapper = (TableSelectionWrapper) selection.getAdapter(TableSelectionWrapper.class);
		Collection<PositionCoordinate> selectedCells = wrapper.getSelectedCells();

		// The row position of the cell
		int rowPosition = 0;

		for (PositionCoordinate coordinate : selectedCells) {

			// get the real index of the row
			rowPosition = coordinate.getRowPosition();
			final RowHideShowLayer rowHideShowLayer = nattableManager.getBodyLayerStack().getRowHideShowLayer();
			int realRowIndex = rowHideShowLayer.getRowIndexByPosition(rowPosition);

			// get the associated element
			final Object rowElement = AxisUtils.getRepresentedElement(nattableManager.getRowElement(realRowIndex));

			// if it is a element not already selected add it into the list of elements
			if ((!elements.contains(rowElement))) {
				elements.add(rowElement);
			}
		}

		// Set the elements list to the button widget
		if (editor != null && !editor.isDisposed()) {
			editor.setSelectedElements(elements);
			editor.updateButtons();
		}

	}

	/**
	 * Initiates a refresh of the table for a move of the specified element.
	 * 
	 * @param element
	 *            a element that was moved
	 * @param oldIndex
	 *            the index in the containment list from which it was moved
	 */
	protected void refreshTable(Object element, int oldIndex) {
		natTableWidget.getDisplay().asyncExec(() -> {
			BodyLayerStack stack = nattableManager.getBodyLayerStack();
			SelectionLayer selectionLayer = stack.getSelectionLayer();

			int index = getInputObservableList().indexOf(element);

			// Follow the moved element with the selection

			// Its index in the selection layer
			int selectionIndex = selectionLayer.getRowPositionByIndex(index);
			if (selectionIndex >= 0) {
				int step = index - oldIndex;
				MoveSelectionCommand follow = new MoveSelectionCommand(
						(step > 0) ? MoveDirectionEnum.DOWN : MoveDirectionEnum.UP,
						abs(step), false, false);
				natTableWidget.doCommand(follow);
			}
		});
	}

	/**
	 * Ensures appropriate handling of exclusion of an element.
	 * 
	 * @param removedElement
	 *            a element that was deleted or excluded
	 */
	protected void handleExclusion(Object removedElement) {
		IAxisManager rowMgr = nattableManager.getRowAxisManager();
		if (rowMgr instanceof ICompositeAxisManager) {
			// Find the row axis manager via a fake axis, because the table configuration won't have one
			ICompositeAxisManager composite = (ICompositeAxisManager) rowMgr;
			IAxis axis = NattableaxisFactory.eINSTANCE.createEStructuralFeatureAxis();
			axis.setManager(nattableManager.getTable().getTableConfiguration().getRowHeaderAxisConfiguration()
					.getAxisManagers().get(0));

			IAxisManager nested = composite.getSubAxisManagerFor(axis);
			if (nested instanceof RTSynchronizedOnFeatureAxisManager) {
				rowMgr = nested;
			}
		}

		if (rowMgr instanceof RTSynchronizedOnFeatureAxisManager) {
			((RTSynchronizedOnFeatureAxisManager) rowMgr).handleExclusion(removedElement);
		}
	}

	/**
	 * Obtains my control manager class.
	 * 
	 * @return my controlManager class
	 */
	public Class<? extends IControlManager> getControlManagerClass() {
		return controlManagerClass;
	}

	/**
	 * Sets my control manager class.
	 * 
	 * @param controlManager
	 *            my control manager class
	 */
	public void setControlManagerClass(Class<? extends IControlManager> controlManager) {
		this.controlManagerClass = controlManager;

		if ((controlManager != null) && (editor != null)) {
			hookControlManager(editor);
		}
	}

	protected void hookControlManager(MultiReferenceControlEditor editor) {
		if ((getControlManagerClass() != null) && (editor != null)) {
			IControlManager controlManager;

			try {
				controlManager = getControlManagerClass().newInstance();
			} catch (Exception e) {
				Activator.log.error(e);
				return;
			}

			editor.enableAddElement(controlManager::canAddElement);
			editor.enableMoveElement(object -> controlManager.canMoveElement(
					nattableManager.getTable().getContext(), object));
			editor.enableRemoveElement(object -> controlManager.canRemoveElement(
					nattableManager.getTable().getContext(), object));

		}
	}

	//
	// Nested types
	//

	/**
	 * Protocol for management of the enablement of the row control buttons atop the table.
	 */
	public interface IControlManager {
		/**
		 * Queries whether the object being edit supports the addition of any (more)
		 * elements in the table.
		 * 
		 * @param editedParent
		 *            the parent element being edit, which owns the elements in the table
		 * 
		 * @return whether another element may be added
		 */
		default boolean canAddElement(Object editedParent) {
			return true;
		}

		/**
		 * Queries whether an {@code object} may be moved within the table for the given parent object.
		 * 
		 * @param editedParent
		 *            the parent element being edit, which owns the elements in the table
		 * @param object
		 *            an object to be moved
		 * 
		 * @return whether the {@code object} may be moved
		 */
		default boolean canMoveElement(Object editedParent, Object object) {
			return true;
		}

		/**
		 * Queries whether an {@code object} may be removed from the table for the given parent object.
		 * There is no distinction between exclusion and deletion.
		 * 
		 * @param editedParent
		 *            the parent element being edit, which owns the elements in the table
		 * @param object
		 *            an object to be removed
		 * 
		 * @return whether the {@code object} may be removed
		 */
		default boolean canRemoveElement(Object editedParent, Object object) {
			return true;
		}
	}
}
