/*******************************************************************************
 * Copyright (c) 2017 Zeligsoft (2009) Limited  and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/

package org.eclipse.papyrusrt.codegen.cpp.validation

import com.google.common.base.Strings
import java.util.List
import org.eclipse.core.runtime.IStatus
import org.eclipse.core.runtime.MultiStatus
import org.eclipse.core.runtime.Status
import org.eclipse.emf.ecore.EObject
import org.eclipse.papyrusrt.codegen.CodeGenPlugin
import org.eclipse.papyrusrt.codegen.cpp.CppCodeGenPlugin
import org.eclipse.papyrusrt.umlrt.uml.UMLRTGuard
import org.eclipse.papyrusrt.xtumlrt.trans.TransformValidator
import org.eclipse.papyrusrt.xtumlrt.util.DetailedException
import org.eclipse.uml2.uml.Constraint
import org.eclipse.uml2.uml.MultiplicityElement
import org.eclipse.uml2.uml.NamedElement
import org.eclipse.uml2.uml.State
import org.eclipse.uml2.uml.StateMachine
import org.eclipse.uml2.uml.Transition
import org.eclipse.uml2.uml.Trigger
import org.eclipse.papyrusrt.codegen.cpp.UMLPrettyPrinter
import static extension org.eclipse.papyrusrt.xtumlrt.util.ContainmentUtils.*
import static extension org.eclipse.papyrusrt.xtumlrt.external.predefined.UMLRTStateMachProfileUtil.*

/**
 * Pre UML2xtumlrt validation
 * @author ysroh
 */
class PreUML2xtumlrtValidator implements TransformValidator<List<EObject>> {

	extension UMLPrettyPrinter prettyPrinter = new UMLPrettyPrinter

	override MultiStatus validate(List<EObject> context) {
		val status = new MultiStatus(CodeGenPlugin.ID, IStatus.INFO, "UML-RT Code Generator Invoked", null)
		for (e : context) {
			e.eAllContents.forEach[validateElement(status); validateDuplicateElement(status)]
		}
		status
	}

	protected dispatch def void validateElement(EObject e, MultiStatus result) {
	}

	protected def dispatch void validateDuplicateElement(NamedElement element, MultiStatus result) {
		if(element.namespace !== null && !Strings.isNullOrEmpty(element.name)){
			val unique = element.namespace.ownedMembers.filter[e | e !== element && e.eClass === element.eClass && e.name == element.name].empty
			if(!unique){
				// should not allow two capsules with same name
				val status = createWarningStatus("More than one element named " + element.qualifiedName 
					+ " exist in the same namespace")
				result.add(status)
			}
		}
	}
	protected def dispatch void validateDuplicateElement(Object element, MultiStatus result) {
	
	}
	protected dispatch def void validateElement(Constraint o, MultiStatus result) {
		val guard = UMLRTGuard.getInstance(o)
		if (guard !== null && !guard.bodies.containsKey(CppCodeGenPlugin.LANGUAGE)) {
			var qualifiedName = o.validQualifiedName
			val status = createErrorStatus("Guard " + qualifiedName + " must have C++ specification")
			result.add(status)
		} 
	}

	protected dispatch def void validateElement(MultiplicityElement element, MultiStatus result) {
		val lower = element.lowerValue
		if (lower !== null) {
			lower.validateElement(result)
		}
		val upper = element.upperValue
		if (upper !== null) {
			upper.validateElement(result)
		}
	}

	/**
	 * Validates a state.
	 * 
	 * <p>Checks whether there are conflicting outgoing transitions from the state. 
	 *  
	 * @param state - A {@link State}.
	 * @param result - A {@link MultiStatus}.
	 */
	protected dispatch def void validateElement(State state, MultiStatus result) {
		val containingStates = 
			state.getAllOwningElementsUptoType(StateMachine)
				.filter[it instanceof State]
				.map[it as State]
		val allOutgoingTransitionsHierarchy = containingStates.map[allOutgoingTransitions].flatten
		for (transition : state.allOutgoingTransitions) {
			val otherEquivalentTransitions = allOutgoingTransitionsHierarchy.filter[conflict(it, transition)]
			if (!otherEquivalentTransitions.empty) {
				val qualifiedName = state.validQualifiedName
				val conflictingTransitions = otherEquivalentTransitions.multiLineListText
				val status = createWarningStatus("State " + qualifiedName 
					+ " has at least two conflicting (ambiguous) outgoing transitions with the same trigger and guard. \n"
					+ "Transition " + transition.text + "conflicts with the following:\n"
					+ conflictingTransitions + "\n"
					+ "The transition with the deepest source will be selected and the others ignored.\n"
					+ "If there is more than one such transitions any one of them will be selected and others will be ignored.\n"
					+ "Note that the transitions may have a different source, namely a composite state that contains " + qualifiedName + ".\n"
				)
				result.add(status)
				return
			}
		}
	}
	
	/**
	 * @param transition1 - A {@link Transition}.
	 * @param transition2 - A {@link Transition}.
	 * @return {@code true} iff the two transition conflict, i.e. if they are not the same but 
	 * they have a common trigger (cf. {@link #commonTrigger}) and the same guard (cf. {@link #sameGuard}).
	 * Note that this method already assumes that the source of one of the transitions is the same as or is
	 * contained in the source of the other transition.
	 */
	private def conflict(Transition transition1, Transition transition2) {
		transition1 !== transition2
		&& commonTrigger(transition1, transition2) 
		&& sameGuard(transition1, transition2)
	}
	
	/**
	 * @param transition1 - A {@link Transition}.
	 * @param transition2 - A {@link Transition}.
	 * @return {@code true} iff the two transition have equal guards or at least guards with equal specification.
	 */
	private def sameGuard(Transition transition1, Transition transition2) {
		transition1.guard == transition2.guard
		|| transition1.guard?.specification == transition2.guard?.specification
	}
	
	/**
	 * @param transition1 - A {@link Transition}.
	 * @param transition2 - A {@link Transition}.
	 * @return {@code true} iff the two transition have at least one equivalent trigger (cf. {@link #equivalentTrigger})
	 */
	private def commonTrigger(Transition transition1, Transition transition2)
	{
		transition1.triggers.exists[t1 | transition2.triggers.exists[t2 | equivalentTrigger(t1, t2)]]
	}
	
	/**
	 * @param trigger1 - A {@link Trigger}.
	 * @param trigger2 - A {@link Trigger}.
	 * @return {@code true} iff the two triggers are eequivalent, i.e., iff they are equal or 
	 * the have the same event and at least one port in common.
	 */
	private def equivalentTrigger(Trigger trigger1, Trigger trigger2) {
		trigger1 == trigger2
		|| trigger1.event == trigger2.event
			&& trigger1.ports.exists[trigger2.ports.contains(it)]
	}

	protected def Status createErrorStatus(String msg) {
		val exception = new DetailedException(msg)
		val status = new Status(IStatus.ERROR, CodeGenPlugin.ID, exception.message, exception)
		status
	}

	protected def Status createWarningStatus(String msg) {
		val exception = new DetailedException(msg)
		val status = new Status(IStatus.WARNING, CodeGenPlugin.ID, exception.message, exception)
		status
	}

	protected def String getValidQualifiedName(EObject context) {
		var qualifiedName = ""
		var container = context
		while (container !== null) {
			if (container instanceof NamedElement) {
				val name = (container as NamedElement).name
				if(qualifiedName != ""){
					qualifiedName = "::" + qualifiedName
				}
				if(Strings.isNullOrEmpty(name)){
					qualifiedName = "[" + container.eClass.name + "]" + qualifiedName
				}else{
					qualifiedName = name + qualifiedName
				}
			}
			container = container.eContainer
		}
		qualifiedName
	}

}
