/*****************************************************************************
 * Copyright (c) 2015, 2016 CEA LIST, Christian W. Damus, and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Celine Janssens (ALL4TEC) celine.janssens@all4tec.net  - Initial API and implementation
 *   Celine Janssens (ALL4TEC) celine.janssens@all4tec.net   - Bug  482694 : add listener on Multiplicity Bounds
 *   Christian W. Damus - bugs 489380, 483637, 496304
 *   
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.editparts;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.gmf.runtime.gef.ui.figures.NodeFigure;
import org.eclipse.gmf.runtime.notation.NotationPackage;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.infra.gmfdiag.common.figure.node.RoundedRectangleNodePlateFigure;
import org.eclipse.papyrus.uml.diagram.composite.custom.edit.parts.ResizablePortEditPart;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.figures.RTPortFigure;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.MultiplicityElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;

/**
 * Edit Part for Port in Papyrus RT.
 * 
 * @author Céline JANSSENS
 *
 */
public class RTPortEditPart extends ResizablePortEditPart implements IRTPortEditPart {

	/**
	 * 
	 */
	private static final int LAYER_OPACITY = 0;

	/**
	 * Horizontal Offset
	 */
	private static final int X_OFFSET = 2;

	/**
	 * Vertical Offset
	 */
	private static final int Y_OFFSET = -X_OFFSET;

	/**
	 * Layer Number
	 */
	private static final int LAYER_NUMBER = 1;

	Adapter multiplicityValueListener;

	/**
	 * 
	 * Constructor.
	 *
	 * @param view
	 *            The view created with the editPart
	 */
	public RTPortEditPart(View view) {
		super(view);


	}

	/**
	 * @see org.eclipse.gmf.runtime.diagram.ui.editparts.GraphicalEditPart#addSemanticListeners()
	 *
	 */
	@Override
	protected void addSemanticListeners() {

		super.addSemanticListeners();
		multiplicityValueListener = getMultiplicityListener();
		getUMLElement().eAdapters().add(multiplicityValueListener);

		if (null != ((MultiplicityElement) getUMLElement()).getLowerValue()) {
			((MultiplicityElement) getUMLElement()).getLowerValue().eAdapters().add(multiplicityValueListener);
		}

		if (null != ((MultiplicityElement) getUMLElement()).getUpperValue()) {
			((MultiplicityElement) getUMLElement()).getUpperValue().eAdapters().add(multiplicityValueListener);
		}


	}

	/**
	 * @see org.eclipse.gmf.runtime.diagram.ui.editparts.GraphicalEditPart#removeSemanticListeners()
	 *
	 */
	@Override
	protected void removeSemanticListeners() {
		super.removeSemanticListeners();

		Element element = getUMLElement();
		if (element != null) { // Would be null in view deletion scenario
			element.eAdapters().remove(multiplicityValueListener);
		}

		if (element instanceof MultiplicityElement) {
			MultiplicityElement mult = (MultiplicityElement) element;

			if (null != mult.getLowerValue()) {
				mult.getLowerValue().eAdapters().remove(multiplicityValueListener);
			}

			if (null != mult.getUpperValue()) {
				mult.getUpperValue().eAdapters().remove(multiplicityValueListener);
			}
			multiplicityValueListener = null;
		}
	}

	/**
	 * Retrieve the listener for Multiplicity Bounds
	 */
	protected Adapter getMultiplicityListener() {
		return new AdapterImpl() {

			@Override
			public void notifyChanged(Notification notification) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();

				if (getUMLElement() instanceof MultiplicityElement && notifier instanceof Port) {

					if ((notifier == getUMLElement()) && (feature == UMLPackage.eINSTANCE.getMultiplicityElement_LowerValue()) && (type == Notification.SET)) {
						ValueSpecification lowerValue = ((MultiplicityElement) notifier).getLowerValue();
						if (lowerValue != null) { // Maybe this is a deletion event
							lowerValue.eAdapters().add(multiplicityValueListener);
						}
					} else if ((notifier == getUMLElement()) && (feature == UMLPackage.eINSTANCE.getMultiplicityElement_UpperValue()) && (type == Notification.SET)) {
						ValueSpecification upperValue = ((MultiplicityElement) notifier).getUpperValue();
						if (upperValue != null) { // Maybe this is a deletion event
							upperValue.eAdapters().add(multiplicityValueListener);
						}
					}

				}
				if (notifier instanceof ValueSpecification && ((ValueSpecification) notifier).eContainer().equals(getUMLElement())) {

					refreshVisuals();
				}
			}
		};
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.uml.diagram.composite.edit.parts.PortEditPart#getPrimaryShape()
	 */
	@Override
	public RTPortFigure getPrimaryShape() {

		return (RTPortFigure) primaryShape;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.uml.diagram.composite.edit.parts.PortEditPart#createNodeShape()
	 * 
	 */
	@Override
	protected IFigure createNodeShape() {
		primaryShape = new RTPortFigure();
		return primaryShape;
	}

	/**
	 * Override the inherited method for a smaller default size.
	 */
	@Override
	protected NodeFigure createNodePlate() {
		Dimension defaultSize = getDefaultSize();

		RoundedRectangleNodePlateFigure figure = new RoundedRectangleNodePlateFigure(
				defaultSize);

		// Get dimension from notation
		int width = ((Integer) getStructuralFeatureValue(NotationPackage.eINSTANCE.getSize_Width())).intValue();
		int height = ((Integer) getStructuralFeatureValue(NotationPackage.eINSTANCE.getSize_Height())).intValue();

		int w = (width > defaultSize.width()) ? width : defaultSize.width();
		int h = (height > defaultSize.height()) ? height : defaultSize.height();

		figure.getBounds().setSize(w, h);
		figure.setDefaultSize(w, h);

		return figure;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.uml.diagram.composite.custom.edit.parts.CustomPortEditPart#refreshVisuals()
	 *
	 */
	@Override
	protected void refreshVisuals() {


		// The edit part is displayed as a stack only if there can be more than one Port
		boolean stack = ((MultiplicityElement) getUMLElement()).getUpper() > 1;
		getPrimaryShape().setStack(stack);
		getPrimaryShape().setLayerNumber(LAYER_NUMBER);
		getPrimaryShape().setXOffSet(X_OFFSET);
		getPrimaryShape().setYOffSet(Y_OFFSET);
		getPrimaryShape().setLayerOpacity(LAYER_OPACITY);

		super.refreshVisuals();


	}

	@Override
	public boolean isPortOnPart() {
		return false;
	}
}
