/*****************************************************************************
 * Copyright (c) 2016 CEA LIST, Christian W. Damus, and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   Christian W. Damus - bugs 472885, 483537
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.drop;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.papyrus.infra.gmfdiag.dnd.policy.CustomizableDropEditPolicy;
import org.eclipse.papyrus.infra.gmfdiag.dnd.strategy.DropStrategy;
import org.eclipse.papyrus.uml.diagram.composite.custom.edit.policies.CustomDiagramDragDropEditPolicy;

/**
 * Edit policy to override {@link CustomDiagramDragDropEditPolicy} for Capsule structure compartment edit part.
 * Workaround for the Bug 472885: [tooling] Papyrus-RT shall support CapsulePart creation using drag n drop from model explorer
 */
public class RTCapsuleStructureDiagramDragDropEditPolicy extends CustomizableDropEditPolicy {

	/** key for the specific edit policy that creates a property while dragging a classifier onto the structure compartment of a classifier. */
	private static final String ORG_ECLIPSE_PAPYRUS_UML_DIAGRAM_DND_CLASSIFIER_TO_STRUCTURE_COMP_AS_PROPERTY_DROP = "org.eclipse.papyrus.uml.diagram.dnd.ClassifierToStructureCompAsPropertyDrop";

	/** Prefix matching strategy IDs that we prefer over other contributions. */
	private static final String PREFERRED_STRATEGY_PREFIX = "org.eclipse.papyrusrt."; //$NON-NLS-1$

	/**
	 * Constructor.
	 *
	 * @param defaultDropEditPolicy
	 * @param defaultCreationEditPolicy
	 */
	public RTCapsuleStructureDiagramDragDropEditPolicy(EditPolicy defaultDropEditPolicy, EditPolicy defaultCreationEditPolicy) {
		super(new FilteredDropEditPolicy(defaultDropEditPolicy), defaultCreationEditPolicy);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Map<DropStrategy, List<Command>> findExtendedStrategies(Request request) {
		Map<DropStrategy, List<Command>> result = super.findExtendedStrategies(request);

		// Only filter out the strategies that we don't want if there are strategies
		// in play that we prefer. This ensures that we don't prevent the correct
		// drop behaviour in a delegation scenario, such as when a protocol-as-port
		// drop converts the protocol drop into the dropping of a port, for which we
		// don't have a preferred strategy
		if (hasPreferredDropStrategies(result.keySet())) {
			Map<DropStrategy, List<Command>> filteredMap = new HashMap<>(result);
			for (DropStrategy strategy : result.keySet()) {
				if (ORG_ECLIPSE_PAPYRUS_UML_DIAGRAM_DND_CLASSIFIER_TO_STRUCTURE_COMP_AS_PROPERTY_DROP.equals(strategy.getID())) {
					filteredMap.remove(strategy);
				} else if ("default".equals(strategy.getID())) {
					filteredMap.remove(strategy);
				}
			}
			result = filteredMap;
		}

		return result;
	}

	/**
	 * Determines whether a group of drop strategies includes any that are preferred
	 * over others. These are typically those that are contributed by the UML-RT
	 * bundles.
	 * 
	 * @param strategies
	 *            a group of contributed drop strategies
	 * @return whether any of them are preferred over others
	 */
	protected boolean hasPreferredDropStrategies(Collection<? extends DropStrategy> strategies) {
		return strategies.stream()
				.map(DropStrategy::getID)
				.anyMatch(id -> id.startsWith(PREFERRED_STRATEGY_PREFIX));
	}
}
