/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.utils;

import static java.lang.Math.rint;

import org.eclipse.draw2d.geometry.Point;

/**
 * An iterator of positions marching along a straight path within the interior of
 * a shape.
 */
public class LinearIterator implements ILocationIterator {
	private final double unitX;
	private final double unitY;
	private Point current;

	private LinearIterator(double angle, Point start) {
		super();

		this.unitX = Math.cos(angle);
		this.unitY = Math.sin(angle);
		this.current = start.getCopy();
	}

	/**
	 * Creates a straight path-iterator that starts at the given location on an edit-part
	 * and marches in the direction given by the specified angle, in radians.
	 * Because the Y coordinate in a diagram is the negative of the mathematical
	 * convention, so does this angle measure clockwise instead of anticlockwise.
	 * 
	 * @param startingAt
	 *            the point from which to start marching
	 * @param direction
	 *            the angular direction, in radians, in which to march
	 * 
	 * @return an iterator of positions from the starting point
	 */
	public static LinearIterator from(Point startingAt, double direction) {
		return new LinearIterator(direction, startingAt);
	}

	/**
	 * Creates a straight path-iterator that starts at the given location on an edit-part
	 * and marches in a due southeast direction.
	 * 
	 * @param startingAt
	 *            the point from which to start marching
	 * 
	 * @return an iterator of positions from the starting point
	 */
	public static LinearIterator from(Point startingAt) {
		return from(startingAt, Math.PI / 4.0);
	}

	@Override
	public Point next(int distance) {
		current = current.getTranslated(rint(unitX * distance), rint(unitY * distance));
		return where();
	}

	@Override
	public Point where() {
		return current;
	}
}
