/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.internal.editpolicies;

import static org.eclipse.papyrusrt.umlrt.tooling.diagram.common.utils.UMLRTStateMachineDiagramUtils.getStateMachineDiagram;

import java.util.List;
import java.util.function.Supplier;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.diagram.core.services.ViewService;
import org.eclipse.gmf.runtime.diagram.core.util.ViewUtil;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.notation.Bounds;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.NotationPackage;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.papyrus.commands.wrappers.GMFtoGEFCommandWrapper;
import org.eclipse.papyrus.infra.gmfdiag.common.adapter.SemanticAdapter;
import org.eclipse.papyrus.infra.gmfdiag.hyperlink.editpolicies.NavigationEditPolicy;
import org.eclipse.papyrus.uml.diagram.statemachine.edit.parts.PseudostateEntryPointEditPart;
import org.eclipse.papyrus.uml.diagram.statemachine.edit.parts.PseudostateExitPointEditPart;
import org.eclipse.papyrus.uml.diagram.statemachine.edit.parts.StateEditPartTN;
import org.eclipse.papyrus.uml.diagram.statemachine.part.UMLDiagramEditorPlugin;
import org.eclipse.papyrusrt.umlrt.common.ui.preferences.DialogPreferences;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.utils.RelativePortLocation;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.internal.commands.CreateNestedStateMachineDiagramCommand;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.internal.commands.RerouteTransitionsToConnectionPointsCommand;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.internal.editparts.IRTPseudostateEditPart;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.PseudostateKind;
import org.eclipse.uml2.uml.State;

/**
 * Custom navigation edit-policy for state in RT state machine diagrams, which
 * implements default double-click handling to convert a state into a composite
 * state with a nested diagram.
 */
public class RTCustomStateNavigationEditPolicy extends NavigationEditPolicy {
	private static final String CONFIRMATION_DIALOG_ID = Activator.PLUGIN_ID + ".confirmCreateNestedSMD"; //$NON-NLS-1$

	/**
	 * Initializes me.
	 */
	public RTCustomStateNavigationEditPolicy() {
		super();
	}

	/**
	 * Override the default behaviour in the case that a state does not yet have
	 * a nested state machine diagram, to always create that diagram and open it.
	 * Thereafter, delegate to the default behaviour because the initial diagram will
	 * initially be a default navigation but the user can customize it.
	 */
	@Override
	protected Command getOpenCommand(Request request) {
		Command result;

		Diagram stateMachineDiagram = getStateMachineDiagram(getState());
		if (stateMachineDiagram != null) {
			// Just do the usual navigation
			result = super.getOpenCommand(request);
		} else {
			switch (promptToCreateNestedDiagram()) {
			case IDialogConstants.NO_ID:
				// User wants the usual navigation
				result = super.getOpenCommand(request);
				break;
			case IDialogConstants.YES_ID:
				ICommand composed;
				ICommand createDiagram = new CreateNestedStateMachineDiagramCommand(
						getEditingDomain(), getState(), getGraphicalHost());
				Supplier<Diagram> newDiagram = () -> (Diagram) createDiagram.getCommandResult().getReturnValue();
				composed = createDiagram;

				ICommand rerouteTransitions = RerouteTransitionsToConnectionPointsCommand.createRerouteTransitionsCommand(
						getEditingDomain(), getState(), getGraphicalHost());
				if (rerouteTransitions != null) {
					composed = composed.compose(rerouteTransitions);
				}

				// A command to align connection points around the new diagram frame as
				// around the state
				composed = composed.compose(getConnectionPointMatchingCommand(newDiagram));

				result = GMFtoGEFCommandWrapper.wrap(composed);
				break;
			default: // CANCEL_ID
				result = null;
				break;
			}
		}

		return result;
	}

	/**
	 * Prompts the user to confirm creation of a nested diagram.
	 * 
	 * @return {@link IDialogConstants#YES_ID} if the user assents to the nested diagram,
	 *         {@link IDialogConstants#NO_ID} if the user wants the default double-click action,
	 *         or {@link IDialogConstants#CANCEL_ID} if the user wants to cancel the double-click
	 */
	int promptToCreateNestedDiagram() {
		Shell parentShell = getHost().getViewer().getControl().getShell();
		return DialogPreferences.yesNoCancel(parentShell,
				"Convert to Composite State",
				"Convert the state to a composite state and open a new diagram on its contents?",
				CONFIRMATION_DIALOG_ID);
	}

	IGraphicalEditPart getGraphicalHost() {
		return (IGraphicalEditPart) getHost();
	}

	TransactionalEditingDomain getEditingDomain() {
		return getGraphicalHost().getEditingDomain();
	}

	State getState() {
		return (State) getGraphicalHost().resolveSemanticElement();
	}

	private ICommand getConnectionPointMatchingCommand(Supplier<? extends Diagram> newDiagram) {
		return new AbstractTransactionalCommand(getEditingDomain(), "Arrange Connection Points", null) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
				Diagram smd = newDiagram.get();
				View frame = (smd == null) ? null : ViewUtil.getChildBySemanticHint(smd, StateEditPartTN.VISUAL_ID);
				if (frame instanceof Node) {
					// This is a new diagram and we haven't yet opened it, so it
					// will not yet have views for the connection points.
					// Create them now
					Rectangle frameRect = getBounds((Node) frame);
					IGraphicalEditPart stateEP = getGraphicalHost();
					Rectangle stateRect = getBounds(stateEP);
					View stateView = stateEP.getNotationView();

					for (Pseudostate connPt : getState().getConnectionPoints()) {
						// We reverse the roles of "capsule-part" and "capsule" in our
						// case because we synchronize from state shape to frame
						Rectangle connPtRect = getBounds(findChildNode(stateView, connPt));
						RelativePortLocation loc = RelativePortLocation.of(connPtRect, stateRect);

						String semanticHint = (connPt.getKind() == PseudostateKind.ENTRY_POINT_LITERAL)
								? PseudostateEntryPointEditPart.VISUAL_ID
								: PseudostateExitPointEditPart.VISUAL_ID;
						Node onFrame = ViewService.getInstance().createNode(new SemanticAdapter(connPt, null),
								frame,
								semanticHint,
								ViewUtil.APPEND,
								UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);

						// Move the new shape
						Point moveTo = loc.applyTo(frameRect, IRTPseudostateEditPart.getDefaultSize(connPt.getKind()));
						ViewUtil.setStructuralFeatureValue(onFrame, NotationPackage.Literals.LOCATION__X, moveTo.x());
						ViewUtil.setStructuralFeatureValue(onFrame, NotationPackage.Literals.LOCATION__Y, moveTo.y());
					}
				}

				return CommandResult.newOKCommandResult();
			}
		};
	}

	Rectangle getBounds(IGraphicalEditPart editPart) {
		return editPart.getFigure().getBounds().getCopy();
	}

	Node findChildNode(View parent, EObject element) {
		return ((List<?>) parent.getChildren()).stream()
				.filter(Node.class::isInstance).map(Node.class::cast)
				.filter(n -> n.getElement() == element)
				.findFirst().get(); // We know a priori that it exists
	}

	Rectangle getBounds(Node node) {
		Bounds result = (Bounds) node.getLayoutConstraint();
		return new Rectangle(result.getX(), result.getY(), result.getWidth(), result.getHeight());
	}

}
