/*****************************************************************************
 * Copyright (c) 2015, 2016 CEA LIST, Christian W. Damus, and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine Janssens (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *  Christian W. Damus - bugs 476984, 495823
 *
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.properties.widget;

import static java.lang.Math.abs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.config.CellConfigAttributes;
import org.eclipse.nebula.widgets.nattable.config.IConfigRegistry;
import org.eclipse.nebula.widgets.nattable.coordinate.PositionCoordinate;
import org.eclipse.nebula.widgets.nattable.hideshow.RowHideShowLayer;
import org.eclipse.nebula.widgets.nattable.layer.cell.ILayerCell;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer.MoveDirectionEnum;
import org.eclipse.nebula.widgets.nattable.selection.command.MoveSelectionCommand;
import org.eclipse.nebula.widgets.nattable.selection.command.SelectCellCommand;
import org.eclipse.papyrus.infra.emf.nattable.selection.EObjectSelectionExtractor;
import org.eclipse.papyrus.infra.nattable.layerstack.BodyLayerStack;
import org.eclipse.papyrus.infra.nattable.manager.table.INattableModelManager;
import org.eclipse.papyrus.infra.nattable.model.nattable.Table;
import org.eclipse.papyrus.infra.nattable.painter.CustomizedCellPainter;
import org.eclipse.papyrus.infra.nattable.provider.TableSelectionProvider;
import org.eclipse.papyrus.infra.nattable.provider.TableStructuredSelection;
import org.eclipse.papyrus.infra.nattable.utils.AxisUtils;
import org.eclipse.papyrus.infra.nattable.utils.NattableModelManagerFactory;
import org.eclipse.papyrus.infra.nattable.utils.TableSelectionWrapper;
import org.eclipse.papyrusrt.umlrt.tooling.properties.editors.ParameterControlEditor;
import org.eclipse.papyrusrt.umlrt.tooling.ui.Messages;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.UMLPackage;

import com.google.common.base.Strings;

public class OwnedParameterOperationPropertyEditor extends RTNatTablePropertyEditor implements ISelectionChangedListener {

	/**
	 * Minimum Height of a row
	 */
	private static final int ROW_HEIGHT = 40;

	/**
	 * Selection provider
	 */
	private TableSelectionProvider selectionProvider;

	/**
	 * Style of the Table
	 */
	private int style;

	/**
	 * Control Button Widget
	 */
	private ParameterControlWidget buttonWidget;

	/**
	 * Nattable Dispose Listener
	 */
	private DisposeListener nattableDisposeListener = null;

	/**
	 * Constructor.
	 *
	 * @param parent
	 *            The parent composite.
	 * @param style
	 *            The style of the composite.
	 */
	public OwnedParameterOperationPropertyEditor(final Composite parent, final int style) {
		super(parent, style);

		this.style = style;
	}

	/**
	 * @see org.eclipse.papyrus.uml.properties.widgets.NattablePropertyEditor#createPreviousWidgets(org.eclipse.emf.ecore.EObject, org.eclipse.emf.ecore.EStructuralFeature)
	 *
	 * @param sourceElement
	 * @param feature
	 */
	@Override
	protected void createPreviousWidgets(EObject sourceElement, EStructuralFeature feature) {
		super.createPreviousWidgets(sourceElement, feature);

		// Create the Button Widget
		disposeButton();
		buttonWidget = createControlButton(self, style, sourceElement, nattableManager);
	}

	@Override
	protected NatTable createNatTableWidget(INattableModelManager manager, Composite parent, int style, Collection<?> rows) {
		NatTable result = super.createNatTableWidget(manager, parent, style, rows);

		// Override the cell painter for display of null types
		result.getConfigRegistry().registerConfigAttribute(CellConfigAttributes.CELL_PAINTER, new CustomizedCellPainter() {
			@Override
			protected String convertDataType(ILayerCell cell, IConfigRegistry configRegistry) {
				String result = super.convertDataType(cell, configRegistry);

				if (Strings.isNullOrEmpty(result) && isTypeColumn(cell.getColumnIndex())) {
					// It's an empty type cell
					result = Messages.NoTypeForTypedElement_Label;
				}

				return result;
			}
		});

		// Hook the selection listener to the table for updating of buttons etc.
		setSelectionProvider();

		return result;
	}

	/**
	 * Configure layout of the different Composite
	 * 
	 * @param sourceElement
	 *            the
	 */
	@Override
	protected void configureLayout(EObject sourceElement) {
		// parent composite
		setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
		GridLayout layout = new GridLayout();
		// The nesting composite provides margins
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		self.setLayout(layout);

		// Buttons Composite
		buttonWidget.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, false));

		// Table Composite
		GridData tableData = new GridData(SWT.FILL, SWT.FILL, true, true);
		tableData.minimumHeight = (((Operation) sourceElement).getOwnedParameters().size() * ROW_HEIGHT);
		natTableWidget.setLayoutData(tableData);

		// Layout the different Widget
		self.layout();
		buttonWidget.getEditor().layout();

		super.configureLayout(sourceElement);
	}


	/**
	 * Set the Selection Provider
	 */
	protected void setSelectionProvider() {
		selectionProvider = new TableSelectionProvider(nattableManager, nattableManager.getBodyLayerStack().getSelectionLayer());
		selectionProvider.addSelectionChangedListener(this);
	}

	/**
	 * Dispose the existing button
	 */
	protected void disposeButton() {
		if (null != buttonWidget) {
			buttonWidget.getEditor().dispose();
		}
	}

	/**
	 * Get the Table Manager
	 * 
	 * @param table
	 * @return
	 */
	public INattableModelManager getTableManager(final Table table) {
		return NattableModelManagerFactory.INSTANCE.createNatTableModelManager(table, new EObjectSelectionExtractor());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.uml.properties.widgets.NattablePropertyEditor#getDisposeListener()
	 * 
	 */
	@Override
	protected DisposeListener getDisposeListener() {
		if (null == nattableDisposeListener) {
			nattableDisposeListener = new DisposeListener() {

				@Override
				public void widgetDisposed(DisposeEvent e) {
					nattableManager.dispose();
					natTableWidget.dispose();
					buttonWidget.getEditor().dispose();
				}
			};
		}
		return nattableDisposeListener;
	}

	/**
	 * Create the Control button widget to manipulate the Owned Parameters
	 * 
	 * @param parent
	 *            The Composite parent
	 * @param style
	 *            the style
	 * @param sourceElement
	 *            The source element of the table
	 * @param nattableManager
	 *            The Table Manager
	 * @return the created widget
	 */
	protected ParameterControlWidget createControlButton(final Composite parent, final int style, EObject sourceElement, INattableModelManager nattableManager) {
		ParameterControlWidget widget = null;
		if (sourceElement instanceof Operation) {
			widget = new ParameterControlWidget(parent, style, (Operation) sourceElement, nattableManager);
			ParameterControlEditor editor = (ParameterControlEditor) widget.getEditor();
			editor.onParameterAdded(this::selectParameterType);
			editor.onParameterMoved(this::refreshParameter);
		}

		return widget;
	}

	/**
	 * When the selection changed set the selected Parameters to the Button Widget.
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		// List of the selected parameters
		List<Parameter> parameters = new ArrayList<>(1);

		// Get the selected cells coordinates
		TableStructuredSelection selection = (TableStructuredSelection) event.getSelection();
		TableSelectionWrapper wrapper = (TableSelectionWrapper) selection.getAdapter(TableSelectionWrapper.class);
		Collection<PositionCoordinate> selectedCells = wrapper.getSelectedCells();

		// The row position of the cell
		int rowPosition = 0;

		for (PositionCoordinate coordinate : selectedCells) {

			// get the real index of the row
			rowPosition = coordinate.getRowPosition();
			final RowHideShowLayer rowHideShowLayer = nattableManager.getBodyLayerStack().getRowHideShowLayer();
			int realRowIndex = rowHideShowLayer.getRowIndexByPosition(rowPosition);

			// get the associated element
			final Object rowElement = AxisUtils.getRepresentedElement(nattableManager.getRowElement(realRowIndex));

			// if it is a Parameter not already selected add it into the list of parameters
			if ((rowElement instanceof Parameter) && (!parameters.contains(rowElement))) {
				parameters.add((Parameter) rowElement);
			}
		}

		// Set the parameters list to the button widget
		buttonWidget.setParameters(parameters);
		((ParameterControlEditor) buttonWidget.getEditor()).updateButtons();


	}

	/**
	 * Selects the specified parameter.
	 * 
	 * @param parameter
	 *            a parameter of the operation
	 */
	protected void selectParameterType(Parameter parameter) {
		natTableWidget.getDisplay().asyncExec(() -> {
			BodyLayerStack stack = nattableManager.getBodyLayerStack();
			SelectionLayer selectionLayer = stack.getSelectionLayer();

			// Select the new parameter's type cell
			int type = getTypeColumnIndex();
			int index = parameter.getOperation().getOwnedParameters().indexOf(parameter);
			// Its index in the selection layer
			int selectionIndex = selectionLayer.getRowPositionByIndex(index);
			if (selectionIndex >= 0) {
				SelectCellCommand select = new SelectCellCommand(selectionLayer,
						type, selectionIndex, false, false);
				select.setForcingEntireCellIntoViewport(true);
				natTableWidget.doCommand(select);
			}
		});
	}

	/**
	 * Initiates a refresh of the table for a move of the specified parameter.
	 * 
	 * @param parameter
	 *            a parameter of the operation
	 * @param oldIndex
	 *            the index in the operation's parameter list from which it was moved
	 */
	protected void refreshParameter(Parameter parameter, int oldIndex) {
		natTableWidget.getDisplay().asyncExec(() -> {
			BodyLayerStack stack = nattableManager.getBodyLayerStack();
			SelectionLayer selectionLayer = stack.getSelectionLayer();

			int index = parameter.getOperation().getOwnedParameters().indexOf(parameter);

			// Follow the moved parameter with the selection

			// Its index in the selection layer
			int selectionIndex = selectionLayer.getRowPositionByIndex(index);
			if (selectionIndex >= 0) {
				int step = index - oldIndex;
				MoveSelectionCommand follow = new MoveSelectionCommand(
						(step > 0) ? MoveDirectionEnum.DOWN : MoveDirectionEnum.UP,
						abs(step), false, false);
				natTableWidget.doCommand(follow);
			}
		});
	}

	private boolean isTypeColumn(int columnIndex) {
		Object element = AxisUtils.getRepresentedElement(nattableManager.getColumnElement(columnIndex));
		return element == UMLPackage.Literals.TYPED_ELEMENT__TYPE;
	}

	private int getTypeColumnIndex() {
		int result = -1;
		int count = nattableManager.getColumnCount();

		for (int i = 0; (result < 0) && (i < count); i++) {
			if (isTypeColumn(i)) {
				result = i;
			}
		}

		return result;
	}
}
