/*****************************************************************************
 * Copyright (c) 2016 Zeligsoft (2009) Limited and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Young-Soo Roh - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.internal.advice;

import java.util.Collections;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateRelationshipRequest;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.types.IUMLRTElementTypes;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTConnector;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.UMLPackage;


/**
 * Edit-helper advice that checks for {@link RTConnector} ends compatibility.
 *
 */
public class ConnectorEndPortsCompatibilityAdvice extends AbstractEditHelperAdvice {

	/**
	 * @see org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice#getBeforeCreateRelationshipCommand(org.eclipse.gmf.runtime.emf.type.core.requests.CreateRelationshipRequest)
	 *
	 * @param request
	 * @return
	 */
	@Override
	protected ICommand getBeforeCreateRelationshipCommand(CreateRelationshipRequest request) {
		return new AbstractTransactionalCommand(request.getEditingDomain(), "Check Connection", Collections.EMPTY_LIST) {

			@Override
			protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {

				// Only care about connectors
				if (request.getContainmentFeature() != UMLPackage.Literals.STRUCTURED_CLASSIFIER__OWNED_CONNECTOR) {
					return CommandResult.newOKCommandResult();
				}

				if (!(request.getSource() instanceof Port) || !(request.getTarget() instanceof Port)) {
					return CommandResult.newOKCommandResult();
				}

				boolean canConnect = true;

				Port port1 = (Port) request.getSource();
				Port port2 = (Port) request.getTarget();

				// check port type compatibility.
				if (port1.getType() == null || !port1.getType().equals(port2.getType())) {
					canConnect = false;
				}

				if (canConnect) {
					View port1View = (View) request.getParameter(org.eclipse.papyrus.infra.services.edit.utils.RequestParameterConstants.EDGE_CREATE_REQUEST_TARGET_VIEW);
					View port2View = (View) request.getParameter(org.eclipse.papyrus.infra.services.edit.utils.RequestParameterConstants.EDGE_CREATE_REQUEST_SOURCE_VIEW);
					if (port2View != null && port2View.eContainer() != null
							&& port1View != null && port1View.eContainer() != null) {
						EObject port1Container = ((View) port2View.eContainer()).getElement();
						EObject port2Container = ((View) port1View.eContainer()).getElement();


						// Connection from or to port of the class should be delegation unless the port is internal behaviour port
						boolean isDelegation = (ElementTypeUtils.matches(port1Container, IUMLRTElementTypes.CAPSULE_ID) && port1.isService())
								|| (ElementTypeUtils.matches(port2Container, IUMLRTElementTypes.CAPSULE_ID) && port2.isService());

						if (!isDelegation) {
							if (port1.isConjugated() == port2.isConjugated()) {
								// assembly connector must have different conjugation
								canConnect = false;
							}
						} else {
							if (port1.isConjugated() != port2.isConjugated()) {
								// delegation connector must have same conjugation
								canConnect = false;
							}
						}
					}
				}

				if (!canConnect) {
					Shell shell = Display.getCurrent().getActiveShell();
					if (shell != null) {
						// ask user if still want to connect
						boolean result = MessageDialog.openConfirm(shell, "Incompatible Ports",
								"The two ports are not compatible. Do you still want to create a connector?");
						if (!result) {
							return CommandResult.newCancelledCommandResult();
						}
					}
				}
				return CommandResult.newOKCommandResult();
			}
		};
	}
}
