/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.internal.modelelement.properties;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Predicate;

import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.papyrus.infra.properties.ui.creation.CreationContext;
import org.eclipse.papyrus.uml.properties.creation.UMLPropertyEditorFactory;
import org.eclipse.papyrusrt.umlrt.tooling.ui.databinding.IFilteredObservableList;

/**
 * <p>
 * A specialized property-editor factory that ensures that new elements
 * are (temporarily) attached to the model while being edited in the creation
 * dialog, so that
 * </p>
 * <ul>
 * <li>the RT profile context is available to support RT editing
 * semantics</li>
 * <li>the transactional editing domain is available to support
 * tables and the edit service</li>
 * </ul>
 */
public class RTPropertyEditorFactory extends UMLPropertyEditorFactory {

	private final IObservableList<?> modelProperty;

	private final Map<EObject, Predicate<Object>> modelPropertyFilters = new HashMap<>();

	/**
	 * Initializes me with the reference that I edit.
	 * 
	 * @param referenceIn
	 *            my reference
	 */
	public RTPropertyEditorFactory(EReference referenceIn) {
		this(referenceIn, null);
	}

	/**
	 * Initializes me with the reference that I edit and the observable list that presents
	 * it in the UI. This is useful with a {@linkplain IFilteredObservableList filtered list},
	 * for example, to filter out of the UI presentation an object that is in the process of
	 * being created, while it is being edited in a dialog.
	 *
	 * @param referenceIn
	 *            my reference
	 * @param modelProperty
	 *            the model property that I am editing. May be {@code null}
	 */
	public RTPropertyEditorFactory(EReference referenceIn, IObservableList<?> modelProperty) {
		super(referenceIn);

		this.modelProperty = modelProperty;
	}

	/**
	 * Provides a customized creation context that sneakily (without EMF notifications)
	 * attaches created model elements to the model for the duration of the dialog.
	 */
	@Override
	protected CreationContext getCreationContext(Object element) {
		CreationContext result = super.getCreationContext(element);

		return (result == null)
				? result
				: wrapCreationContext(result);
	}

	private CreationContext wrapCreationContext(CreationContext context) {
		return new AttachedCreationContext(context);
	}

	CreateIn getCreateIn(CreationContext context, EObject newElement) {
		CreateIn result = null;

		if (referenceIn.isContainment()) {
			// Implicit create-in case
			result = new CreateIn() {
				// Pass
			};
			result.createInReference = referenceIn;
			result.createInObject = (EObject) context.getCreationContextElement();
		} else {
			result = createIn.get(newElement);
		}

		return result;
	}

	void filter(EObject createdObject) {
		IFilteredObservableList<?> list = IFilteredObservableList.getFilteredList(modelProperty);
		if (list != null) {
			list.addFilter(getFilter(createdObject));
		}
	}

	private Predicate<Object> getFilter(EObject createdObject) {
		return modelPropertyFilters.computeIfAbsent(createdObject,
				v -> (o -> o != v));
	}

	void unfilter(EObject createdObject) {
		IFilteredObservableList<?> list = IFilteredObservableList.getFilteredList(modelProperty);
		if (list != null) {
			list.removeFilter(modelPropertyFilters.remove(createdObject));
		}
	}

	//
	// Nested types
	//

	/**
	 * A decorating {@link CreationContext} that attaches the newly created
	 * element temporarily while it is being edited in the dialog.
	 */
	class AttachedCreationContext implements CreationContext {
		private final CreationContext delegate;

		AttachedCreationContext(CreationContext delegate) {
			super();

			this.delegate = delegate;
		}

		/**
		 * Attaches the {@code newElement} to its intended container in
		 * the model so that it may be correctly configured by the dialog.
		 */
		@Override
		public void pushCreatedElement(Object newElement) {
			if (newElement instanceof EObject) {
				EObject created = (EObject) newElement;
				CreateIn createIn = getCreateIn(this, created);
				if (createIn != null) {
					// Don't let the model property see this new element. We don't
					// want it to show in the UI temporarily
					filter(created);
					eAdd(createIn.createInObject, createIn.createInReference, created);
				}
			}

			delegate.pushCreatedElement(newElement);
		}

		@SuppressWarnings("unchecked")
		private void eAdd(EObject owner, EStructuralFeature feature, Object value) {
			if (feature.isMany()) {
				((EList<Object>) owner.eGet(feature)).add(value);
			} else {
				owner.eSet(feature, value);
			}
		}

		/**
		 * Detaches the {@code newElement} from its interim container so
		 * that it may later be added if the user completes the dialog normally.
		 */
		@Override
		public void popCreatedElement(Object newElement) {
			delegate.popCreatedElement(newElement);

			if (newElement instanceof EObject) {
				EObject created = (EObject) newElement;
				CreateIn createIn = getCreateIn(this, created);
				if (createIn != null) {
					eRemove(createIn.createInObject, createIn.createInReference, created);

					// Remove the UI filter from the model property
					unfilter(created);
				}
			}
		}

		@SuppressWarnings("unchecked")
		private void eRemove(EObject owner, EStructuralFeature feature, Object value) {
			if (feature.isMany()) {
				((EList<Object>) owner.eGet(feature)).remove(value);
			} else if (owner.eGet(feature) == value) {
				owner.eUnset(feature);
			}
		}

		@Override
		public EObject getCreationContextElement() {
			return (EObject) delegate.getCreationContextElement();
		}
	}

}
