/*****************************************************************************
 * Copyright (c) 2010, 2016 CEA LIST, Christian W. Damus, and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * 		Celine JANSSENS (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 * 		Christian W. Damus - bug 492408
 *
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.modelelement;

import java.lang.reflect.InvocationTargetException;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.core.databinding.observable.ChangeEvent;
import org.eclipse.core.databinding.observable.IChangeListener;
import org.eclipse.core.databinding.observable.IObservable;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.papyrus.infra.tools.util.PlatformHelper;
import org.eclipse.papyrus.infra.widgets.providers.IStaticContentProvider;
import org.eclipse.papyrus.infra.widgets.providers.StaticContentProvider;
import org.eclipse.papyrus.uml.properties.modelelement.StereotypeModelElement;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsulePartKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.IRealTimeConstants;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.CapsulePart;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.PortRegistrationType;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.RTPort;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.UMLRealTimePackage;
import org.eclipse.papyrusrt.umlrt.tooling.ui.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.ui.internal.modelelement.ChangeListenerUtils;
import org.eclipse.papyrusrt.umlrt.tooling.ui.widgets.CapsulePartKindObservableValue;
import org.eclipse.papyrusrt.umlrt.tooling.ui.widgets.PortRTKindObservableValue;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Stereotype;

/**
 * Model Element for Real Time Stereotype properties.
 * Used for RTPort properties like "isWired", "isPublish", "isNotification"
 * Used for CapsulePart
 * 
 * @author Céline JANSSENS
 *
 */
public class RTStereotypeModelElement extends StereotypeModelElement {

	/** label provider for RTPort::Registration kind */
	protected static final ILabelProvider registrationLabelProvider = new RegistrationLabelProvider();

	public RTStereotypeModelElement(final EObject stereoApplication, final Stereotype stereotype, final EditingDomain domain) {
		super(stereoApplication, stereotype, domain);
	}

	@Override
	protected boolean isFeatureEditable(final String propertyPath) {
		boolean editable = true;
		EStructuralFeature feature = getFeature(propertyPath);

		EObject source = getSource();
		if (source instanceof RTPort) {
			Port port = ((RTPort) source).getBase_Port();
			if (null != port) {
				RTPort rtPort = (RTPort) source;
				if (IRealTimeConstants.KIND.equals(propertyPath)) {
					editable = true;
				} else if (UMLRealTimePackage.eINSTANCE.getRTPort_IsNotification().equals(feature)) {
					editable = port.isBehavior(); // isNotification disabled if !isBehavior
				} else if (UMLRealTimePackage.eINSTANCE.getRTPort_IsWired().equals(feature)) {
					editable = port.isBehavior() && !RTPortUtils.isConnected(port) && !RTPortUtils.isLegacySpp(port) && !RTPortUtils.isLegacySap(port); // isWired disabled if connected || !isBehavior || isLegacySpp || isLegacySap
				} else if (UMLRealTimePackage.eINSTANCE.getRTPort_IsPublish().equals(feature)) {
					editable = false; // isPublish always disabled
				} else if (UMLRealTimePackage.eINSTANCE.getRTPort_Registration().equals(feature)) {
					editable = !rtPort.isWired(); // registration kind disabled if isWired
				} else if (UMLRealTimePackage.eINSTANCE.getRTPort_RegistrationOverride().equals(feature)) {
					editable = !rtPort.isWired(); // registration override disabled if isWired
				} else {
					editable = super.isFeatureEditable(propertyPath);
				}
			}
		} else if (source instanceof CapsulePart) {
			editable = true;
		} else {
			editable = super.isFeatureEditable(propertyPath);
		}
		return editable;

	}


	@Override
	public IObservable doGetObservable(final String propertyPath) {
		IObservable observe = null;

		// Case of Kind value in the Port RT Property View
		if (IRealTimeConstants.KIND.equals(propertyPath)) {
			EObject source = getSource();
			if (source instanceof RTPort) {
				observe = new PortRTKindObservableValue(source, (TransactionalEditingDomain) domain);
			} else if (source instanceof CapsulePart) {
				observe = new CapsulePartKindObservableValue(source, (TransactionalEditingDomain) domain);
			}

		} else {
			observe = super.doGetObservable(propertyPath);
		}
		if (getSource() instanceof RTPort) {
			observe.addChangeListener(new RTStereotypeModelElementChangeListener());
		}
		return observe;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IStaticContentProvider getContentProvider(String propertyPath) {
		IStaticContentProvider provider = null;
		EObject source = getSource();
		if (IRealTimeConstants.KIND.equals(propertyPath) && source instanceof RTPort) {
			Map<Object, String> map = new LinkedHashMap<>(RTPortKindEnum.values().length);

			map.put(RTPortKindEnum.EXTERNAL, RTPortKindEnum.EXTERNAL.getLabel());
			map.put(RTPortKindEnum.INTERNAL, RTPortKindEnum.INTERNAL.getLabel());
			map.put(RTPortKindEnum.RELAY, RTPortKindEnum.RELAY.getLabel());
			map.put(RTPortKindEnum.SAP, RTPortKindEnum.SAP.getLabel());
			map.put(RTPortKindEnum.SPP, RTPortKindEnum.SPP.getLabel());

			provider = new StaticContentProvider(map.keySet().toArray());

		} else if (IRealTimeConstants.KIND.equals(propertyPath) && source instanceof CapsulePart) {
			Map<Object, String> map = new LinkedHashMap<>(CapsulePartKindEnum.values().length);

			map.put(CapsulePartKindEnum.FIXED, CapsulePartKindEnum.FIXED.getLabel());
			map.put(CapsulePartKindEnum.OPTIONAL, CapsulePartKindEnum.OPTIONAL.getLabel());
			map.put(CapsulePartKindEnum.PLUGIN, CapsulePartKindEnum.PLUGIN.getLabel());

			provider = new StaticContentProvider(map.keySet().toArray());

		} else {
			provider = super.getContentProvider(propertyPath);
		}
		return provider;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ILabelProvider getLabelProvider(String propertyPath) {
		if ("registration".equals(propertyPath) && source instanceof RTPort) {
			return registrationLabelProvider;
		}
		return super.getLabelProvider(propertyPath);
	}

	@Override
	public boolean forceRefresh(String propertyPath) {
		return true;
	}

	/**
	 * Queries whether my data-source encapsulates an object that is not
	 * deleted.
	 * 
	 * @return whether the model element or diagram view that I encapsulate is still viable
	 */
	boolean isViable() {
		boolean result;

		Object selected = dataSource.getSelection().getFirstElement();
		if (selected == null) {
			// Obvious case
			result = false;
		} else {
			// If it's an edit-part, it's not viable if its view is detached
			View view = PlatformHelper.getAdapter(selected, View.class);
			result = (view == null) || (view.eResource() != null);

			if (result) {
				// Not an edit-part or attached. Check the semantic element
				EObject source = getSource();
				result = (source != null) && (source.eResource() != null);
			}
		}

		return result;
	}

	public class RTStereotypeModelElementChangeListener implements IChangeListener {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void handleChange(ChangeEvent event) {
			// Don't notify for a UML element or notation view that was deleted
			if (isViable()) {
				try {
					ChangeListenerUtils.fireDataSourceChanged(dataSource);
				} catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException | RuntimeException e) {
					Activator.log.debug(e.getMessage());
				}
			}
		}
	}

	public static class RegistrationLabelProvider extends LabelProvider {
		@Override
		public String getText(Object element) {
			if (element instanceof PortRegistrationType) {
				switch ((PortRegistrationType) element) {
				case APPLICATION:
					return "Application";
				case AUTOMATIC:
					return "Automatic";
				case AUTOMATIC_LOCKED:
					return "Automatic (locked)";
				default:
					return super.getText(element);
				}
			}

			return super.getText(element);
		}
	}

}
