/*****************************************************************************
 * Copyright (c) 2015, 2016 CEA LIST, Christian W. Damus, and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine JANSSENS (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *  Christian W. Damus - bug 489434
 *  
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.IObserving;
import org.eclipse.core.databinding.observable.value.ValueDiff;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsulePartKindEnum;
import org.eclipse.papyrusrt.umlrt.core.utils.MultipleAdapter;
import org.eclipse.papyrusrt.umlrt.profile.UMLRealTime.CapsulePart;
import org.eclipse.uml2.uml.AggregationKind;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralUnlimitedNatural;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * The Class CapsulePartKindObservableValue.
 */
public class CapsulePartKindObservableValue extends PapyrusObservableValue implements IObserving {

	/** The Constant LABEL_MULTIPLICITY. */
	private static final String LABEL_MULTIPLICITY = "Set Multiplicity";

	/** The Constant LABEL_AGGREGATION. */
	private static final String LABEL_AGGREGATION = "Set Aggregation";

	/** The capsule part element. */
	private Property capsulePartElement;

	/** The aggregation Enum value. */
	private AggregationKind aggregation;

	/** The aggregation Enum value. */
	private LiteralInteger lowerValue;

	/** The aggregation Enum value. */
	private LiteralUnlimitedNatural upperValue;

	private PartAdapter partAdapter;

	/**
	 * Instantiates a new capsule part kind observable value.
	 *
	 * @param RTApplication
	 *            the RT application
	 * @param domain
	 *            the domain
	 */
	public CapsulePartKindObservableValue(final EObject RTApplication, final TransactionalEditingDomain domain) {

		super(UMLUtil.getBaseElement(RTApplication), UMLUtil.getBaseElement(RTApplication).eContainingFeature(), domain);

		if (RTApplication instanceof CapsulePart) {
			setUMLPropertiesValue();

			capsulePartElement = (Property) UMLUtil.getBaseElement(RTApplication);

			// add listeners
			capsulePartElement.eAdapters().add(getPartAdapter());

			capsulePartElement.getLowerValue().eAdapters().add(getPartAdapter());
			capsulePartElement.getUpperValue().eAdapters().add(getPartAdapter());

		}

	}

	@Override
	public synchronized void dispose() {
		try {
			if (partAdapter != null) {
				partAdapter.dispose();
				partAdapter = null;
			}
		} finally {
			super.dispose();
		}
	}

	/**
	 * Sets the uml properties value.
	 */
	protected void setUMLPropertiesValue() {
		if (capsulePartElement instanceof Property) {
			// Set fields value
			aggregation = capsulePartElement.getAggregation();
			lowerValue = (LiteralInteger) capsulePartElement.getLowerValue();
			upperValue = (LiteralUnlimitedNatural) capsulePartElement.getUpperValue();
		}
	}

	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#doGetValue()
	 *
	 * @return
	 */
	@Override
	protected Object doGetValue() {
		CapsulePartKindEnum kind = null;
		setUMLPropertiesValue();

		// Define what is the kind of the capsulePart based on the multiplicity and the aggregation value
		if (0 < capsulePartElement.getLower()) {
			if (AggregationKind.COMPOSITE_LITERAL.equals(aggregation)) {
				kind = CapsulePartKindEnum.FIXED;
			}
		} else if (0 == capsulePartElement.getLower()) {
			if (AggregationKind.COMPOSITE_LITERAL.equals(aggregation)) {
				kind = CapsulePartKindEnum.OPTIONAL;
			} else if (AggregationKind.SHARED_LITERAL.equals(aggregation)) {
				kind = CapsulePartKindEnum.PLUGIN;
			}
		}
		return kind;
	}



	/**
	 * @see org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue#getCommand(java.lang.Object)
	 *
	 * @param value
	 * @return
	 */
	@Override
	public Command getCommand(final Object value) {
		setUMLPropertiesValue();
		// Set the new value based on the kind set
		LiteralInteger newLowerValue = UMLFactory.eINSTANCE.createLiteralInteger();
		LiteralUnlimitedNatural newUpperValue = UMLFactory.eINSTANCE.createLiteralUnlimitedNatural();
		AggregationKind newAggregation = AggregationKind.COMPOSITE_LITERAL;
		newUpperValue.setValue(upperValue.getValue());


		if (value instanceof CapsulePartKindEnum) {
			// For Each Kind of Capsule Part, set the property accordingly
			switch ((CapsulePartKindEnum) value) {
			case FIXED:
				newLowerValue.setValue(upperValue.getValue());
				newAggregation = AggregationKind.COMPOSITE_LITERAL;
				break;
			case OPTIONAL:
				newLowerValue.setValue(0);
				newAggregation = AggregationKind.COMPOSITE_LITERAL;
				break;
			case PLUGIN:
				newLowerValue.setValue(0);
				newAggregation = AggregationKind.SHARED_LITERAL;
				break;
			default:
				lowerValue.setValue(upperValue.getValue());
				newAggregation = AggregationKind.COMPOSITE_LITERAL;
				break;
			}
		}


		Command command = getCommandForCapsulePart(newLowerValue, newAggregation);

		return command;
	}

	/**
	 * Gets the command for capsule part.
	 * 
	 * @param newLowerValue
	 * @param newAggregation
	 * 
	 * @return the command for capsule part
	 */
	protected Command getCommandForCapsulePart(final ValueSpecification newLowerValue, final AggregationKind newAggregation) {
		CompoundCommand command = new CompoundCommand();

		command.append(getSetAggregationCommand(newAggregation));
		command.append(getSetMultiplicityCommand(newLowerValue));

		return command;
	}



	/**
	 * Gets the sets the multiplicity command.
	 * 
	 * @param newLowerValue
	 * 
	 * @return the sets the multiplicity command
	 */
	protected Command getSetMultiplicityCommand(final ValueSpecification newLowerValue) {

		RecordingCommand multiplicityCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				capsulePartElement.setLowerValue(newLowerValue);

			}

			@Override
			public boolean canExecute() {
				return true;
			}

			@Override
			public String getLabel() {
				return LABEL_MULTIPLICITY;
			}
		};

		return multiplicityCommand;
	}




	/**
	 * Gets the sets the aggregation command.
	 * 
	 * @param newAggregation
	 * 
	 * @return the sets the aggregation command
	 */
	protected Command getSetAggregationCommand(final AggregationKind newAggregation) {
		RecordingCommand aggregationCommand = new RecordingCommand((TransactionalEditingDomain) domain) {
			@Override
			protected void doExecute() {
				capsulePartElement.setAggregation(newAggregation);

			}

			@Override
			public boolean canExecute() {
				return true;
			}

			@Override
			public String getLabel() {
				return LABEL_AGGREGATION;
			}
		};

		return aggregationCommand;
	}

	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#getObserved()
	 * 
	 */
	@Override
	public Object getObserved() {
		return capsulePartElement;
	}


	/**
	 * @see org.eclipse.emf.databinding.EObjectObservableValue#getValueType()
	 * 
	 */
	@Override
	public Object getValueType() {
		return CapsulePartKindEnum.class;
	}

	/**
	 * Retrieve the listener for Multiplicity Bounds
	 */
	protected Adapter getMultiplicityListener() {
		// For API compatibility, continue providing this, but now
		// it's the same listener reacting to all changes
		return getPartAdapter();
	}

	/**
	 * Retrieve the listener for Aggregation
	 */
	protected Adapter getAggregationListener() {
		// For API compatibility, continue providing this, but now
		// it's the same listener reacting to all changes
		return getPartAdapter();
	}

	private Adapter getPartAdapter() {
		if (partAdapter == null) {
			partAdapter = new PartAdapter();
		}
		return partAdapter;
	}

	//
	// Nested types
	//

	private class PartAdapter extends MultipleAdapter {
		PartAdapter() {
			super(4);
		}

		@Override
		public void notifyChanged(Notification notification) {
			if (!notification.isTouch()) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();

				if ((notifier == capsulePartElement) && (type == Notification.SET)) {
					if (feature == UMLPackage.Literals.PROPERTY__AGGREGATION) {
						fireDiff(notification);
					} else if ((feature == UMLPackage.Literals.MULTIPLICITY_ELEMENT__LOWER_VALUE)
							|| (feature == UMLPackage.Literals.MULTIPLICITY_ELEMENT__UPPER_VALUE)) {
						// Switch listener to the new one
						if (notification.getOldValue() != null) {
							((ValueSpecification) notification.getOldValue()).eAdapters().remove(this);
						}
						if (notification.getNewValue() != null) {
							((ValueSpecification) notification.getNewValue()).eAdapters().add(this);
						}
						fireDiff(notification);
					}
				} else if ((notifier == capsulePartElement.getLowerValue()) && (type == Notification.SET)) {
					fireDiff(notification);
				} else if ((notifier == capsulePartElement.getUpperValue()) && (type == Notification.SET)) {
					fireDiff(notification);
				}
			}
		}

		private void fireDiff(Notification notification) {
			final ValueDiff diff = Diffs.createValueDiff(notification.getOldValue(), notification.getNewValue());
			getRealm().exec(new Runnable() {
				@Override
				public void run() {
					fireValueChange(diff);
				}
			});
		}

	}
}




