/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.diagram.statemachine.internal.commands;

import java.util.Arrays;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.infra.core.sashwindows.di.service.IPageManager;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.infra.gmfdiag.hyperlink.helper.EditorHyperLinkHelper;
import org.eclipse.papyrusrt.umlrt.core.utils.RegionUtils;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.utils.UMLRTStateMachineDiagramUtils;
import org.eclipse.uml2.uml.State;

/**
 * Create a nested state machine diagram for a state (thereby converting it, if necessary,
 * to a composite state) with default double-click navigation to the new diagram.
 */
public class CreateNestedStateMachineDiagramCommand extends AbstractTransactionalCommand {

	private final State state;
	private final IGraphicalEditPart stateEditPart;

	public CreateNestedStateMachineDiagramCommand(TransactionalEditingDomain domain, State state, IGraphicalEditPart stateEditPart) {
		super(domain, "Create State Machine Diagram", getWorkspaceFiles(Arrays.asList(state, stateEditPart.getNotationView())));

		this.state = state;
		this.stateEditPart = stateEditPart;
	}

	@Override
	protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
		View stateView = stateEditPart.getNotationView();

		// Let the diagram label be computed
		Diagram result = UMLRTStateMachineDiagramUtils.createStateMachineDiagram(state, null);

		if (result != null) {
			// Make the region an RTRegion
			state.getRegions().forEach(RegionUtils::applyStereotype);

			// And open the new diagram
			try {
				IPageManager pages = ServiceUtilsForEObject.getInstance().getIPageManager(result);
				pages.openPage(result);
			} catch (ServiceException e) {
				return CommandResult.newErrorCommandResult(e);
			}
		}

		// Create the default hyperlink
		EditorHyperLinkHelper helper = new EditorHyperLinkHelper();
		Command hyperlink = helper.getCreateHyperlinkCommand(getEditingDomain(), stateView, result);
		if ((hyperlink != null) && hyperlink.canExecute()) {
			hyperlink.execute();
		}

		return (result == null)
				? CommandResult.newCancelledCommandResult()
				: CommandResult.newOKCommandResult(result);
	}

}
