/*******************************************************************************
* Copyright (c) 2014-2015 Zeligsoft (2009) Limited  and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*******************************************************************************/
package org.eclipse.papyrusrt.xtumlrt.trans.preproc

import org.eclipse.papyrusrt.xtumlrt.trans.Transformation
import org.eclipse.papyrusrt.xtumlrt.common.Model
import org.eclipse.papyrusrt.xtumlrt.trans.TransformationContext
import org.eclipse.papyrusrt.xtumlrt.common.Entity
import org.eclipse.papyrusrt.xtumlrt.common.Package
import org.eclipse.papyrusrt.xtumlrt.common.Protocol
import org.eclipse.papyrusrt.xtumlrt.common.Capsule
import org.eclipse.papyrusrt.xtumlrt.common.MultiplicityElement
import org.eclipse.papyrusrt.xtumlrt.common.Port
import org.eclipse.papyrusrt.xtumlrt.common.CapsulePart
import org.eclipse.papyrusrt.xtumlrt.common.CapsuleKind
import org.eclipse.papyrusrt.xtumlrt.common.StructuredType
import org.eclipse.papyrusrt.xtumlrt.statemach.StateMachine
import org.eclipse.papyrusrt.xtumlrt.common.Signal
import org.eclipse.papyrusrt.xtumlrt.common.Attribute
import org.eclipse.papyrusrt.xtumlrt.common.Operation
import org.eclipse.papyrusrt.xtumlrt.common.Parameter
import org.eclipse.papyrusrt.xtumlrt.common.Behaviour

class StructureDefaultsPreprocessor implements Transformation<Model, Model>
{
    static val behaviourPreprocessor = new StateMachineDefaultsPreprocessor
    
    override transform( Model input, TransformationContext context )
    {
        visit( input )
        return input    // This is an in-place transformation
    }

    dispatch def void visit( Model model )
    {
        model.entities.forEach[ visit ]
        model.protocols.forEach[ visit ]
        model.packages.forEach[ visit ]
    }

    dispatch def void visit( Entity entity )
    {
        visitEntity( entity )
    }

    protected def void visitStructuredType( StructuredType structuredType )
    {
        structuredType.attributes.forEach[ visit ]
        structuredType.operations.forEach[ visit ]
    }
    
    protected def void visitEntity( Entity entity )
    {
        visitStructuredType( entity )
        if (entity.behaviour !== null)
            visit( entity.behaviour )
    }
    
    dispatch def void visit( Capsule capsule )
    {
        visitEntity( capsule )
        capsule.ports.forEach[ visit ]
        capsule.parts.forEach[ visit ]
    }
    
    dispatch def void visit( Protocol protocol )
    {
        protocol.protocolBehaviourFeatures.forEach[ visit ]
    }
    
    dispatch def void visit( Signal signal )
    {
        signal.parameters.forEach[ visit ]
    }
    
    dispatch def void visit( Package packge )
    {
        packge.entities.forEach[ visit ]
        packge.protocols.forEach[ visit ]
        packge.packages.forEach[ visit ]
    }
    
    dispatch def void visit( Port port )
    {
        visitMultiplicityElement( port )
    }
    
    dispatch def void visit( CapsulePart part )
    {
        visitMultiplicityElement( part )
        if (part.kind == CapsuleKind.OPTIONAL || part.kind == CapsuleKind.PLUGIN)
        {
            part.lowerBound = "0"
        }
    }
    
    dispatch def void visit( Attribute attribute )
    {
        visitMultiplicityElement( attribute )
    }
    
    dispatch def void visit( Operation operation )
    {
        operation.parameters.forEach[ visit ]
    }
    
    dispatch def void visit( Parameter parameter )
    {
        visitMultiplicityElement( parameter )
    }
    
    protected def void visitMultiplicityElement( MultiplicityElement element )
    {
        val lowerBound = element.lowerBound
        val upperBound = element.upperBound
        // We make lower and upper bounds the same, as they should represent the replication value of the port
        if (lowerBound === null && upperBound === null)
        {
            element.lowerBound = "1"
            element.upperBound = "1"
        }
        else if (lowerBound === null && upperBound !== null)
        {
            element.lowerBound = element.upperBound
        }
        else if (lowerBound !== null && upperBound === null)
        {
            element.upperBound = element.lowerBound
        }
        else
        {
            element.lowerBound = element.upperBound
        }
    }

    dispatch def void visit( Behaviour behaviour )
    {
    }

    dispatch def void visit( StateMachine machine )
    {
        behaviourPreprocessor.transform( machine, null )
    }

}