/*****************************************************************************
 * Copyright (c) 2014, 2017 CEA LIST, Christian W. Damus, and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  CEA LIST - Initial API and implementation
 *  Christian W. Damus - bugs 493866, 510315
 *  Young-Soo Roh - bug 510024
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.core.types.advice;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gmf.runtime.common.core.command.CompositeCommand;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.MoveRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.papyrus.uml.service.types.element.UMLElementTypes;
import org.eclipse.papyrusrt.umlrt.core.commands.ExcludeDependentsRequest;
import org.eclipse.papyrusrt.umlrt.core.types.ElementTypeUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.RegionUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.StateMachineUtils;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTNamedElement;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTState;
import org.eclipse.papyrusrt.umlrt.uml.UMLRTTransition;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * Edit Helper Advice for State machines.
 */
public class RTStateEditHelperAdvice extends AbstractEditHelperAdvice implements IInheritanceEditHelperAdvice {

	/**
	 * Constructor.
	 */
	public RTStateEditHelperAdvice() {
		// empty
	}

	/**
	 * Dispatches to specific methods, or super otherwise.
	 */
	@Override
	public boolean approveRequest(IEditCommandRequest request) {
		Boolean result = null;

		if (request instanceof CreateElementRequest) {
			result = approveCreateElementRequest((CreateElementRequest) request);
		} else if (request instanceof MoveRequest) {
			result = approveMoveRequest((MoveRequest) request);
		} else if (request instanceof SetRequest) {
			result = approveSetRequest((SetRequest) request);
		}

		if (result == null) {
			result = super.approveRequest(request);
		}

		return result;
	}

	/**
	 * Approves the specified set request.
	 * 
	 * @param request
	 *            the request to approve
	 * @return <code>true</code> if that advice accepts this request
	 */
	protected boolean approveSetRequest(SetRequest request) {
		EStructuralFeature feature = request.getFeature();
		for (Object o : request.getElementsToEdit()) {
			if (o instanceof State &&
					UMLPackage.Literals.STATE__CONNECTION_POINT.equals(feature)) {
				// prevent drag & drop of entry/exit point
				return false;
			}
		}
		return super.approveRequest(request);
	}

	/**
	 * Approves the specified move request or reject it.
	 * 
	 * @param request
	 *            the request to approve.
	 * @return <code>true</code> if the specified request is approved
	 */
	protected boolean approveMoveRequest(MoveRequest request) {
		return RegionUtils.shouldApproveMoveRequest(request);
	}

	/**
	 * Disallow creation of another region in a composite state: composite
	 * states have exactly one region.
	 * 
	 * @param request
	 *            an element creation request
	 * 
	 * @return approval status if the {@code request} is creating a region in a state;
	 *         {@code null} to delegate to the superclass approval, otherwise
	 */
	private Boolean approveCreateElementRequest(CreateElementRequest request) {
		Boolean result = null;

		IElementType typeToCreate = request.getElementType();
		if (ElementTypeUtils.isTypeCompatible(typeToCreate, UMLElementTypes.REGION)) {
			// Creating some kind of region. In a state?
			if (request.getContainer() instanceof State) {
				State stateToEdit = (State) request.getContainer();
				result = !stateToEdit.isComposite();
			}
		}

		return result;
	}

	@Override
	public ICommand getBeforeEditCommand(IEditCommandRequest request) {
		ICommand result = getInheritanceEditCommand(request);

		if (result == null) {
			result = super.getBeforeEditCommand(request);
		}

		return result;
	}

	@Override
	public ICommand getExcludeDependentsCommand(ExcludeDependentsRequest request) {
		ICommand result = IInheritanceEditHelperAdvice.super.getExcludeDependentsCommand(request);

		if (request.isExclude() && (request.getElementToExclude() instanceof State)) {
			// Exclude incoming and outgoing transitions, also
			UMLRTState state = UMLRTState.getInstance((State) request.getElementToExclude());
			if (state != null) {
				Predicate<UMLRTNamedElement> alreadyExcluded = UMLRTNamedElement::isExcluded;
				List<Transition> transitionsToExclude = Stream.concat(
						state.getIncomings().stream(),
						state.getOutgoings().stream()).distinct()
						.filter(alreadyExcluded.negate())
						.map(UMLRTTransition::toUML)
						.collect(Collectors.toList());

				// And for composite states, transitions into entry-points and out of exit-points.
				// Don't need to worry about transitions on the inside, because they are encapsuled
				// within the composite state, which is excluded
				List<Transition> connectionPointTransitions = Stream.concat(
						state.getEntryPoints().stream().flatMap(cp -> cp.getIncomings().stream()),
						state.getExitPoints().stream().flatMap(cp -> cp.getOutgoings().stream())).distinct()
						.filter(alreadyExcluded.negate())
						.map(UMLRTTransition::toUML)
						.collect(Collectors.toList());

				if (!connectionPointTransitions.isEmpty()) {
					if (transitionsToExclude.isEmpty()) {
						// This is the usual case for composite states
						transitionsToExclude = connectionPointTransitions;
					} else {
						transitionsToExclude.addAll(connectionPointTransitions);
					}
				}

				if (!transitionsToExclude.isEmpty()) {
					ICommand excludeTransitions = request.getExcludeDependentsCommand(transitionsToExclude);
					if (excludeTransitions != null) {
						result = CompositeCommand.compose(result, excludeTransitions);
					}
				}
			}
		}

		return result;
	}

	/**
	 * @see org.eclipse.gmf.runtime.emf.type.core.edithelper.AbstractEditHelperAdvice#configureRequest(org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest)
	 *
	 * @param request
	 */
	@Override
	public void configureRequest(IEditCommandRequest request) {
		if (request instanceof MoveRequest) {
			StateMachineUtils.retargetToRegion((MoveRequest) request);
		}
	}
}
