/*****************************************************************************
 * Copyright (c) 2015, 2017 CEA LIST, Christian W. Damus, Zeligsoft (2009), and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine Janssens (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *  Christian W. Damus - bugs 476984, 495908, 510315, 507282
 *  Young-Soo Roh - Refactored to generic multi-reference control editor
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.properties.editors;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;

import org.eclipse.core.databinding.observable.IChangeListener;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandWrapper;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.MoveCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.ElementTypeRegistry;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.DestroyElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.osgi.util.NLS;
import org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrus.infra.nattable.manager.table.INattableModelManager;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.infra.widgets.creation.ReferenceValueFactory;
import org.eclipse.papyrus.infra.widgets.editors.AbstractListEditor;
import org.eclipse.papyrus.infra.widgets.editors.ICommitListener;
import org.eclipse.papyrus.uml.diagram.common.util.CommandUtil;
import org.eclipse.papyrus.uml.service.types.element.UMLElementTypes;
import org.eclipse.papyrusrt.umlrt.core.commands.ExcludeRequest;
import org.eclipse.papyrusrt.umlrt.core.utils.EMFHacks;
import org.eclipse.papyrusrt.umlrt.tooling.properties.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.properties.messages.Messages;
import org.eclipse.papyrusrt.umlrt.tooling.properties.widget.RTNatTableMultiReferencePropertyEditor;
import org.eclipse.papyrusrt.umlrt.uml.util.UMLRTExtensionUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.uml2.common.util.UML2Util;
import org.eclipse.uml2.uml.Element;


/**
 * Button Control Editor for the multi reference property editor.
 * 
 * @see RTNatTableMultiReferencePropertyEditor
 * 
 * @author Céline JANSSENS
 *
 */
public class MultiReferenceControlEditor extends AbstractListEditor implements SelectionListener, DisposeListener {

	/**
	 * Icon for the delete button
	 */
	protected static final String DELETE_BUTTON_ICON = "/icons/Delete_12x12.gif"; //$NON-NLS-1$

	/**
	 * Icon for the Add button
	 */
	protected static final String ADD_BUTTON_ICON = "/icons/Add_12x12.gif";//$NON-NLS-1$

	/**
	 * Icon for the Add button
	 */
	protected static final String EDIT_BUTTON_ICON = "/icons/Edit_12x12.gif";//$NON-NLS-1$

	/**
	 * Icon for the Down button
	 */
	protected static final String DOWN_BUTTON_ICON = "/icons/Down_12x12.gif";//$NON-NLS-1$

	/**
	 * Icon for the Up button
	 */
	protected static final String UP_BUTTON_ICON = "/icons/Up_12x12.gif";//$NON-NLS-1$

	/**
	 * Command provider for the Transition
	 */
	protected IElementEditService provider = ElementEditServiceUtils.getCommandProvider(UMLElementTypes.TRANSITION);

	/**
	 * A Composite containing the different control buttons
	 * (Add, remove, ...)
	 */
	protected Composite controlsSection;

	/**
	 * The Add control
	 */
	protected Button add;

	/**
	 * The edit control
	 */
	protected Button edit;

	/**
	 * The Remove control
	 */
	protected Button remove;

	/**
	 * The Up control
	 */
	protected Button up;

	/**
	 * The Down control
	 */
	protected Button down;


	/**
	 * The current element
	 */
	protected Object selectedElement;

	/**
	 * The list of Selected element
	 */
	@SuppressWarnings("rawtypes")
	protected List selectedElements;

	/**
	 * The Transition relater to the selected elements
	 */
	protected EObject context;

	protected EReference containementFeature;

	protected String createElementTypeId;

	/**
	 * Action to invoke on addition of a new element.
	 */
	protected Consumer<Object> newElementAction = MultiReferenceControlEditor::pass;

	/**
	 * Action to invoke on moving of a element.
	 */
	protected BiConsumer<Object, ? super Integer> movedElementAction = MultiReferenceControlEditor::pass;

	/**
	 * Action to invoke on removal of an element.
	 */
	protected Consumer<Object> removedElementAction = MultiReferenceControlEditor::pass;


	protected Predicate<Object> canAddElement = Objects::nonNull;

	protected Predicate<Object> canMoveElement = Objects::nonNull;

	protected Predicate<Object> canRemoveElement = Objects::nonNull;

	protected IChangeListener modelChangeListener;

	/**
	 * The factory for creating and editing values from
	 * this editor
	 */
	protected ReferenceValueFactory referenceFactory;

	/**
	 * 
	 * Constructor.
	 *
	 * @param parent
	 *            The Parent Composite
	 * @param style
	 *            The Style
	 * @param context
	 *            The Related Operation on which the element is added
	 * @param nattableManager
	 *            The Table Manager of the element Table
	 */
	public MultiReferenceControlEditor(final Composite parent, final int style, final EObject context, final EStructuralFeature containment, final INattableModelManager nattableManager) {
		super(parent, style);
		this.context = context;
		this.containementFeature = (EReference) containment;
		GridLayout layout = new GridLayout(label == null ? 1 : 2, false);
		layout.marginHeight = 0;
		setLayout(layout);

		createControlSelection();

		createControlButtons();

		updateButtons();
	}

	@Override
	public void dispose() {
		try {
			if ((modelProperty != null) && (modelChangeListener != null)) {
				modelProperty.removeChangeListener(modelChangeListener);
			}
		} finally {
			super.dispose();
		}
	}

	@Override
	protected void doBinding() {
		super.doBinding();

		if (modelProperty != null) {
			modelProperty.addChangeListener(getModelChangeListener());

			// This does not depend entirely on the selection in the table
			updateButtons();
		}
	}

	protected IChangeListener getModelChangeListener() {
		if (modelChangeListener == null) {
			modelChangeListener = __ -> updateButtons();
		}

		return modelChangeListener;
	}

	/**
	 * Create the Composite for the button
	 */
	protected void createControlSelection() {
		controlsSection = new Composite(this, SWT.NONE);
		controlsSection.setLayout(new FillLayout());
		controlsSection.setLayoutData(new GridData(SWT.END, SWT.CENTER, false, false));
	}

	protected void createControlButtons() {
		up = createButton(Activator.getDefault().getImage(UP_BUTTON_ICON), Messages.MultiReferenceControlEditor_MoveUpButtonLabel); // $NON-NLS-1$
		down = createButton(Activator.getDefault().getImage(DOWN_BUTTON_ICON), Messages.MultiReferenceControlEditor_MoveDownButtonLabel); // $NON-NLS-1$
		add = createButton(Activator.getDefault().getImage(ADD_BUTTON_ICON), Messages.MultiReferenceControlEditor_AddButtonLabel); // $NON-NLS-1$
		remove = createButton(Activator.getDefault().getImage(DELETE_BUTTON_ICON), Messages.MultiReferenceControlEditor_RemoveButtonLabel); // $NON-NLS-1$
		edit = createButton(Activator.getDefault().getImage(EDIT_BUTTON_ICON), Messages.MultiReferenceControlEditor_EditButtonLabel); // $NON-NLS-1$
	}

	/**
	 * Create the Button
	 * 
	 * @param image
	 *            Image of the Button
	 * @param toolTipText
	 *            Tooltip of the new button
	 * @return the new Button
	 */
	protected Button createButton(final Image image, final String toolTipText) {
		Button button = new Button(controlsSection, SWT.PUSH);
		button.setImage(image);
		button.addSelectionListener(this);
		button.setToolTipText(toolTipText);
		return button;
	}

	/**
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractListEditor#setModelObservable(org.eclipse.core.databinding.observable.list.IObservableList)
	 *
	 * @param modelProperty
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public void setModelObservable(IObservableList modelProperty) {
		super.setModelObservable(modelProperty);
		if (modelProperty instanceof ICommitListener) {
			this.addCommitListener((ICommitListener) modelProperty);
		}
	}

	/**
	 * Sets the {@link ReferenceValueFactory} for this editor. The {@link ReferenceValueFactory} is used to create
	 * new instances and edit existing ones.
	 *
	 * @param factory
	 *            The {@link ReferenceValueFactory} to be used by this editor
	 */
	public void setFactory(ReferenceValueFactory factory) {
		this.referenceFactory = factory;
		updateButtons();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#getEditableType()
	 *
	 * @return
	 */
	@Override
	public Object getEditableType() {
		return Collection.class;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#setReadOnly(boolean)
	 * 
	 */
	@Override
	public void setReadOnly(final boolean readOnly) {
		// Nothing to do
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#isReadOnly()
	 * 
	 */
	@Override
	public boolean isReadOnly() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
	 *
	 */
	@Override
	public void widgetDisposed(final DisposeEvent e) {
		super.widgetDisposed(e);
		add.dispose();
		up.dispose();
		remove.dispose();
		down.dispose();
		edit.dispose();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	@Override
	public void widgetSelected(final SelectionEvent e) {

		if (add == e.widget) {
			addAction();
		}

		if (null != getSelectedElements()) {
			for (Object element : getSelectedElements()) {

				setSelectedElement(element);


				if (null != e.widget) {

					if (remove == e.widget) {
						removeAction();
					} else if (up == e.widget) {
						upAction();
					} else if (down == e.widget) {
						downAction();
					} else if (edit == e.widget) {
						editAction();
					}
				}
			}
		}

		updateButtons();

	}


	/**
	 * Down Action
	 */
	protected void downAction() {

		// Move the selected element item to the next position.
		// Note that the edit-helpers don't provide move commands within
		// a list feature, so we simply do it the EMF way. There isn't any
		// useful prospect for advice on this reordering, anyways
		int indexOf = modelProperty.indexOf(selectedElement);
		EditingDomain domain = EMFHelper.resolveEditingDomain(context);
		if ((domain != null) && (indexOf + 1) < modelProperty.size()) {
			Command move = MoveCommand.create(domain,
					context, containementFeature,
					selectedElement, indexOf + 1);
			if (move != null) {
				// Customize the label
				move = new CommandWrapper(Messages.MultiReferenceControlEditor_MoveDownAction, Messages.MultiReferenceControlEditor_MoveDownAction, move);
				CommandUtil.executeCommandInStack(move, context);

				// Process the moved element
				movedElementAction.accept(selectedElement, indexOf);
			}
		}
	}


	/**
	 * Up Action
	 */
	protected void upAction() {
		// Move the selected element item to the previous position
		// Note that the edit-helpers don't provide move commands within
		// a list feature, so we simply do it the EMF way. There isn't any
		// useful prospect for advice on this reordering, anyways
		int indexOf = modelProperty.indexOf(selectedElement);
		EditingDomain domain = EMFHelper.resolveEditingDomain(context);
		if ((domain != null) && (0 <= (indexOf - 1))) {
			Command move = MoveCommand.create(domain,
					context, containementFeature,
					selectedElement, indexOf - 1);
			if (move != null) {
				// Customize the label
				move = new CommandWrapper(Messages.MultiReferenceControlEditor_MoveUpAction, Messages.MultiReferenceControlEditor_MoveUpAction, move);
				CommandUtil.executeCommandInStack(move, context);

				// Process the moved element
				movedElementAction.accept(selectedElement, indexOf);
			}
		}
	}


	/**
	 * Remove Action
	 */
	protected void removeAction() {
		IEditCommandRequest request;

		EObject subject = (EObject) selectedElement;
		if ((subject instanceof Element) && UMLRTExtensionUtil.isInherited((Element) subject)) {
			// Exclude the element (it cannot be destroyed)
			request = new ExcludeRequest((Element) subject, true);
		} else {
			// Destroy the element
			request = new DestroyElementRequest((EObject) selectedElement, false);
		}

		// Execute the Command
		if (null != getProvider()) {
			ICommand command = getProvider().getEditCommand(request);

			if (null != command) {
				command.setLabel(NLS.bind(Messages.MultiReferenceControlEditor_RemoveAction, containementFeature.getEType().getName()));
				Command wrapperCommand = GMFtoEMFCommandWrapper.wrap(command);
				CommandUtil.executeCommandInStack(wrapperCommand, context);

				CommandResult result = command.getCommandResult();
				if (succeeded(result)) {
					// It would appear to have been removed
					removedElementAction.accept(subject);
				}
			}
		}
	}

	/**
	 * Set create element ID.
	 * 
	 * @param id
	 *            element ID
	 */
	public void setCreateElementTypeID(String id) {
		createElementTypeId = id;
	}

	/**
	 * Add Action
	 */
	protected void addAction() {

		// no specific creation id specified. so use reference factory.
		if (UML2Util.isEmpty(createElementTypeId)) {
			if (referenceFactory != null && referenceFactory.canCreateObject()) {

				getOperationExecutor(context).execute(new Runnable() {

					@SuppressWarnings("unchecked")
					@Override
					public void run() {
						Object newElement = referenceFactory.createObject(MultiReferenceControlEditor.this, context);
						if (newElement != null) {
							modelProperty.add(newElement);
							commit();
							newElementAction.accept(newElement);
						}
					}
				}, NLS.bind(Messages.MultiReferenceControlEditor_AddAction, containementFeature.getEType().getName()));

			}
			return;
		}

		// Build the request
		CreateElementRequest request = new CreateElementRequest(
				context,
				ElementTypeRegistry.getInstance().getType(createElementTypeId),
				containementFeature);


		// Execute the command accordingly
		if (null != getContextCommandProvider()) {
			// Get the Creation Command from the Service Edit Provider of the Parameter
			ICommand createCommand = getContextCommandProvider().getEditCommand(request);

			if (null != createCommand) {
				createCommand.setLabel(NLS.bind(Messages.MultiReferenceControlEditor_AddAction, containementFeature.getEType().getName()));
				Command wrapperCommand = GMFtoEMFCommandWrapper.wrap(createCommand);

				// Is the table in a dialog, which is in the context of an open transaction?
				boolean nested = EMFHacks.isReadWriteTransactionActive(context);

				if (nested) {
					// Temporarily disable notification of addition of the new parameter
					// so that the UI doesn't update prematurely, then notify the addition
					// explicitly on its behalf later.
					// XXX: This is potentially dangerous because if the command adds
					// another parameter also to the same operation, then the transaction
					// will record changes out of order and undo will be broken.
					EMFHacks.silently(context, op -> CommandUtil.executeCommandInStack(wrapperCommand, context));
				} else {
					// The table takes care of delaying the update, itself
					CommandUtil.executeCommandInStack(wrapperCommand, context);
				}

				CommandResult result = createCommand.getCommandResult();
				if (succeeded(result)) {
					EObject added = (EObject) result.getReturnValue();

					if (nested) {
						// Notify addition of the parameter
						EMFHacks.notifyAdded(added);
					}

					// Post-process the new parameter
					newElementAction.accept(added);
				}
			}
		}

	}

	private static boolean succeeded(CommandResult commandResult) {
		return (commandResult != null) && (commandResult.getStatus() != null)
				&& (commandResult.getStatus().getSeverity() < IStatus.ERROR);
	}

	protected void editAction() {
		getOperationExecutor(selectedElement).execute(new Runnable() {

			@SuppressWarnings("unchecked")
			@Override
			public void run() {
				int indexOf = modelProperty.indexOf(selectedElement);
				try {
					Object newValue = referenceFactory.edit(MultiReferenceControlEditor.this.edit, selectedElement);

					if (newValue != selectedElement && newValue != null) {
						modelProperty.remove(indexOf);
						modelProperty.add(indexOf, newValue);
					}

					commit();
				} catch (OperationCanceledException e) {
					// refresh in case of cancel, since started (and then undone) modifications had been
					// directly reflected in the table and must now show the original contents again.
					refreshValue();
				}
			}
		}, NLS.bind(Messages.MultiReferenceControlEditor_EditAction, containementFeature.getEType().getName()));
	}

	/**
	 * Get Command Provider
	 * 
	 * @return the command Provider
	 */
	protected IElementEditService getElementCommandProvider() {
		IElementEditService commandProvider = ElementEditServiceUtils.getCommandProvider(context);
		return commandProvider;
	}



	/**
	 * Update the Button by setting the isEnable field.
	 */
	@SuppressWarnings("unchecked")
	public void updateButtons() {
		/* Disable the button 'add' if the upperBound is reached */
		if ((null == selectedElements) || (selectedElements.isEmpty())) {
			remove.setEnabled(false);
			up.setEnabled(false);
			down.setEnabled(false);
			edit.setEnabled(false);
		} else {
			remove.setEnabled(canRemove());
			edit.setEnabled(true);

			if (!canMove()) {
				up.setEnabled(false);
				down.setEnabled(false);
			} else {
				OptionalInt minSelectionIndex = selectedElements.stream().mapToInt(this::indexOf).min();
				OptionalInt maxSelectionIndex = selectedElements.stream().mapToInt(this::indexOf).max();
				up.setEnabled(minSelectionIndex.orElse(-1) > 0);
				down.setEnabled(maxSelectionIndex.orElse(Integer.MAX_VALUE) < (modelProperty.size() - 1));
			}
		}

		add.setEnabled(canAdd());
	}

	protected int indexOf(Object element) {
		return modelProperty.indexOf(element);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	@Override
	public void widgetDefaultSelected(final SelectionEvent e) {
		// Nothing selected by default
	}


	/**
	 * Setter of selectParamater
	 */
	public void setSelectedElement(final Object selectedElement) {
		this.selectedElement = selectedElement;

	}


	/**
	 * Setter of selectParamaters
	 */
	@SuppressWarnings("rawtypes")
	public void setSelectedElements(final List selectedElementList) {
		this.selectedElements = selectedElementList;

	}


	/**
	 * Getter of the Command Provider
	 */
	public IElementEditService getProvider() {
		return provider;
	}


	/**
	 * Setter of the Command Provider
	 */
	public void setProvider(IElementEditService provider) {
		this.provider = provider;
	}

	/**
	 * Get Command Provider
	 * 
	 * @return the command Provider
	 */
	protected IElementEditService getContextCommandProvider() {
		IElementEditService commandProvider = ElementEditServiceUtils.getCommandProvider(context);
		return commandProvider;
	}


	/**
	 * Getter of selectedElements
	 */
	@SuppressWarnings("rawtypes")
	public List getSelectedElements() {
		return selectedElements;
	}

	/**
	 * Registers an action to be performed on the addition of a new element.
	 * 
	 * @param newElementAction
	 *            an action that accepts the new element
	 */
	public void onElementAdded(Consumer<Object> newElementAction) {
		this.newElementAction = (newElementAction != null)
				? newElementAction
				: MultiReferenceControlEditor::pass;
	}

	private static void pass(Object __) {
		// Pass
	}

	/**
	 * Registers an action to be performed on the reordering of a element.
	 * 
	 * @param movedElementAction
	 *            an action that accepts the moved element and its previous index in the elements list
	 */
	public void onElementMoved(BiConsumer<Object, ? super Integer> movedElementAction) {
		this.movedElementAction = (movedElementAction != null)
				? movedElementAction
				: MultiReferenceControlEditor::pass;
	}

	private static void pass(Object _1, Object _2) {
		// Pass
	}

	/**
	 * Registers an action to be performed on the removal of an element.
	 * 
	 * @param removedElementAction
	 *            an action that accepts the removed element
	 */
	public void onElementRemoved(Consumer<Object> removedElementAction) {
		this.removedElementAction = (removedElementAction != null)
				? removedElementAction
				: MultiReferenceControlEditor::pass;
	}

	/**
	 * Registers a predicate determining whether an element may be added to the element
	 * being edited.
	 * 
	 * @param canAddElement
	 *            a predicate on the element being edited (which would be the
	 *            container of any element that would be added)
	 */
	public void enableAddElement(Predicate<Object> canAddElement) {
		this.canAddElement = this.canAddElement.and(canAddElement);

		// This does not depend on the selection
		updateButtons();
	}

	protected boolean canAdd() {
		return canAddElement.test(getContextElement());
	}

	/**
	 * Registers a predicate determining whether an element may be moved.
	 * It is not necessary here to account for whether the element is
	 * at the beginning or end of the list.
	 * 
	 * @param canMoveElement
	 *            a predicate on the element being moved
	 */
	public void enableMoveElement(Predicate<Object> canMoveElement) {
		this.canMoveElement = this.canMoveElement.and(canMoveElement);
	}

	protected boolean canMove() {
		return ((List<?>) getSelectedElements()).stream().allMatch(canMoveElement);
	}

	/**
	 * Registers a predicate determining whether an element may be removed.
	 * This applies equally to exclusion as to deletion.
	 * 
	 * @param canRemoveElement
	 *            a predicate on the element being removed
	 */
	public void enableRemoveElement(Predicate<Object> canRemoveElement) {
		this.canRemoveElement = this.canRemoveElement.and(canRemoveElement);
	}

	protected boolean canRemove() {
		return ((List<?>) getSelectedElements()).stream().allMatch(canRemoveElement);
	}
}
