/*******************************************************************************
 * Copyright (c) 2017 Zeligsoft (2009) Limited and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Young-Soo Roh - Initial API and implementation
 *******************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.ICodeSnippetTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.OperationTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.StateEntryTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.StateExitTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.TransitionEffectTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.TransitionGuardTab;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs.TriggerGuardTab;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.Page;

/**
 * <p>
 * Code snippet page containing available tabs {@link ICodeSnippetTab}. This
 * page controls life cycle of visible tabs.
 * </p>
 * 
 * @author ysroh
 *
 */
public class CodeSnippetPage extends Page implements ICodeSnippetPage {

	/**
	 * Main page composite.
	 */
	private Composite pageControl;

	/**
	 * Selected EObject.
	 */
	private EObject currentSelectedEObject;

	/**
	 * Tab folder.
	 */
	private CTabFolder tabFolder;

	/**
	 * Default empty tab item.
	 */
	private CTabItem defaultTabItem;
	/**
	 * Available tabs.
	 */
	private List<ICodeSnippetTab> tabs = new ArrayList<>();

	/**
	 * Active tab items.
	 */
	private Map<ICodeSnippetTab, CTabItem> tabItems = new HashMap<ICodeSnippetTab, CTabItem>();

	/**
	 * Listeners.
	 */
	private List<ISelectionChangedListener> listeners = new ArrayList<ISelectionChangedListener>();

	/**
	 * 
	 * Constructor.
	 *
	 */
	public CodeSnippetPage() {
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			if (selection.isEmpty()) {
				currentSelectedEObject = null;
			} else {
				EObject eo = EMFHelper.getEObject(((IStructuredSelection) selection).getFirstElement());

				if (eo != null && !eo.equals(currentSelectedEObject)) {
					currentSelectedEObject = eo;
				}
			}

			for (ICodeSnippetTab tab : tabs) {
				if (tab.shouldShowThisTab(currentSelectedEObject)) {
					CTabItem tabItem = tabItems.get(tab);
					if (tabItem == null) {
						tabItem = tab.createControl(tabFolder);
						tabItems.put(tab, tabItem);
						if (tabFolder.getSelection() == null) {
							tabFolder.setSelection(0);
						}
					}
					// notify tab of new selection
					tab.setInput(currentSelectedEObject);
				} else {
					// remove tab
					tabItems.remove(tab);
					// dispose tab composite
					tab.dispose();
				}
			}

			if (tabItems.isEmpty()) {
				createDefaultTabItem();
				tabFolder.setSelection(0);
			} else {
				if (defaultTabItem != null) {
					defaultTabItem.dispose();
					defaultTabItem = null;
				}
			}

			fireSelectionChangedEvent();
		}
	}

	/**
	 * Notify listeners of selection change.
	 */
	private void fireSelectionChangedEvent() {
		for (ISelectionChangedListener l : listeners) {
			l.selectionChanged(new SelectionChangedEvent(this, getSelection()));
		}
	}

	@Override
	public void createControl(Composite parent) {

		// ToDo: we can provide extension mechanism if needed
		// but for now it is fairly limited tabs we need.
		tabs.add(new TransitionEffectTab());
		tabs.add(new TransitionGuardTab());
		tabs.add(new StateEntryTab());
		tabs.add(new StateExitTab());
		tabs.add(new TriggerGuardTab());
		tabs.add(new OperationTab());

		// compose page control
		pageControl = new Composite(parent, SWT.NONE);
		pageControl.setLayout(new FillLayout());

		// create tab folder
		tabFolder = new CTabFolder(pageControl, SWT.BOTTOM);
		tabFolder.setLayoutData(new GridData(GridData.FILL_BOTH));

		getSite().setSelectionProvider(this);
	}

	/**
	 * Create default empty tab.
	 */
	private void createDefaultTabItem() {
		if (defaultTabItem == null) {
			defaultTabItem = new CTabItem(tabFolder, SWT.NONE);
			final Composite tabComposite = new Composite(tabFolder, SWT.NONE);
			tabComposite.setLayoutData(new GridData());
			tabComposite.setLayout(new GridLayout());
			CLabel label = new CLabel(tabComposite, SWT.NONE);
			label.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, false, false));
			label.setText("Code snippet view is not available for the current selection.");
			defaultTabItem.setControl(tabComposite);
			defaultTabItem.addDisposeListener(new DisposeListener() {

				@Override
				public void widgetDisposed(DisposeEvent e) {
					if (tabComposite != null) {
						tabComposite.dispose();
					}
				}
			});
		}
	}

	@Override
	public Control getControl() {
		return pageControl;
	}

	@Override
	public void setFocus() {
		pageControl.setFocus();
	}

	@Override
	public void dispose() {
		getSite().setSelectionProvider(null);
		if (pageControl != null) {
			pageControl.dispose();
		}
		super.dispose();
	}

	@Override
	public ISelection getSelection() {
		StructuredSelection selection = new StructuredSelection();
		if (currentSelectedEObject != null) {
			boolean isTrigger = tabItems.keySet().stream().anyMatch(t -> t instanceof TriggerGuardTab);
			if (isTrigger) {
				// show its transition in the properties view
				selection = new StructuredSelection(currentSelectedEObject.eContainer());
			} else {
				selection = new StructuredSelection(currentSelectedEObject);
			}
		}
		return selection;
	}

	@Override
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		listeners.add(listener);
	}

	@Override
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		listeners.remove(listener);
	}

	@Override
	public void setSelection(ISelection selection) {
		// not required for code snippet view.
	}
}
