/*******************************************************************************
 * Copyright (c) 2017 Zeligsoft (2009) Limited and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Young-Soo Roh - Initial API and implementation
 *******************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.tabs;

import java.util.Collections;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.databinding.observable.ChangeEvent;
import org.eclipse.core.databinding.observable.IChangeListener;
import org.eclipse.core.databinding.observable.IObservable;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.ValueChangeEvent;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.infra.services.labelprovider.service.LabelProviderService;
import org.eclipse.papyrus.uml.properties.expression.ExpressionList;
import org.eclipse.papyrus.uml.properties.expression.ExpressionList.Expression;
import org.eclipse.papyrus.uml.properties.modelelement.UMLModelElement;
import org.eclipse.papyrusrt.umlrt.core.internal.defaultlanguage.NoDefautLanguage;
import org.eclipse.papyrusrt.umlrt.core.utils.CodeSnippetUtils;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.editors.ILanguageEditor;
import org.eclipse.papyrusrt.umlrt.tooling.views.codesnippet.editors.LanguageEditorRegistry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.uml2.common.util.UML2Util;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;

/**
 * <p>
 * Abstract code snippet tab. Code snippet tabs should extend this class to
 * utilize basic functionalities provided by this abstract class.
 * </p>
 * 
 * @author ysroh
 *
 */
@SuppressWarnings("restriction")
public abstract class AbstractCodeSnippetTab implements ICodeSnippetTab, IChangeListener {

	/**
	 * Bold font.
	 */
	private static Font bold = JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT);

	/**
	 * Italic font.
	 */
	private static Font italic = JFaceResources.getFontRegistry().getItalic(JFaceResources.DEFAULT_FONT);

	/**
	 * Current input.
	 */
	protected EObject input;

	/**
	 * Element label provider.
	 */
	protected ILabelProvider labelProvider;

	/**
	 * Main composite.
	 */
	protected Composite tabComposite;

	/**
	 * Tab item.
	 */
	protected CTabItem tabItem;

	/**
	 * Label.
	 */
	protected CLabel label;

	/**
	 * Language label.
	 */
	protected CLabel languageLabel;

	/**
	 * Indicator for creation of composite.
	 */
	protected boolean controlCreated = false;

	/**
	 * Default language.
	 */
	protected String defaultLanguage = UML2Util.EMPTY_STRING;

	/**
	 * Observable for feature containing source code.
	 */
	protected IObservable featureObservable;

	/**
	 * Selected language.
	 */
	protected String selectedLanguage;

	/**
	 * Observable list for source and body pairs.
	 */
	protected ExpressionList expressionObservableList;

	/**
	 * Editor for selected language.
	 */
	protected ILanguageEditor languageEditor;

	@Override
	public void setInput(EObject input) {
		this.input = input;
		// update default language
		defaultLanguage = CodeSnippetUtils.getDefaultLanguage((Element) input).getName();
		selectedLanguage = defaultLanguage;

		// update observable
		updateFeatureObservable();
		updateExpressionObservableList();

		// update label
		updateLabel();

		updateLanguageEditor();

		// refresh contents
		refresh();

	}

	/**
	 * Update language editor.
	 */
	protected void updateLanguageEditor() {
		if (languageEditor != null) {
			languageEditor.dispose();
		}
		languageEditor = LanguageEditorRegistry.getInstance().getNewLanguageEditorInstance(selectedLanguage);
		languageEditor.createControl(tabComposite);
		languageEditor.addValueChangeListener(new IValueChangeListener<Expression>() {
			@Override
			public void handleValueChange(ValueChangeEvent<? extends Expression> event) {
				commit();
			}
		});
		tabComposite.layout();
	}

	/**
	 * Dispose context observable.
	 */
	private void disposeFeatureObservable() {
		if (featureObservable != null) {
			featureObservable.removeChangeListener(this);
			featureObservable.dispose();
			featureObservable = null;
		}
	}

	/**
	 * Dispose observable list for language and body pairs.
	 */
	private void disposeExpressionObservableList() {
		if (expressionObservableList != null) {
			expressionObservableList.removeChangeListener(this);
			for (Object o : expressionObservableList) {
				Expression ex = (Expression) o;
				ex.removeChangeListener(this);
			}
			expressionObservableList.dispose();
			expressionObservableList = null;
		}
	}

	/**
	 * Update feature observable.
	 */
	protected void updateFeatureObservable() {

		disposeFeatureObservable();
		featureObservable = getFeatureObservable();

		if (featureObservable != null) {
			featureObservable.addChangeListener(this);
		}
	}

	/**
	 * Update expression observable list.
	 */
	protected void updateExpressionObservableList() {

		disposeExpressionObservableList();

		// set to default font
		tabItem.setFont(italic);

		if (input != null) {
			expressionObservableList = getExpressionObservableList();
			if (expressionObservableList != null) {
				expressionObservableList.addChangeListener(this);
				for (Object o : expressionObservableList) {
					Expression ex = (Expression) o;
					ex.addChangeListener(this);
					if (ex.getLanguage().equals(selectedLanguage) && !UML2Util.isEmpty(ex.getBody())) {
						// body defined for selected language so make it
						// bold
						tabItem.setFont(bold);
					}
				}
			}
		}
	}

	@Override
	public CTabItem createControl(CTabFolder parent) {
		controlCreated = true;

		// create tab composite
		tabComposite = new Composite(parent, SWT.NONE);
		tabComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		tabComposite.setLayout(new GridLayout());

		// create labels
		Composite labelComposite = new Composite(tabComposite, SWT.NONE);
		GridData data = new GridData(SWT.LEFT, SWT.TOP, false, false);
		labelComposite.setLayoutData(data);
		GridLayout layout = new GridLayout(2, false);
		layout.horizontalSpacing = 0;
		layout.verticalSpacing = 0;
		layout.marginHeight = 0;
		labelComposite.setLayout(layout);
		label = new CLabel(labelComposite, SWT.NONE);
		languageLabel = new CLabel(labelComposite, SWT.NONE);

		// ToDo: support CDT integration in the future.
		languageEditor = LanguageEditorRegistry.getInstance()
				.getNewLanguageEditorInstance(NoDefautLanguage.INSTANCE.getName());
		languageEditor.createControl(tabComposite);

		tabItem = new CTabItem(parent, SWT.NONE);
		tabItem.setControl(tabComposite);

		return tabItem;
	}

	@Override
	public void handleChange(ChangeEvent event) {
		if (tabItem != null && !tabItem.isDisposed()) {

			IObservable observable = event.getObservable();
			if (!(observable instanceof Expression) && !(observable instanceof ExpressionList)) {
				// owner of expression has update so update observable
				updateExpressionObservableList();
			}
			// refresh contents
			refresh();
		}
	}

	/**
	 * Update label.
	 */
	protected void updateLabel() {
		if (controlCreated && input != null) {

			// update the label for the object
			try {
				labelProvider = ServiceUtilsForEObject.getInstance().getService(LabelProviderService.class, input)
						.getLabelProvider();
				label.setImage(labelProvider.getImage(input));
				label.setText(labelProvider.getText(input));
				languageLabel.setText("(" + selectedLanguage + ")");

				tabComposite.layout();

				tabItem.setText(getTitle());
				tabItem.setImage(getImage());

			} catch (ServiceException e) {

			}
		}
	}

	@Override
	public boolean controlCreated() {
		return controlCreated;
	}

	@Override
	public void dispose() {
		disposeFeatureObservable();
		disposeExpressionObservableList();
		if (tabItem != null && !tabItem.isDisposed()) {
			// dispose tabItem
			tabItem.dispose();
		}
		if (tabComposite != null && !tabComposite.isDisposed()) {
			tabComposite.dispose();
		}
		tabComposite = null;
		controlCreated = false;
		input = null;
	}

	/**
	 * Subclass to override to refresh contents.
	 */
	protected void refresh() {

		final Expression ex = new Expression();
		ex.setLanguage(selectedLanguage);
		ex.setBody(UML2Util.EMPTY_STRING);
		if (expressionObservableList != null) {
			int index = expressionObservableList.indexOf(ex);
			if (index != -1) {
				ex.setBody(((Expression) expressionObservableList.get(index)).getBody());
			}
		}

		SafeRunnable runnable = new SafeRunnable() {

			@Override
			public void run() throws Exception {
				languageEditor.setValue(ex);
			}
		};
		SafeRunnable.run(runnable);
	}

	/**
	 * Commit.
	 */
	@SuppressWarnings("unchecked")
	protected void commit() {
		TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(input);
		AbstractTransactionalCommand cmd = null;
		Expression ex = languageEditor.getValue();

		String commandLabel = "Save " + getTitle() + " Code";

		if (expressionObservableList != null) {
			// save to existing element containing expression
			int index = expressionObservableList.indexOf(ex);
			if (index != -1) {
				Expression expression = (Expression) expressionObservableList.get(index);
				if (ex.getBody().equals(expression.getBody())) {
					// do not commit if there is no change
					// this will prevent creation of redefined elements.
					return;
				}
				expression.setBody(ex.getBody());
			} else {
				// add new language
				expressionObservableList.add(ex);
			}
			cmd = new AbstractTransactionalCommand(domain, commandLabel, Collections.EMPTY_LIST) {

				@Override
				protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info)
						throws ExecutionException {

					expressionObservableList.commit(null);

					return CommandResult.newOKCommandResult();
				}
			};
		} else if (!UML2Util.isEmpty(ex.getBody())) {
			// need to create an element to store initial expression.
			cmd = new AbstractTransactionalCommand(domain, commandLabel, Collections.EMPTY_LIST) {

				@Override
				protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info)
						throws ExecutionException {

					saveInitialExpression(ex);

					return CommandResult.newOKCommandResult();
				}
			};
		}

		if (cmd != null && cmd.canExecute()) {
			domain.getCommandStack().execute(new GMFtoEMFCommandWrapper(cmd));
		}
	}

	/**
	 * Get expression list from element.
	 * 
	 * @param element
	 *            element
	 * @return Expression list
	 */
	@SuppressWarnings("rawtypes")
	protected ExpressionList getExpressionList(EObject element) {
		if (element instanceof OpaqueBehavior || element instanceof OpaqueExpression) {
			UMLModelElement modelElement = new UMLModelElement(element);
			IObservableList languages = (IObservableList) modelElement.doGetObservable("language");
			IObservableList bodies = (IObservableList) modelElement.doGetObservable("body");
			ExpressionList el = new ExpressionList(languages, bodies);
			return el;
		}
		return null;
	}

	/**
	 * Get tab title.
	 * 
	 * @return title
	 */
	protected String getTitle() {
		return "";
	}

	/**
	 * Get tab image.
	 * 
	 * @return image
	 */
	protected Image getImage() {
		return null;
	}

	/**
	 * Get feature observable.
	 * 
	 * @return feature observable.
	 */
	protected abstract IObservable getFeatureObservable();

	/**
	 * Queries expression observable list for this feature.
	 * 
	 * @return Expression list
	 */
	protected abstract ExpressionList getExpressionObservableList();

	/**
	 * Save user code for the first time. May need to create feature containing
	 * source code first.
	 * 
	 * @param expression
	 *            expression
	 */
	protected abstract void saveInitialExpression(Expression expression);

}
