/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.common.ui.preferences;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.papyrusrt.umlrt.common.ui.Activator;
import org.eclipse.papyrusrt.umlrt.common.ui.internal.preferences.ToggleDialogDelegateRegistry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.eclipse.ui.statushandlers.IStatusAdapterConstants;
import org.eclipse.ui.statushandlers.StatusAdapter;
import org.eclipse.ui.statushandlers.StatusManager;
import org.osgi.service.prefs.BackingStoreException;

/**
 * Utility for working with dialog preferences, especially of the "do not
 * ask again" variety.
 */
public class DialogPreferences {

	private static final String DIALOG_TOGGLE_NODE = "dialogToggle"; //$NON-NLS-1$

	private static final String DIALOG_TOGGLE_PATH = Activator.PLUGIN_ID + '/' + DIALOG_TOGGLE_NODE;

	private static final IPreferenceStore dialogTogglePreferences = new ScopedPreferenceStore(
			InstanceScope.INSTANCE, DIALOG_TOGGLE_PATH);

	private static ToggleDialogDelegateRegistry delegateRegistry;

	/**
	 * Not instantiable by clients.
	 */
	private DialogPreferences() {
		super();
	}

	/**
	 * Obtains the preference store to use for management of the dialog
	 * "Remember my decision" preferences. Other preferences that are more
	 * specific to a particular dialog should not use this but their own
	 * host bundle's preference store.
	 * 
	 * @return the common dialog toggle preference store. Every key in this store
	 *         is the unique identifier of a dialog
	 */
	public static IPreferenceStore getDialogTogglePreferenceStore() {
		return dialogTogglePreferences;
	}

	/**
	 * Convenience method to open a yes/no question dialog,
	 * with a "Remember my decision" toggle, returning the user's answer.
	 * The dialog is only opened if necessary (as determined by the preferences).
	 *
	 * @param parent
	 *            the parent shell of the dialog, or {@code null} if none
	 * @param title
	 *            the dialog's title, or {@code null} if none
	 * @param message
	 *            the message
	 * @param dialogID
	 *            the unique identifier of the dialog for persistence of the
	 *            toggle preference
	 * 
	 * @return the dialog result code, with or without having shown the dialog,
	 *         either {@link IDialogConstants#YES_ID} or {@link IDialogConstants#NO_ID}
	 */
	public static int yesNo(Shell parent, String title,
			String message, String dialogID) {

		return prompt(MessageDialog.QUESTION, parent, title, message, dialogID, SWT.SHEET);
	}

	/**
	 * Convenience method to open a yes/no/cancel question dialog,
	 * with a "Remember my decision" toggle, returning the user's answer.
	 * The dialog is only opened if necessary (as determined by the preferences).
	 *
	 * @param parent
	 *            the parent shell of the dialog, or {@code null} if none
	 * @param title
	 *            the dialog's title, or {@code null} if none
	 * @param message
	 *            the message
	 * @param dialogID
	 *            the unique identifier of the dialog for persistence of the
	 *            toggle preference
	 * 
	 * @return the dialog result code, with or without having shown the dialog,
	 *         one of {@link IDialogConstants#YES_ID}, {@link IDialogConstants#NO_ID,
	 *         or {@link IDialogConstants#CANCEL_ID}
	 */
	public static int yesNoCancel(Shell parent, String title,
			String message, String dialogID) {

		return prompt(MessageDialog.QUESTION_WITH_CANCEL, parent, title, message, dialogID, SWT.SHEET);
	}

	/**
	 * Convenience method to open a yes/no question dialog,
	 * with a "Remember my decision" toggle, returning the user's answer.
	 * The dialog is only opened if necessary (as determined by the preferences).
	 *
	 * @param parent
	 *            the parent shell of the dialog, or {@code null} if none
	 * @param title
	 *            the dialog's title, or {@code null} if none
	 * @param message
	 *            the message
	 * @param dialogID
	 *            the unique identifier of the dialog for persistence of the
	 *            toggle preference
	 * 
	 * @return the dialog result code, with or without having shown the dialog,
	 *         either {@link IDialogConstants#OK_ID} or {@link IDialogConstants#CANCEL_ID}
	 */
	public static int confirm(Shell parent, String title,
			String message, String dialogID) {

		return prompt(MessageDialog.CONFIRM, parent, title, message, dialogID, SWT.SHEET);
	}

	/**
	 * Convenience method to open a simple dialog of the given {@code kind},
	 * with a "Remember my decision" toggle, returning the user's answer.
	 * The dialog is only opened if necessary (as determined by the preferences).
	 *
	 * @param kind
	 *            the kind of dialog to open, one of {@link MessageDialog#ERROR},
	 *            {@link MessageDialog#INFORMATION}, {@link MessageDialog#QUESTION},
	 *            {@link MessageDialog#WARNING}, {@link MessageDialog#CONFIRM}, or
	 *            {@link MessageDialog#QUESTION_WITH_CANCEL}.
	 * @param parent
	 *            the parent shell of the dialog, or {@code null} if none
	 * @param title
	 *            the dialog's title, or {@code null} if none
	 * @param message
	 *            the message
	 * @param dialogID
	 *            the unique identifier of the dialog for persistence of the
	 *            toggle preference
	 * @param style
	 *            {@link SWT#NONE} for a default dialog, or {@link SWT#SHEET} for
	 *            a dialog with sheet behavior
	 * 
	 * @return the dialog result code, with or without having shown the dialog
	 */
	public static int prompt(int kind, Shell parent, String title,
			String message, String dialogID, int style) {

		int result = IDialogConstants.CANCEL_ID;

		IPreferenceStore store = getDialogTogglePreferenceStore();

		String preference = store.getString(dialogID);
		if (preference == null) {
			preference = MessageDialogWithToggle.PROMPT;
		}

		switch (preference) {
		case MessageDialogWithToggle.ALWAYS:
			switch (kind) {
			case MessageDialog.QUESTION:
			case MessageDialog.QUESTION_WITH_CANCEL:
				result = IDialogConstants.YES_ID;
				break;
			default:
				result = IDialogConstants.OK_ID;
				break;
			}
			break;
		case MessageDialogWithToggle.NEVER:
			switch (kind) {
			case MessageDialog.QUESTION:
			case MessageDialog.QUESTION_WITH_CANCEL:
				result = IDialogConstants.NO_ID;
				break;
			default:
				result = IDialogConstants.CANCEL_ID;
				break;
			}
			break;
		default:
			MessageDialogWithToggle dialog = MessageDialogWithToggle.open(
					kind, parent, title, message,
					null, false, store, dialogID, style);
			result = dialog.getReturnCode();
			break;
		}

		return result;
	}

	/**
	 * Clears all "Remember my decision" dialog toggle settings managed through
	 * this API.
	 * 
	 * @see #getDialogTogglePreferenceStore()
	 */
	public static void clearDialogToggles() {
		IEclipsePreferences node = InstanceScope.INSTANCE.getNode(DIALOG_TOGGLE_PATH);
		if (node != null) {
			// Delete all of these dialog toggles
			try {
				node.removeNode();
			} catch (BackingStoreException e) {
				StatusAdapter adapter = new StatusAdapter(new Status(IStatus.ERROR, Activator.PLUGIN_ID, Messages.DialogPreferences_failedMsg, e));
				adapter.setProperty(IStatusAdapterConstants.TIMESTAMP_PROPERTY, System.currentTimeMillis());
				adapter.setProperty(IStatusAdapterConstants.TITLE_PROPERTY, Messages.DialogPreferences_failedTitle);
				StatusManager.getManager().handle(adapter, StatusManager.SHOW);
			}
		}

		// And invoke delegates
		getDelegateRegistry().getToggleDialogDelegates().forEach(d -> {
			try {
				d.clearUserDecisions();
			} catch (Exception e) {
				Activator.log.error("Uncaught exception in toggle-dialog delegate.", e); //$NON-NLS-1$
			}
		});
	}

	private static ToggleDialogDelegateRegistry getDelegateRegistry() {
		if (delegateRegistry == null) {
			delegateRegistry = new ToggleDialogDelegateRegistry();
		}

		return delegateRegistry;
	}
}
