/*****************************************************************************
 * Copyright (c) 2016 EclipseSource Services GmbH 
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Philip Langer (EclipseSource) - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.compare.internal;

import static com.google.common.base.Predicates.instanceOf;
import static com.google.common.base.Predicates.or;
import static com.google.common.collect.Iterables.any;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.find;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.Monitor;
import org.eclipse.emf.compare.Comparison;
import org.eclipse.emf.compare.Conflict;
import org.eclipse.emf.compare.Diff;
import org.eclipse.emf.compare.Match;
import org.eclipse.emf.compare.internal.postprocessor.factories.IChangeFactory;
import org.eclipse.emf.compare.postprocessor.IPostProcessor;
import org.eclipse.emf.compare.uml2.internal.UMLDiff;
import org.eclipse.papyrusrt.umlrt.tooling.compare.umlrt.internal.UMLRTDiff;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;

@SuppressWarnings("restriction")
public class UMLRTPostProcessor implements IPostProcessor {

	private Map<Class<? extends Diff>, IChangeFactory> umlRtExtensionFactoriesMap;

	@Override
	public void postComparison(Comparison comparison, Monitor monitor) {
		applyUMLRTDiffExtensions(comparison);
		rewriteConflictsInvolvingUMLRTDiffs(comparison);
	}

	/**
	 * Rewrite conflicts among refining diffs into conflicts that only contain their refined diffs.
	 * <p>
	 * If we have conflicts among diffs that refine UMLRT diffs, we may end up with conflicts that
	 * only contain one diff, because we filter refining diffs of UMLRTDiffs. To ensure that the
	 * conflicts are shown correctly in the user interface, we have to rewrite those conflicts into
	 * conflicts that contain the UMLRTDiff (refined diff) instead of their refining diffs only.
	 * As there may be multiple conflicts involving refining diffs of the same refined diffs, we
	 * may have to join multiple diffs into one diff that only contain the refined diff, because
	 * one diff must only be part of one conflict at a time.
	 * </p>
	 * 
	 * @param comparison
	 *            The comparison to rewrite the conflicts in.
	 */
	private void rewriteConflictsInvolvingUMLRTDiffs(Comparison comparison) {
		final List<Conflict> needUpdate = getConflictsInvolvingUMLRTDiffs(comparison);
		if (needUpdate.size() < 2) {
			return;
		}

		while (!needUpdate.isEmpty()) {
			final Conflict currentConflict = needUpdate.get(0);
			final ArrayList<Conflict> updated = new ArrayList<>();
			updated.add(currentConflict);
			final UMLRTDiff change1 = getUMLRTDiff(currentConflict);
			if (change1 != null) {
				final UMLRTDiff change2 = getConflictingUMLRTDiff(currentConflict, change1);
				if (change2 != null) {
					updated.add(change2.getConflict());
					// check if there is an existing conflict which contains either of the changes
					if (change1.getConflict() != null && change2.getConflict() != null && !change1.getConflict().equals(change2.getConflict())) {
						// re-use existing conflict by joining the conflicts of change1 and change2
						joinConflicts(change1, change2, comparison);
					}
				}
			}
			needUpdate.removeAll(updated);
		}
	}

	private List<Conflict> getConflictsInvolvingUMLRTDiffs(Comparison comparison) {
		return Lists.newArrayList(filter(comparison.getConflicts(), conflictInvolvesUMLRTDiff()));
	}

	private static Predicate<Conflict> conflictInvolvesUMLRTDiff() {
		return new Predicate<Conflict>() {
			@Override
			public boolean apply(Conflict input) {
				return input != null && any(input.getDifferences(),
						or(instanceOf(UMLRTDiff.class), diffRefinesDiffType(UMLRTDiff.class)));
			}
		};
	}

	private static Predicate<Diff> diffRefinesDiffType(Class<?> clazz) {
		return new Predicate<Diff>() {
			@Override
			public boolean apply(Diff input) {
				return input != null && any(input.getRefines(), instanceOf(clazz));
			}
		};
	}

	private void joinConflicts(UMLRTDiff change1, UMLRTDiff change2, Comparison comparison) {
		final Conflict change2Conflict = change2.getConflict();
		change1.getConflict().getDifferences().addAll(change2Conflict.getDifferences());
		comparison.getConflicts().remove(change2Conflict);
	}

	private UMLRTDiff getUMLRTDiff(Conflict conflict) {
		return (UMLRTDiff) find(conflict.getDifferences(), instanceOf(UMLRTDiff.class), null);
	}

	private UMLRTDiff getConflictingUMLRTDiff(Conflict conflict, UMLRTDiff change) {
		for (Diff conflictingDiff : conflict.getDifferences()) {
			if (conflictingDiff == change || conflictingDiff.getSource() == change.getSource()) {
				continue;
			}
			if (conflictingDiff instanceof UMLRTDiff) {
				return (UMLRTDiff) conflictingDiff;
			} else {
				UMLRTDiff result = (UMLRTDiff) find(conflictingDiff.getRefines(), instanceOf(UMLRTDiff.class), null);
				if (result == change) {
					continue;
				}
				return result;
			}
		}
		return null;
	}

	private void applyUMLRTDiffExtensions(Comparison comparison) {
		umlRtExtensionFactoriesMap = createExtensionFactories();
		createUMLRTDiffExtensions(comparison);
		fillRequiredDifferences(comparison);
	}

	private Map<Class<? extends Diff>, IChangeFactory> createExtensionFactories() {
		return UMLRTExtensionFactoryRegistry.createExtensionFactories();
	}

	private void createUMLRTDiffExtensions(Comparison comparison) {
		for (Diff diff : comparison.getDifferences()) {
			createUMLRTDiffExtension(diff);
		}
	}

	private void createUMLRTDiffExtension(Diff element) {
		for (IChangeFactory factory : umlRtExtensionFactoriesMap.values()) {
			if (factory.handles(element)) {
				final Diff extension = factory.create(element);
				if (!extension.getRefinedBy().isEmpty()) {
					final Match match = factory.getParentMatch(element);
					if (match != null) {
						match.getDifferences().add(extension);
					}
				}
			}
		}
	}

	private void fillRequiredDifferences(Comparison comparison) {
		for (Diff umlDiff : comparison.getDifferences()) {
			if (umlDiff instanceof UMLDiff) {
				final Class<?> classDiffElement = umlDiff.eClass().getInstanceClass();
				final IChangeFactory diffFactory = umlRtExtensionFactoriesMap.get(classDiffElement);
				if (diffFactory != null) {
					diffFactory.fillRequiredDifferences(comparison, umlDiff);
				}
			}
		}
	}

	@Override
	public void postConflicts(Comparison comparison, Monitor monitor) {
		// nothing to do
	}

	@Override
	public void postMatch(Comparison comparison, Monitor monitor) {
		// nothing to do
	}

	@Override
	public void postDiff(Comparison comparison, Monitor monitor) {
		// nothing to do
	}

	@Override
	public void postRequirements(Comparison comparison, Monitor monitor) {
		// nothing to do
	}

	@Override
	public void postEquivalences(Comparison comparison, Monitor monitor) {
		// nothing to do
	}
}
