/*******************************************************************************
 * Copyright (c) 2016 EclipseSource Muenchen GmbH and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Alexandra Buzila - initial API and implementation
 *******************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.compare.ui.internal;

import org.eclipse.emf.compare.Comparison;
import org.eclipse.emf.compare.Diff;
import org.eclipse.emf.compare.rcp.ui.structuremergeviewer.filters.AbstractDifferenceFilter;
import org.eclipse.emf.compare.scope.IComparisonScope;
import org.eclipse.emf.compare.uml2.internal.UMLDiff;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.tree.TreeNode;
import org.eclipse.papyrusrt.umlrt.tooling.compare.umlrt.internal.UMLRTDiff;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;


/** Filter that does not filter out UMLRT differences, but filters out differences that refine other UMLDiffs. This filter replaces the behavior of the UMLRefinedElementFilter from EMFCompare. */
@SuppressWarnings("restriction")
public class UMLRTElementsFilter extends AbstractDifferenceFilter {

	/**
	 * The predicate use by this filter when it is selected.
	 */
	private static final Predicate<? super EObject> PREDICATE_WHEN_SELECTED = new Predicate<EObject>() {
		@Override
		public boolean apply(EObject input) {
			if (input instanceof TreeNode) {
				TreeNode node = (TreeNode) input;
				EObject data = node.getData();
				// we don't filter out UMLRTDiffs or nodes containing UMLRTDiffs as children
				if (data instanceof UMLRTDiff || containsUMLRTDiffChild(node)) {
					return false;
				} else if (data instanceof Diff) {
					// behaviour of UMLRefinedElementFilter from EMFCompare
					Diff diff = (Diff) data;
					boolean refines = Iterables.any(diff.getRefines(), Predicates.instanceOf(UMLDiff.class));
					return refines;
				}
			}
			return false;
		}
	};

	/**
	 * Constructs the filter with the appropriate predicate.
	 */
	public UMLRTElementsFilter() {
		super();
	}

	@Override
	public Predicate<? super EObject> getPredicateWhenSelected() {
		return PREDICATE_WHEN_SELECTED;
	}

	/**
	 * Checks whether the given node has a child of type {@link UMLRTDiff}.
	 * 
	 * @param parentNode
	 *            the parent {@link TreeNode}
	 * @return <code>true</code> if the given node contains a child node of type {@link UMLRTDiff}
	 */
	private static boolean containsUMLRTDiffChild(TreeNode parentNode) {
		for (TreeNode node : parentNode.getChildren()) {
			if (node.getData() instanceof UMLRTDiff) {
				return true;
			}
			if (containsUMLRTDiffChild(node)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public boolean isEnabled(IComparisonScope scope, Comparison comparison) {
		if (scope != null) {
			for (String nsURI : scope.getNsURIs()) {
				if (nsURI.matches("http://www\\.eclipse\\.org/papyrus.*/umlrt")) { //$NON-NLS-1$
					return true;
				}
			}
		}
		return false;
	}
}
