/*****************************************************************************
 * Copyright (c) 2015, 2016 CEA LIST, Christian W. Damus, and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Celine Janssens (ALL4TEC) celine.janssens@all4tec.net  - Initial API and implementation
 *   Christian W. Damus - bugs 477819, 482599, 472885, 489939, 496304, 500743
 *
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.editparts;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.PositionConstants;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gmf.runtime.diagram.ui.figures.IBorderItemLocator;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.uml.diagram.composite.edit.parts.PropertyPartEditPartCN;
import org.eclipse.papyrusrt.umlrt.core.types.advice.CapsulePartEditHelperAdvice;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.editparts.providers.RTEditPartProvider;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.figures.RTPropertyPartFigure;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.internal.editpolicies.PortContainerEditPolicy;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.locator.RTPortPositionLocator;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.MultiplicityElement;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;

/**
 * This Class redefines the PropertyPartEditPartCN for the Real Time Context
 * This editPart is used to control the CapsulePart.
 *
 * @see {@link RTEditPartProvider}
 *
 * @author Céline JANSSENS
 *
 */
public class RTPropertyPartEditPart extends PropertyPartEditPartCN {

	/**
	 *
	 */
	private static final int LAYER_OPACITY = 0;

	/**
	 * Horizontal Offset
	 */
	private static final int X_OFFSET = 3;

	/**
	 * Vertical Offset
	 */
	private static final int Y_OFFSET = X_OFFSET;

	/**
	 * Number of Layer
	 */
	private static final int LAYER_NUMBER = 1;

	/**
	 * Path of the hash pattern to use
	 */
	private static final String PATTERN_PATH = "/icons/hash_pattern_grey.png";//$NON-NLS-1$

	/**
	 * The multiplicity Adapter
	 */
	Adapter multiplicityValueListener;


	/**
	 * Constructor.
	 *
	 * @param view
	 */
	public RTPropertyPartEditPart(final View view) {
		super(view);
	}

	@Override
	protected void createDefaultEditPolicies() {
		super.createDefaultEditPolicies();

		// Custom arrange behaviour
		installEditPolicy(EditPolicy.CONTAINER_ROLE, new PortContainerEditPolicy());
	}

	@Override
	protected void addSemanticListeners() {

		super.addSemanticListeners();
		multiplicityValueListener = getMultiplicityListener();

		if (hasNotationView() && getNotationView().isSetElement()) {
			Element element = getUMLElement();

			element.eAdapters().add(multiplicityValueListener);

			if (null != ((MultiplicityElement) element).getLowerValue()) {
				((MultiplicityElement) element).getLowerValue().eAdapters().add(multiplicityValueListener);
			}

			if (null != ((MultiplicityElement) element).getUpperValue()) {
				((MultiplicityElement) getUMLElement()).getUpperValue().eAdapters().add(multiplicityValueListener);
			}
		}

	}


	@Override
	protected void removeSemanticListeners() {
		super.removeSemanticListeners();

		// Only do this if we have a notation view that references an
		// element of its own. Otherwise, we are in the process of
		// deleting the element and/or the view of it
		if (hasNotationView() && getNotationView().isSetElement()) {

			Element element = getUMLElement();
			if (element != null) { // Would be null in view deletion scenario
				element.eAdapters().remove(multiplicityValueListener);
			}

			if (element instanceof MultiplicityElement) {
				MultiplicityElement mult = (MultiplicityElement) element;

				if (null != mult.getLowerValue()) {
					mult.getLowerValue().eAdapters().remove(multiplicityValueListener);
				}

				if (null != mult.getUpperValue()) {
					mult.getUpperValue().eAdapters().remove(multiplicityValueListener);
				}
				multiplicityValueListener = null;
			}
		}
	}


	/**
	 * Retrieve the listener for Multiplicity Bounds
	 * When the ValueSpecification is created , a listener is put on the lower and the Upper Value
	 * When the bounds value are set, we refresh the visual
	 * 
	 * {@link CapsulePartEditHelperAdvice#getBeforeConfigureCommand()}
	 */
	protected Adapter getMultiplicityListener() {
		return new AdapterImpl() {

			@Override
			public void notifyChanged(Notification notification) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();


				if (hasNotationView() && getNotationView().isSetElement()) {
					Element element = getUMLElement();

					if ((notifier == element) && (feature == UMLPackage.eINSTANCE.getMultiplicityElement_LowerValue()) && (type == Notification.SET)) {
						ValueSpecification lowerValue = ((MultiplicityElement) notifier).getLowerValue();
						if (lowerValue != null) { // Maybe this is a deletion event
							lowerValue.eAdapters().add(multiplicityValueListener);
						}

					} else if ((notifier == element) && (feature == UMLPackage.eINSTANCE.getMultiplicityElement_UpperValue()) && (type == Notification.SET)) {
						ValueSpecification upperValue = ((MultiplicityElement) notifier).getUpperValue();
						if (upperValue != null) { // Maybe this is a deletion event
							upperValue.eAdapters().add(multiplicityValueListener);
						}

					}
					if (notifier instanceof ValueSpecification && ((ValueSpecification) notifier).eContainer().equals(element)) {

						refreshVisuals();
					}
				}
			}
		};
	}

	@Override
	protected IFigure createNodeShape() {
		primaryShape = new RTPropertyPartFigure();
		return primaryShape;
	}

	@Override
	public RTPropertyPartFigure getPrimaryShape() {
		return (RTPropertyPartFigure) primaryShape;
	}

	@Override
	protected void refreshVisuals() {

		if (hasNotationView() && getNotationView().isSetElement()) {
			boolean isHash = 0 == ((MultiplicityElement) getUMLElement()).getLower();
			getPrimaryShape().setHashed(isHash);

			// The edit part is displayed as a stack only if there can be more than one Capsule Part
			boolean stack = 1 < ((MultiplicityElement) getUMLElement()).getUpper();

			getPrimaryShape().setStack(stack);
		}
		getPrimaryShape().setLayerNumber(LAYER_NUMBER);
		getPrimaryShape().setXOffSet(X_OFFSET);
		getPrimaryShape().setYOffSet(Y_OFFSET);
		getPrimaryShape().setPathPattern(PATTERN_PATH);
		getPrimaryShape().setLayerOpacity(LAYER_OPACITY);

		super.refreshVisuals();

	}

	/**
	 * Overridden to install the {@link RTPortPositionLocator} as the constraint for port figures.
	 */
	@Override
	protected boolean addFixedChild(EditPart childEditPart) {
		boolean result;

		if (IRTPortEditPart.isPortOnPart(childEditPart)) {
			IRTPortEditPart portOnPart = (IRTPortEditPart) childEditPart;
			if (hasNotationView() && getNotationView().isSetElement()) {
				IBorderItemLocator locator = new RTPortPositionLocator(
						portOnPart.getPort(), getMainFigure(),
						PositionConstants.NONE, portOnPart.getDefaultScaleFactor());

				getBorderedFigure().getBorderItemContainer().add(portOnPart.getFigure(), locator);
			}

			result = true;
		} else {
			result = super.addFixedChild(childEditPart);
		}

		return result;
	}
}
