/*****************************************************************************
 * Copyright (c) 2016 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Celine JANSSENS (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *   
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.diagram.common.utils;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gmf.runtime.draw2d.ui.graphics.ColorRegistry;
import org.eclipse.gmf.runtime.gef.ui.figures.NodeFigure;
import org.eclipse.papyrus.infra.gmfdiag.common.figure.node.IRoundedRectangleFigure;
import org.eclipse.papyrus.uml.diagram.common.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.diagram.common.figures.IStackedFigure;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;

/**
 * Utils for all the Figure Drawing
 * 
 * @author Céline JANSSENS
 *
 */
public class DrawFigureUtils {

	/**
	 * Paint different layers including main figure to form a kind of stack.
	 * 
	 * @param figure
	 *            The original figure
	 * 
	 * @param graphics
	 *            the current graphics
	 * 
	 */
	public static void paintStack(final IStackedFigure figure, final Graphics graphics) {

		if ((figure instanceof NodeFigure) && (figure instanceof IRoundedRectangleFigure)) {

			Rectangle bounds = ((NodeFigure) figure).getBounds();
			Rectangle clip = bounds.getCopy();
			IRoundedRectangleFigure roundedFigure = (IRoundedRectangleFigure) figure;

			int nl = figure.getLayerNumber();

			// Calculate the offsets
			int xOffSet = (int) Math.round(nl * figure.getXOffSet());
			int yOffSet = (int) Math.round(nl * figure.getYOffSet());
			int xDelta = Math.abs(xOffSet) + 1;
			int yDelta = Math.abs(yOffSet) + 1;

			// reduce width and height of the rectangle, calculate coordinates in loop
			Rectangle layer = new Rectangle(
					0, 0, bounds.width - xDelta, bounds.height - yDelta);

			for (int i = nl; i >= 0; i--) {

				// Translate the layer of the Offset, reduce width and height
				layer.x = bounds.x + i * xOffSet;
				layer.y = bounds.y + (nl - i) * yOffSet;

				graphics.pushState();

				// Calculate the Clip
				Rectangle clipRectangle = graphics.getClip(clip).getUnion(layer).expand(1, 1);
				graphics.setClip(clipRectangle.getCopy());

				int rx = 0;
				int ry = 0;
				if (roundedFigure.getCornerDimensions().width > xDelta) {
					rx = roundedFigure.getCornerDimensions().width - xDelta;
				}
				if (roundedFigure.getCornerDimensions().height > 0) {
					ry = roundedFigure.getCornerDimensions().height - yDelta;
				}
				if (i == 0) {
					// 0 == original figure, fill only this one
					graphics.setBackgroundColor(figure.getLayerColor());
					graphics.fillRoundRectangle(layer, rx, ry);
					replicateBackground((NodeFigure) figure, graphics, layer);
				}

				graphics.popState();

				// draw the layer
				graphics.setLineWidth(figure.getLayerLineWidth());
				graphics.drawRoundRectangle(layer, rx, ry);
			}
		}
	}

	/**
	 * Fill the Background of the current graphics
	 * Fill a gradient if the figure has gradient
	 * Fill with a color if not.
	 * 
	 * @param figure
	 *            the original figure (must be a NodeFigure)
	 * @param graphics
	 *            the current graphic of the layer
	 * @param layer
	 *            the bounds of the current layer
	 */
	public static void replicateBackground(final NodeFigure figure, final Graphics graphics, Rectangle layer) {

		if ((figure).isUsingGradient()) {
			// if figure has gradient, fill the layer with the gradient
			replicateGradient(figure, graphics, layer);
		} else {
			// otherwise fill the background of the layer with the same color.
			graphics.setBackgroundColor((figure).getBackgroundColor());
		}

	}




	/**
	 * Fill the Gradient of the graphics based on the figure gradient
	 * 
	 * @param figure
	 *            The based figure to replicate the gradient
	 * @param graphics
	 *            the graphics on which the gradient is set
	 * @param rectangle
	 *            The portion of the graphics filled.
	 */
	public static void replicateGradient(final NodeFigure figure, final Graphics graphics, final Rectangle rectangle) {

		if (figure instanceof NodeFigure) {

			fillGradient(
					getColor(figure.getGradientColor2()),
					getColor(figure.getGradientColor1()),
					graphics,
					rectangle,
					figure.getGradientStyle() == 0);
		}
	}


	/**
	 * Get the Color from the integer color
	 * 
	 * @param integerColor
	 */
	public static Color getColor(int integerColor) {
		return ColorRegistry.getInstance().getColor(integerColor);

	}

	/**
	 * Fill the gradient of a Graphics
	 * 
	 * @param gradiantColor1
	 *            First Color of the gradient
	 * @param gradientColor2
	 *            Second Color of the gradient
	 * @param graphics
	 *            The graphics to draw the gradient
	 * @param rectangle
	 *            the rectangle area in which the gradient is filled
	 * @param style
	 *            true if vertical gradient
	 */
	public static void fillGradient(final Color gradiantColor1, final Color gradientColor2, final Graphics graphics, final Rectangle rectangle, final boolean style) {
		// Set the first Color of the gradient
		graphics.setForegroundColor(gradiantColor1);
		// Set the second Color of the gradient
		graphics.setBackgroundColor(gradientColor2);
		// Fill the gradient
		graphics.fillGradient(rectangle.getCopy(), style);
	}


	/**
	 * Paint a Pattern on top of a Figure
	 * TODO: Take the rounded corner into account.
	 * To be called AFTER painting the figure.
	 * 
	 * @param figure
	 *            the figure on which the pattern is applied
	 * @param graphics
	 *            The current graphics being draw
	 */
	public static void paintPattern(final IRoundedRectangleFigure figure, final Graphics graphics, final String patternPath) {

		// Create the Image of the pattern
		Image hashImage = Activator.getPluginIconImage(org.eclipse.papyrusrt.umlrt.tooling.ui.Activator.PLUGIN_ID, patternPath);


		// Get the dimension of the pattern
		int imageWidth = hashImage.getBounds().width;
		int imageHeight = hashImage.getBounds().height;

		// Defines how many times the pattern fill the figure horizontally and vertically
		int horizontalTimes = (figure.getBounds().width() / imageWidth) + 1;
		int verticalTimes = (figure.getBounds().height() / imageHeight) + 1;

		// Define the origin point to start the pattern filling
		Point origin = new Point(figure.getBounds().getLocation());
		Rectangle clipRectangle = figure.getBounds().getCopy();

		// Then fill the figure with the pattern and loop horizontally and vertically to fill all the figure with the pattern
		for (int verticalTime = 0; verticalTime < verticalTimes; verticalTime++) {
			for (int horizontalTime = 0; horizontalTime < horizontalTimes; horizontalTime++) {
				graphics.pushState();
				graphics.getClip(clipRectangle);
				graphics.setClip(clipRectangle.expand(-1, -1));

				graphics.drawImage(hashImage, origin);
				graphics.popState();
				// translate the original point accordingly
				origin.translate(imageWidth, 0);

			}
			// Reset the line origin
			origin.setX(figure.getBounds().getLocation().x());
			// Translate the column origin
			origin.translate(0, imageHeight);
		}
	}

}
