/*****************************************************************************
 * Copyright (c) 2015, 2016 CEA LIST, Christian W. Damus, and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Celine Janssens (ALL4TEC) celine.janssens@all4tec.net - Initial API and implementation
 *  Christian W. Damus - bugs 476984, 495908
 *
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.properties.editors;

import java.util.Collection;
import java.util.List;
import java.util.OptionalInt;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandWrapper;
import org.eclipse.emf.edit.command.MoveCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.DestroyElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.DestroyRequest;
import org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrus.infra.nattable.manager.table.INattableModelManager;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.infra.widgets.editors.AbstractListEditor;
import org.eclipse.papyrus.infra.widgets.messages.Messages;
import org.eclipse.papyrus.uml.diagram.common.util.CommandUtil;
import org.eclipse.papyrus.uml.service.types.element.UMLElementTypes;
import org.eclipse.papyrusrt.umlrt.core.utils.EMFHacks;
import org.eclipse.papyrusrt.umlrt.tooling.properties.Activator;
import org.eclipse.papyrusrt.umlrt.tooling.properties.widget.OwnedParameterOperationPropertyEditor;
import org.eclipse.papyrusrt.umlrt.tooling.properties.widget.ParameterControlWidget;
import org.eclipse.papyrusrt.umlrt.tooling.ui.internal.types.UMLRTUIElementTypesEnumerator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * Button Control Editor for the parameter.
 * Used in the Owned Parameter view of the Protocol Message
 * Button creations, with the related action (add, up, down, remove)
 * 
 * @see ParameterControlWidget
 * @see OwnedParameterOperationPropertyEditor
 * 
 * @author Céline JANSSENS
 *
 */
public class ParameterControlEditor extends AbstractListEditor implements SelectionListener, DisposeListener {

	/**
	 * Move Down Command Label
	 */
	private static final String MOVE_DOWN = "Move Down";

	/**
	 * Remove Element Command Label
	 */
	private static final String REMOVE_ELEMENT = "Remove Element";

	/**
	 * Move Up Command Label
	 */
	private static final String MOVE_UP = "Move Up";

	/**
	 * Icon for the delete button
	 */
	private static final String DELETE_BUTTON_ICON = "/icons/Delete_12x12.gif"; //$NON-NLS-1$

	/**
	 * Icon for the Add button
	 */
	private static final String ADD_BUTTON_ICON = "/icons/Add_12x12.gif";//$NON-NLS-1$

	/**
	 * Icon for the Down button
	 */
	private static final String DOWN_BUTTON_ICON = "/icons/Down_12x12.gif";//$NON-NLS-1$

	/**
	 * Icon for the Up button
	 */
	private static final String UP_BUTTON_ICON = "/icons/Up_12x12.gif";//$NON-NLS-1$

	/**
	 * Label for Protocol Message Parameter
	 */
	private static final String CREATE_PROTOCOL_MESSAGE_PARAMETER = "Creation of Protocol Message Parameter";

	/**
	 * Command provider for the Operation
	 */
	private IElementEditService provider = ElementEditServiceUtils.getCommandProvider(UMLElementTypes.OPERATION);

	/**
	 * A Composite containing the different control buttons
	 * (Add, remove, ...)
	 */
	protected Composite controlsSection;

	/**
	 * The Add control
	 */
	protected Button add;

	/**
	 * The Remove control
	 */
	protected Button remove;

	/**
	 * The Up control
	 */
	protected Button up;

	/**
	 * The Down control
	 */
	protected Button down;


	/**
	 * The current parameter
	 */
	private Parameter selectedParameter;

	/**
	 * The list of Selected Parameters
	 */
	private List<Parameter> selectedParameters;

	/**
	 * The operation relater to the selected Parameters
	 */
	private Operation operation;

	/**
	 * Action to invoke on addition of a new parameter.
	 */
	private Consumer<? super Parameter> newParameterAction = __ -> {
	}; // No-op

	/**
	 * Action to invoke on moving of a parameter.
	 */
	private BiConsumer<? super Parameter, ? super Integer> movedParameterAction = (_1, _2) -> {
	}; // No-op

	/**
	 * 
	 * Constructor.
	 *
	 * @param parent
	 *            The Parent Composite
	 * @param style
	 *            The Style
	 * @param operation
	 *            The Related Operation on which the parameter is added
	 * @param nattableManager
	 *            The Table Manager of the Parameter Table
	 */
	public ParameterControlEditor(final Composite parent, final int style, final Operation operation, final INattableModelManager nattableManager) {
		super(parent, style);
		this.operation = operation;

		GridLayout layout = new GridLayout(label == null ? 1 : 2, false);
		layout.marginHeight = 0;
		setLayout(layout);

		createControlSelection();

		up = createButton(Activator.getDefault().getImage(UP_BUTTON_ICON), Messages.MultipleValueEditor_MoveSelectedElementsUp); // $NON-NLS-1$
		down = createButton(Activator.getDefault().getImage(DOWN_BUTTON_ICON), Messages.MultipleValueEditor_MoveSelectedElementsDown); // $NON-NLS-1$
		add = createButton(Activator.getDefault().getImage(ADD_BUTTON_ICON), Messages.MultipleValueEditor_AddElements); // $NON-NLS-1$
		remove = createButton(Activator.getDefault().getImage(DELETE_BUTTON_ICON), Messages.MultipleValueEditor_RemoveSelectedElements); // $NON-NLS-1$

		updateButtons();


	}

	/**
	 * Create the Composite for the button
	 */
	protected void createControlSelection() {
		controlsSection = new Composite(this, SWT.NONE);
		controlsSection.setLayout(new FillLayout());
		controlsSection.setLayoutData(new GridData(SWT.END, SWT.CENTER, false, false));
	}

	/**
	 * Create the Button
	 * 
	 * @param image
	 *            Image of the Button
	 * @param toolTipText
	 *            Tooltip of the new button
	 * @return the new Button
	 */
	protected Button createButton(final Image image, final String toolTipText) {
		Button button = new Button(controlsSection, SWT.PUSH);
		button.setImage(image);
		button.addSelectionListener(this);
		button.setToolTipText(toolTipText);
		return button;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#getEditableType()
	 *
	 * @return
	 */
	@Override
	public Object getEditableType() {
		return Collection.class;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#setReadOnly(boolean)
	 * 
	 */
	@Override
	public void setReadOnly(final boolean readOnly) {
		// Nothing to do
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#isReadOnly()
	 * 
	 */
	@Override
	public boolean isReadOnly() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.papyrus.infra.widgets.editors.AbstractEditor#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
	 *
	 */
	@Override
	public void widgetDisposed(final DisposeEvent e) {
		super.widgetDisposed(e);
		add.dispose();
		up.dispose();
		remove.dispose();
		down.dispose();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	@Override
	public void widgetSelected(final SelectionEvent e) {

		if (add == e.widget) {
			addAction();
		}

		if (null != getSelectedParameters()) {
			for (Parameter singleParameter : getSelectedParameters()) {

				setSelectedParameter(singleParameter);


				if (null != e.widget) {

					if (remove == e.widget) {
						removeAction();
					} else if (up == e.widget) {
						upAction();
					} else if (down == e.widget) {
						downAction();

					}
				}
			}
		}

		updateButtons();

	}



	/**
	 * Down Action
	 */
	protected void downAction() {

		// Move the selected parameter item to the next position.
		// Note that the edit-helpers don't provide move commands within
		// a list feature, so we simply do it the EMF way. There isn't any
		// useful prospect for advice on this reordering, anyways
		List<Parameter> ownedParameters = operation.getOwnedParameters();
		int indexOf = ownedParameters.indexOf(selectedParameter);
		EditingDomain domain = EMFHelper.resolveEditingDomain(operation);
		if ((domain != null) && (indexOf + 1) < ownedParameters.size()) {
			Command move = MoveCommand.create(domain,
					operation, UMLPackage.Literals.BEHAVIORAL_FEATURE__OWNED_PARAMETER,
					selectedParameter, indexOf + 1);
			if (move != null) {
				// Customize the label
				move = new CommandWrapper(MOVE_DOWN, MOVE_DOWN, move);
				CommandUtil.executeCommandInStack(move, operation);

				// Process the moved parameter
				movedParameterAction.accept(selectedParameter, indexOf);
			}
		}
	}


	/**
	 * Up Action
	 */
	protected void upAction() {
		// Move the selected parameter item to the previous position
		// Note that the edit-helpers don't provide move commands within
		// a list feature, so we simply do it the EMF way. There isn't any
		// useful prospect for advice on this reordering, anyways
		int indexOf = operation.getOwnedParameters().indexOf(selectedParameter);
		EditingDomain domain = EMFHelper.resolveEditingDomain(operation);
		if ((domain != null) && (0 <= (indexOf - 1))) {
			Command move = MoveCommand.create(domain,
					operation, UMLPackage.Literals.BEHAVIORAL_FEATURE__OWNED_PARAMETER,
					selectedParameter, indexOf - 1);
			if (move != null) {
				// Customize the label
				move = new CommandWrapper(MOVE_UP, MOVE_UP, move);
				CommandUtil.executeCommandInStack(move, operation);

				// Process the moved parameter
				movedParameterAction.accept(selectedParameter, indexOf);
			}
		}
	}


	/**
	 * Remove Action
	 */
	protected void removeAction() {

		// Build the Destroy Request
		DestroyRequest request = new DestroyElementRequest(selectedParameter, false);

		// Execute the Command
		if (null != getProvider()) {
			ICommand setCommand = getProvider().getEditCommand(request);


			if (null != setCommand) {
				setCommand.setLabel(REMOVE_ELEMENT);
				Command wrapperCommand = GMFtoEMFCommandWrapper.wrap(setCommand);
				CommandUtil.executeCommandInStack(wrapperCommand, operation);
			}
		}
	}


	/**
	 * Add Action
	 */
	protected void addAction() {

		// Build the request
		CreateElementRequest request = new CreateElementRequest(
				operation,
				UMLRTUIElementTypesEnumerator.PROTOCOL_MESSAGE_PARAMETER_CREATION_WITH_UI,
				UMLPackage.eINSTANCE.getBehavioralFeature_OwnedParameter());


		// Execute the command accordingly
		if (null != getParameterProvider()) {
			// Get the Creation Command from the Service Edit Provider of the Parameter
			ICommand createCommand = getParameterProvider().getEditCommand(request);

			if (null != createCommand) {
				createCommand.setLabel(CREATE_PROTOCOL_MESSAGE_PARAMETER);
				Command wrapperCommand = GMFtoEMFCommandWrapper.wrap(createCommand);

				// Is the table in a dialog, which is in the context of an open transaction?
				boolean nested = EMFHacks.isReadWriteTransactionActive(operation);

				if (nested) {
					// Temporarily disable notification of addition of the new parameter
					// so that the UI doesn't update prematurely, then notify the addition
					// explicitly on its behalf later.
					// XXX: This is potentially dangerous because if the command adds
					// another parameter also to the same operation, then the transaction
					// will record changes out of order and undo will be broken.
					EMFHacks.silently(operation, op -> CommandUtil.executeCommandInStack(wrapperCommand, operation));
				} else {
					// The table takes care of delaying the update, itself
					CommandUtil.executeCommandInStack(wrapperCommand, operation);
				}

				CommandResult result = createCommand.getCommandResult();
				if (result.getStatus().isOK() && (result.getReturnValue() instanceof Parameter)) {
					Parameter added = (Parameter) result.getReturnValue();

					if (nested) {
						// Notify addition of the parameter
						EMFHacks.notifyAdded(added);
					}

					// Post-process the new parameter
					newParameterAction.accept(added);
				}
			}
		}

	}

	/**
	 * Get Command Provider
	 * 
	 * @return the command Provider
	 */
	protected IElementEditService getParameterProvider() {
		IElementEditService parameterProvider = ElementEditServiceUtils.getCommandProvider(operation);
		return parameterProvider;
	}



	/**
	 * Update the Button by setting the isEnable field.
	 */
	public void updateButtons() {
		/* Disable the button 'add' if the upperBound is reached */
		if ((null == selectedParameters) || (selectedParameters.isEmpty())) {
			remove.setEnabled(false);
			up.setEnabled(false);
			down.setEnabled(false);
		} else {
			remove.setEnabled(true);

			OptionalInt minSelectionIndex = selectedParameters.stream().mapToInt(this::indexOf).min();
			OptionalInt maxSelectionIndex = selectedParameters.stream().mapToInt(this::indexOf).max();
			up.setEnabled(minSelectionIndex.orElse(-1) > 0);
			down.setEnabled(maxSelectionIndex.orElse(Integer.MAX_VALUE) < (operation.getOwnedParameters().size() - 1));
		}

		add.setEnabled(true);

	}

	private int indexOf(Parameter parameter) {
		return operation.getOwnedParameters().indexOf(parameter);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	@Override
	public void widgetDefaultSelected(final SelectionEvent e) {
		// Nothing selected by default
	}


	/**
	 * Setter of selectParamater
	 */
	public void setSelectedParameter(final Parameter selectedParameter) {
		this.selectedParameter = selectedParameter;

	}


	/**
	 * Setter of selectParamaters
	 */
	public void setSelectedParameters(final List<Parameter> selectedParameterList) {
		this.selectedParameters = selectedParameterList;

	}


	/**
	 * Getter of the Command Provider
	 */
	public IElementEditService getProvider() {
		return provider;
	}


	/**
	 * Setter of the Command Provider
	 */
	public void setProvider(IElementEditService provider) {
		this.provider = provider;
	}


	/**
	 * Getter of selectedParameters
	 */
	public List<Parameter> getSelectedParameters() {
		return selectedParameters;
	}

	/**
	 * Registers an action to be performed on the addition of a new parameter.
	 * 
	 * @param newParameterAction
	 *            an action that accepts the new parameter
	 */
	public void onParameterAdded(Consumer<? super Parameter> newParameterAction) {
		this.newParameterAction = (newParameterAction != null)
				? newParameterAction
				: __ -> {
				}; // No-op
	}

	/**
	 * Registers an action to be performed on the reordering of a parameter.
	 * 
	 * @param movedParameterAction
	 *            an action that accepts the moved parameter and its previous index in the parameters list
	 */
	public void onParameterMoved(BiConsumer<? super Parameter, ? super Integer> movedParameterAction) {
		this.movedParameterAction = (movedParameterAction != null)
				? movedParameterAction
				: (_1, _2) -> {
				}; // No-op
	}
}
