/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   Young-Soo Roh - bug495146
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.properties.widget;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Field;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.Collection;
import java.util.Map;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.emf.workspace.AbstractResourceUndoContextPolicy;
import org.eclipse.emf.workspace.IResourceUndoContextPolicy;
import org.eclipse.emf.workspace.IWorkspaceCommandStack;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.command.ILayerCommand;
import org.eclipse.nebula.widgets.nattable.command.ILayerCommandHandler;
import org.eclipse.nebula.widgets.nattable.grid.layer.GridLayer;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.nattable.layer.PapyrusGridLayer;
import org.eclipse.papyrus.infra.nattable.manager.table.INattableModelManager;
import org.eclipse.papyrus.infra.nattable.model.nattable.Table;
import org.eclipse.papyrus.infra.nattable.model.nattable.nattablestyle.BooleanValueStyle;
import org.eclipse.papyrus.infra.nattable.model.nattable.nattablestyle.NattablestylePackage;
import org.eclipse.papyrus.infra.nattable.model.nattable.nattablestyle.Style;
import org.eclipse.papyrus.infra.nattable.utils.NamedStyleConstants;
import org.eclipse.papyrus.infra.tools.util.TypeUtils;
import org.eclipse.papyrus.uml.properties.widgets.NattablePropertyEditor;
import org.eclipse.papyrusrt.umlrt.tooling.tables.internal.handlers.RTTransactionalEditCellCommandHandler;
import org.eclipse.swt.widgets.Composite;

/**
 * Customization of the standard Papyrus table property editor for UML-RT.
 */
public class RTNatTablePropertyEditor extends NattablePropertyEditor {

	/**
	 * Initializes me with my {@code parent} component and {@code style} bits.
	 *
	 * @param parent
	 *            my parent composite
	 * @param style
	 *            my styling bits
	 */
	public RTNatTablePropertyEditor(Composite parent, int style) {
		super(parent, style);
	}

	/**
	 * Overridden to ensure that the editing domain doesn't consider the referenced UML
	 * model resource as being affected when the table context is set to an element.
	 */
	@Override
	protected ServicesRegistry createServiceRegistry(EObject sourceElement) throws Exception {
		ServicesRegistry result = super.createServiceRegistry(sourceElement);

		TransactionalEditingDomain domain = result.getService(TransactionalEditingDomain.class);
		CommandStack stack = domain.getCommandStack();

		if (stack instanceof IWorkspaceCommandStack) {
			// In Papyrus, there are several different implementations of the
			// IWorkspaceCommandStack protocol
			MethodHandle policySetter = MethodHandles.publicLookup().findVirtual(
					stack.getClass(),
					"setResourceUndoContextPolicy", //$NON-NLS-1$
					MethodType.methodType(void.class, IResourceUndoContextPolicy.class));

			IResourceUndoContextPolicy policy = new AbstractResourceUndoContextPolicy() {
				@Override
				protected boolean pessimisticCrossReferences() {
					return false;
				}
			};

			try {
				policySetter.invoke(stack, policy);
			} catch (Error e) {
				throw e;
			} catch (Exception e) {
				throw e;
			} catch (Throwable t) {
				throw new UndeclaredThrowableException(t);
			}
		}

		return result;
	}

	/**
	 * Overridden to replace the edit-cell command handler in the table's grid
	 * layer with an RT-specific implementation.
	 */
	@Override
	protected NatTable createNatTableWidget(INattableModelManager manager, Composite parent, int style, Collection<?> rows) {
		NatTable result = super.createNatTableWidget(manager, parent, style, rows);

		GridLayer grid = TypeUtils.as((result == null) ? null : result.getLayer(), GridLayer.class);
		if (grid != null) {
			replaceEditCellCommandHandler(manager, grid);
		}

		// Ensure that the table has fillColumsWidth style set to true so that columns
		// size fit to initial window size. This code can be removed once bug504716 is fixed.
		// Percentage sizing should be implemented instead of fillColumnsSize style when bug504077 is fixed.
		TransactionalEditingDomain domain = getTableEditingDomain();
		Command fillColumnsCommand = getSetFillColumnTableStyleCommand(domain, manager.getTable());
		if (fillColumnsCommand.canExecute()) {
			domain.getCommandStack().execute(fillColumnsCommand);
		}

		return result;
	}

	/**
	 * Set {@code NamedStyleConstants.FILL_COLUMNS_SIZE} style to true.
	 * 
	 * @param domain
	 *            editing domain
	 * @param resource
	 *            resource
	 * @param table
	 *            table
	 * @return set style command
	 */
	private static final Command getSetFillColumnTableStyleCommand(final TransactionalEditingDomain domain, final Table table) {
		return new RecordingCommand(domain) {

			@Override
			protected void doExecute() {
				Style fillColumnsSizeStyle = table.getNamedStyle(NattablestylePackage.eINSTANCE.getBooleanValueStyle(), NamedStyleConstants.FILL_COLUMNS_SIZE);
				if (fillColumnsSizeStyle == null) {
					fillColumnsSizeStyle = table.createStyle(NattablestylePackage.eINSTANCE.getBooleanValueStyle());
					((BooleanValueStyle) fillColumnsSizeStyle).setName(NamedStyleConstants.FILL_COLUMNS_SIZE);
				}
				((BooleanValueStyle) fillColumnsSizeStyle).setBooleanValue(true);
			}
		};
	}

	private void replaceEditCellCommandHandler(INattableModelManager manager, GridLayer grid) {
		// Only do this for the layer that already installs a transactional handler
		if (grid instanceof PapyrusGridLayer) {
			Map<Class<? extends ILayerCommand>, ILayerCommandHandler<? extends ILayerCommand>> commandHandlers = getField(grid, "commandHandlers");
			if (commandHandlers != null) {
				TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(manager.getTable().getContext());
				RTTransactionalEditCellCommandHandler handler = new RTTransactionalEditCellCommandHandler(domain);
				commandHandlers.put(handler.getCommandClass(), handler);
			}
		}
	}

	/**
	 * Obtains the value of the named field of an {@code object}.
	 * 
	 * @param object
	 *            an object
	 * @param fieldName
	 *            the name of a field
	 * 
	 * @return the field's value, or {@code null} if the fields does not exist
	 *         or is not accessible (or simply has no value)
	 */
	@SuppressWarnings("unchecked")
	static <T> T getField(Object object, String fieldName) {
		T result = null;

		for (Class<?> search = object.getClass(); search != null; search = search.getSuperclass()) {
			try {
				Field field = search.getDeclaredField(fieldName);
				field.setAccessible(true);
				result = (T) field.get(object);
				break;
			} catch (Exception e) {
				// Try superclass
			}
		}

		return result;
	}
}
