/*****************************************************************************
 * Copyright (c) 2016 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrusrt.umlrt.tooling.ui.databinding;

import java.util.function.Predicate;

import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.papyrus.infra.tools.databinding.IDelegatingObservable;

/**
 * Mix-in interface for {@link IObservableList}s that support filters
 * to exclude elements.
 */
public interface IFilteredObservableList<E> {
	/**
	 * Add a filter to match objects that should be included in my filtered view
	 * of my wrapped list.
	 * 
	 * @param filter
	 *            a filter matching elements to include
	 */
	void addFilter(Predicate<? super E> filter);

	/**
	 * Removes a filter that was previously {@linkplain #addFilter(Predicate) added}
	 * to me.
	 * 
	 * @param filter
	 *            a filter to remove
	 */
	void removeFilter(Predicate<? super E> filter);

	/**
	 * Obtains the filtered list that is or that underlies a given {@code list}.
	 * 
	 * @param list
	 *            a list, which may itself be filtered or which may wrap a filtered list
	 * 
	 * @return the filtered list, or {@code null} if the {@code list} does not
	 *         support filtering
	 */
	static <E> IFilteredObservableList<E> getFilteredList(IObservableList<E> list) {
		IFilteredObservableList<E> result = null;

		if (list instanceof IFilteredObservableList<?>) {
			@SuppressWarnings("unchecked")
			IFilteredObservableList<E> filtered = (IFilteredObservableList<E>) list;
			result = filtered;
		} else if (list instanceof IDelegatingObservable) {
			@SuppressWarnings("unchecked")
			IObservableList<E> delegate = (IObservableList<E>) ((IDelegatingObservable) list).getDelegate();
			result = getFilteredList(delegate);
		}

		return result;
	}
}
