/*****************************************************************************
 * Copyright (c) 2016 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   
 *****************************************************************************/
package org.eclipse.papyrusrt.umlrt.tooling.ui.widgets;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.IObserving;
import org.eclipse.core.databinding.observable.value.ValueDiff;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.UnexecutableCommand;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.common.core.command.CompositeCommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.IEditCommandRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.emf.gmf.command.GMFtoEMFCommandWrapper;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.infra.services.edit.service.ElementEditServiceUtils;
import org.eclipse.papyrus.infra.services.edit.service.IElementEditService;
import org.eclipse.papyrus.uml.tools.Activator;
import org.eclipse.papyrus.uml.tools.databinding.PapyrusObservableValue;
import org.eclipse.papyrusrt.umlrt.core.defaultlanguage.IDefaultLanguage;
import org.eclipse.papyrusrt.umlrt.core.defaultlanguage.IDefaultLanguageService;
import org.eclipse.papyrusrt.umlrt.core.utils.CapsulePartUtils;
import org.eclipse.papyrusrt.umlrt.core.utils.MultipleAdapter;
import org.eclipse.papyrusrt.umlrt.core.utils.RTPortUtils;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralUnlimitedNatural;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;


/**
 * @author as247872
 *
 */
public class PropertyReplicationObservableValue extends PapyrusObservableValue implements IObserving {


	public static final String STAR = "*"; //$NON-NLS-1$

	private Property propertyElement;

	private PartAdapter partAdapter;

	/**
	 * Instantiates a new capsule part Replication observable value.
	 *
	 */
	public PropertyReplicationObservableValue(final EObject property, final TransactionalEditingDomain domain) {

		super(property, property.eContainingFeature(), domain);

		if (property instanceof Property) {

			propertyElement = (Property) property;

			// add listeners
			propertyElement.eAdapters().add(getPartAdapter());
			if (propertyElement.getLowerValue() != null) {
				propertyElement.getLowerValue().eAdapters().add(getPartAdapter());
			}
			if (propertyElement.getUpperValue() != null) {
				propertyElement.getUpperValue().eAdapters().add(getPartAdapter());
			}

		}

	}

	@Override
	public synchronized void dispose() {
		try {
			if (partAdapter != null) {
				partAdapter.dispose();
				partAdapter = null;
			}
		} finally {
			super.dispose();
		}
	}

	@Override
	protected Object doGetValue() {

		Object value = Integer.toString(propertyElement.getUpper());

		if (propertyElement.getUpperValue() != null && propertyElement.getLowerValue() != null) {
			value = propertyElement.getUpperValue().stringValue();
		} else if (propertyElement.getLowerValue() == null && propertyElement.getUpperValue() == null) {
			value = "None (1)";
		}
		return value;
	}

	@Override
	public Command getCommand(final Object value) {

		// Set the new value based on the Replication Value
		ValueSpecification newUpperValue = null;
		ValueSpecification newLowerValue = null;
		int upperValue;
		int lowerValue;
		if (value instanceof String && value != null) {

			// if it is an Integer, create a LiteralUnlimittedNatural for Upper and LiteralInteger for Lower
			if (((String) value).matches("[0-9]*|\\*")) {
				if (value.equals(STAR)) {
					upperValue = -1;
				} else {
					upperValue = Integer.decode((String) value);
				}

				// for capsulePart keep the 0 value of the lowervalue
				if (CapsulePartUtils.isCapsulePart(propertyElement) && propertyElement.getLower() == 0) {
					lowerValue = 0;
				} else {
					if (upperValue == -1) {
						lowerValue = 0;
					} else {
						lowerValue = upperValue;
					}
				}
				newUpperValue = UMLFactory.eINSTANCE.createLiteralUnlimitedNatural();
				((LiteralUnlimitedNatural) newUpperValue).setValue(upperValue);
				newLowerValue = UMLFactory.eINSTANCE.createLiteralInteger();
				((LiteralInteger) newLowerValue).setValue(lowerValue);

				// if the value ="None (1)" unset upper and lower values:
			} else if (((String) value).equals("None (1)")) {
				newLowerValue = null;
				newUpperValue = null;
			}

			// create an opaque expression with body = value and language = default Language
			else {
				newUpperValue = UMLFactory.eINSTANCE.createOpaqueExpression();
				setOpaqueExpressionProperties((OpaqueExpression) newUpperValue, (String) value);

				// case of Capsule Part
				if (CapsulePartUtils.isCapsulePart(propertyElement)) {
					if (propertyElement.getLower() != 0) {
						newLowerValue = UMLFactory.eINSTANCE.createOpaqueExpression();
						setOpaqueExpressionProperties((OpaqueExpression) newLowerValue, (String) value);
					} else {
						newLowerValue = UMLFactory.eINSTANCE.createLiteralInteger();
						((LiteralInteger) newLowerValue).setValue(0);
					}
					// cade of RTPort
				} else if (RTPortUtils.isRTPort(propertyElement)) {
					newLowerValue = UMLFactory.eINSTANCE.createOpaqueExpression();
					setOpaqueExpressionProperties((OpaqueExpression) newLowerValue, (String) value);

				}
			}
		}
		Command command = getCommandForCapsulePart(newLowerValue, newUpperValue);
		return command;
	}

	private String getDefaultLanguage() {
		String defaultLanguage = null;
		IDefaultLanguageService service;
		try {
			if (null != propertyElement) {
				// Retrieve the default language of the project (by default it is C++)
				service = ServiceUtilsForEObject.getInstance().getService(IDefaultLanguageService.class, propertyElement);
				service.startService();
				IDefaultLanguage language = service.getActiveDefaultLanguage(propertyElement);
				defaultLanguage = language.getName();
			}

		} catch (ServiceException e) {
			e.printStackTrace();
		}
		return defaultLanguage;
	}

	private void setOpaqueExpressionProperties(OpaqueExpression oe, String value) {
		oe.getBodies().add(value);
		// regex to enhance, only take into account simple arithmetic expression do not manage "()"...
		// if (value.matches("[0-9A-Za-a]*( ){0,}([+-/*]( ){0,}[0-9A-Za-a]*( ){0,})*")) {}
		// the oe language is set to the defaultLanguage Name (here C++).
		oe.getLanguages().add(getDefaultLanguage());
	}

	protected Command getCommandForCapsulePart(final ValueSpecification newLowerValue, final ValueSpecification newUpperValue) {

		try {
			IElementEditService provider = ElementEditServiceUtils.getCommandProvider(getObserved());
			if (provider != null) {
				CompositeCommand cc = new CompositeCommand("Edit value");
				IEditCommandRequest createUpperSetRequest = createSetRequest((TransactionalEditingDomain) domain, eObject, UMLPackage.eINSTANCE.getMultiplicityElement_UpperValue(), newUpperValue);
				IEditCommandRequest createLowerSetRequest = createSetRequest((TransactionalEditingDomain) domain, eObject, UMLPackage.eINSTANCE.getMultiplicityElement_LowerValue(), newLowerValue);
				if (createLowerSetRequest == null || createUpperSetRequest == null) {
					return UnexecutableCommand.INSTANCE;
				}
				cc.add(provider.getEditCommand(createUpperSetRequest));
				cc.add(provider.getEditCommand(createLowerSetRequest));

				return new GMFtoEMFCommandWrapper(cc);
			}
		} catch (Exception ex) {
			Activator.log.error(ex);
		}
		return UnexecutableCommand.INSTANCE;
	}


	@Override
	protected IEditCommandRequest createSetRequest(TransactionalEditingDomain domain, EObject owner, EStructuralFeature feature, Object value) {
		return new SetRequest(domain, owner, feature, value);
	}

	@Override
	public Object getObserved() {
		return propertyElement;
	}

	@Override
	public Object getValueType() {
		return String.class;
	}

	/**
	 * Retrieve the listener for Multiplicity Bounds
	 */
	protected Adapter getMultiplicityListener() {
		// For API compatibility, continue providing this, but now
		// it's the same listener reacting to all changes
		return getPartAdapter();
	}

	/**
	 * Retrieve the listener for Aggregation
	 */
	private Adapter getPartAdapter() {
		if (partAdapter == null) {
			partAdapter = new PartAdapter();
		}
		return partAdapter;
	}

	private class PartAdapter extends MultipleAdapter {
		PartAdapter() {
			super(4);
		}

		@Override
		public void notifyChanged(Notification notification) {
			if (!notification.isTouch()) {
				Object notifier = notification.getNotifier();
				int type = notification.getEventType();
				Object feature = notification.getFeature();

				if ((notifier == propertyElement) && (type == Notification.SET)) {
					if ((feature == UMLPackage.Literals.MULTIPLICITY_ELEMENT__LOWER_VALUE)
							|| (feature == UMLPackage.Literals.MULTIPLICITY_ELEMENT__UPPER_VALUE)) {
						// Switch listener to the new one
						if (notification.getOldValue() != null) {
							((ValueSpecification) notification.getOldValue()).eAdapters().remove(this);
						}
						if (notification.getNewValue() != null) {
							((ValueSpecification) notification.getNewValue()).eAdapters().add(this);
						}
						fireDiff(notification);
					}
				} else if ((notifier == propertyElement.getLowerValue()) && (type == Notification.SET)) {
					fireDiff(notification);
				} else if ((notifier == propertyElement.getUpperValue()) && (type == Notification.SET)) {
					fireDiff(notification);
				}
			}
		}

		private void fireDiff(Notification notification) {
			final ValueDiff diff = Diffs.createValueDiff(notification.getOldValue(), notification.getNewValue());
			getRealm().exec(new Runnable() {
				@Override
				public void run() {
					fireValueChange(diff);
				}
			});
		}

	}
}