/*******************************************************************************
* Copyright (c) 2015-2016 Zeligsoft (2009) Limited  and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*******************************************************************************/
package org.eclipse.papyrusrt.xtumlrt.trans.from.uml

import java.util.HashMap
import org.eclipse.emf.common.util.EList
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.common.util.BasicEList
import org.eclipse.papyrusrt.xtumlrt.common.CommonElement
import java.util.Collection
import org.eclipse.papyrusrt.xtumlrt.common.NamedElement
import org.eclipse.papyrusrt.xtumlrt.common.Annotation
import org.eclipse.uml2.uml.Stereotype
import org.eclipse.papyrusrt.xtumlrt.common.CommonFactory

/**
 * This class provides an API for accessing the generation properties given to
 * xtUMLrt elements vie the stereotypes of the corresponding UML elements.
 *
 * @author Ernesto Posse
 */
class GenerationProperties
{
    /** Maps each xtumlrt element to the list of UML stereotype applications from the C++ Properties Set profile */
    static val genProperties = new HashMap<CommonElement, EList<EObject>>

    static def setGenProperties
    (
        org.eclipse.uml2.uml.Element umlElement,
        CommonElement element
    )
    {
        if (umlElement === null || element === null || umlElement.applicableStereotypes === null)
            return;
        val stereotypeApplications =
            if (umlElement.stereotypeApplications !== null)
                umlElement.stereotypeApplications
            else
                new BasicEList<EObject>
        genProperties.put( element, stereotypeApplications )
        if (element instanceof NamedElement)
            translateStereotypes( element, stereotypeApplications )
    }

    static def getAllGenProperties( CommonElement element )
    {
        genProperties.get( element )
    }

    static dispatch def getProperty( CommonElement element, String stereotype, String property )
    {
        if (element === null || stereotype === null || property === null) return null;
        var Object propertyValue = null
        if (element instanceof NamedElement)
        {
            val annotations = element.annotations
            propertyValue = findProperty( stereotype, property, annotations )
        }
        if (propertyValue === null && genProperties.containsKey( element ))
        {
            val elemStereotypes = genProperties.get( element )
            propertyValue = findProperty( stereotype, property, elemStereotypes )
        }
        return propertyValue
    }

    protected static dispatch def Object findProperty( String stereotype, String property, Iterable<? extends EObject> annotations )
    {
        if (annotations !== null && ! annotations.empty)
        {
            val annotation = annotations.findFirst [ annotationName == stereotype ]
            if (annotation !== null)
                getAnnotationProperty( annotation, property )
        }
    }

    static dispatch def getProperty( CommonElement element, Collection<String> stereotypes, String property )
    {
        if (stereotypes === null || property === null) return null;
        var Object propertyValue = null
        if (element instanceof NamedElement)
        {
            val annotations = element.annotations
            propertyValue = findProperty( stereotypes, property, annotations )
        }
        if (propertyValue === null && genProperties.containsKey( element ))
        {
            val elemStereotypes = genProperties.get( element )
            propertyValue = findProperty( stereotypes, property, elemStereotypes )
        }
        return propertyValue
    }

    protected static dispatch def Object findProperty( Collection<String> stereotypes, String property, Iterable<? extends EObject> annotations )
    {
        if (annotations !== null && ! annotations.empty)
        {
            val annotation = annotations.findFirst [ stereotypes.contains( annotationName ) ]
            if (annotation !== null)
                getAnnotationProperty( annotation, property )
        }
    }

    protected static dispatch def getAnnotationName( Annotation annotation )
    {
        annotation.name
    }

    protected static dispatch def getAnnotationName( Stereotype stereotype )
    {
        stereotype.eClass.name
    }

    protected static dispatch def getAnnotationName( EObject eobj )
    {
        eobj.eClass.name
    }

    protected static dispatch def getAnnotationProperty( Annotation annotation, String property )
    {
        val param = annotation?.parameters?.findFirst[ it.name == property ]
        param?.value
    }

    protected static dispatch def getAnnotationProperty( Stereotype stereotype, String property )
    {
        val feature = stereotype?.eClass?.EAllStructuralFeatures?.findFirst[ it.name == property ]
        if (feature !== null)
            stereotype?.eGet( feature )
    }

    protected static dispatch def getAnnotationProperty( EObject eobj, String property )
    {
        val feature = eobj?.eClass?.EAllStructuralFeatures?.findFirst[ it.name == property ]
        if (feature !== null)
            eobj?.eGet( feature )
    }

    protected static def translateStereotypes( NamedElement element, Iterable<EObject> stereotypeApplications )
    {
        for (stereotypeApplication : stereotypeApplications)
        {
            val stereotype = stereotypeApplication.eClass
            if (stereotype instanceof Stereotype)
            {
                val stereotypeName = stereotype.name
                val currentAnnotations = element.annotations
                if (!currentAnnotations.exists[name == stereotypeName])
                {
                    val newAnnotation = CommonFactory.eINSTANCE.createAnnotation
                    newAnnotation.name = stereotypeName
                    for (stereotypeProperty : stereotype.attributes)
                    {
                        val stereotypePropertyName = stereotypeProperty.name
                        val newAnnotationParameter = CommonFactory.eINSTANCE.createAnnotationParameter
                        newAnnotationParameter.name = stereotypePropertyName
                        val value = getAnnotationProperty(stereotype, stereotypePropertyName )
                        newAnnotationParameter.value = if (value instanceof String) value else ""
                    }
                    element.annotations.add( newAnnotation )
                }
            }
        }
    }

}