/*******************************************************************************
* Copyright (c) 2015 - 2016 Zeligsoft (2009) Limited  and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*******************************************************************************/

package org.eclipse.papyrusrt.xtumlrt.trans.from.uml

import org.eclipse.emf.ecore.EObject
import org.eclipse.papyrusrt.codegen.CodeGenPlugin
import org.eclipse.papyrusrt.xtumlrt.util.DetailedException
import org.eclipse.papyrusrt.codegen.utils.UML2CppUtil
import org.eclipse.papyrusrt.xtumlrt.common.ActionCode
import org.eclipse.papyrusrt.xtumlrt.common.CommonElement
import org.eclipse.papyrusrt.xtumlrt.common.Signal
import org.eclipse.papyrusrt.xtumlrt.statemach.ChoicePoint
import org.eclipse.papyrusrt.xtumlrt.statemach.CompositeState
import org.eclipse.papyrusrt.xtumlrt.statemach.DeepHistory
import org.eclipse.papyrusrt.xtumlrt.statemach.EntryPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.ExitPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.Guard
import org.eclipse.papyrusrt.xtumlrt.statemach.InitialPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.JunctionPoint
import org.eclipse.papyrusrt.xtumlrt.statemach.Pseudostate
import org.eclipse.papyrusrt.xtumlrt.statemach.SimpleState
import org.eclipse.papyrusrt.xtumlrt.statemach.StateMachine
import org.eclipse.papyrusrt.xtumlrt.statemach.StatemachFactory
import org.eclipse.papyrusrt.xtumlrt.statemach.Transition
import org.eclipse.papyrusrt.xtumlrt.statemach.Trigger
import org.eclipse.papyrusrt.xtumlrt.statemach.Vertex
import org.eclipse.papyrusrt.xtumlrt.statemachext.EntryAction
import org.eclipse.papyrusrt.xtumlrt.statemachext.ExitAction
import org.eclipse.papyrusrt.xtumlrt.statemachext.StatemachextFactory
import org.eclipse.papyrusrt.xtumlrt.statemachext.TransitionAction
import org.eclipse.papyrusrt.xtumlrt.umlrt.PortKind
import org.eclipse.papyrusrt.xtumlrt.umlrt.RTPort
import org.eclipse.papyrusrt.xtumlrt.umlrt.RTTrigger
import org.eclipse.papyrusrt.xtumlrt.umlrt.UmlrtFactory
import org.eclipse.uml2.uml.AnyReceiveEvent
import org.eclipse.uml2.uml.Behavior
import org.eclipse.uml2.uml.CallEvent
import org.eclipse.uml2.uml.Constraint
import org.eclipse.uml2.uml.Element
import org.eclipse.uml2.uml.PseudostateKind
import org.eclipse.uml2.uml.Region
import org.eclipse.uml2.uml.State

import static extension org.eclipse.papyrusrt.xtumlrt.util.GeneralUtil.*
import static extension org.eclipse.papyrusrt.xtumlrt.external.predefined.UMLRTStateMachProfileUtil.*

class UML2xtumlrtSMTranslator extends UML2xtumlrtTranslator
{

    // UML2xtumlrtTranslator provides methods to translate ports and signals, used in triggers
    UML2xtumlrtModelTranslator uml2xtumlrtTranslator

    new (UML2xtumlrtModelTranslator uml2xtumlrtTranslator)
    {
        this.uml2xtumlrtTranslator = uml2xtumlrtTranslator
    }

    dispatch def CommonElement translate( Element element )
    {
    }

    /**
     * This is the main method. It creates a dummy composite state to be used
     * as the State Machine's top composite state.
     *
     * @param originalStateMachine  - A {@link org.eclipse.uml2.uml.StateMachine}
     * @return A {@link StateMachine}
     */
    dispatch def StateMachine
    create StatemachFactory.eINSTANCE.createStateMachine
    translate( org.eclipse.uml2.uml.StateMachine originalStateMachine )
    {
        name = originalStateMachine.name
        top = translateElement( originalStateMachine.ownedRegion ) as CompositeState
        top.validate
        translateRedefinableElement( originalStateMachine.ownedRegion, top )
        translateRedefinableElement( originalStateMachine, it )
        GenerationProperties.setGenProperties( originalStateMachine, it )
        
    }
    
    dispatch def void validate(EObject o)
    {
        for (child : o.eContents)
        {
            child.validate
        }
    }
    
    dispatch def void validate(JunctionPoint junction)
    {
        if (junction.outgoingTransitions.size != 1)
        {
            throw new DetailedException(
                "Junction point '" +
                    (junction.source as org.eclipse.uml2.uml.Pseudostate).qualifiedName +
                    "' must have exactly one outgoing transition")
                }
            }
    
    /**
     * Validate if the transition belongs to correct state
     * let n1 be the source node of t and n2 its target node.
     * 1) sibling transition: the owner of n1 is S, the owner of n2 is also S and the owner of t must be S as well.
     * 2) entering transition: the owner of n2 is S; then the owner of t must be S as well (and n1 is S or one of its entry points)
     * 3) exiting transition: the owner of n1 is S; then the owner of t must be S as well (and n2 is S or one of its exit points)
     * 4) through transition: then n1 = n2 is a composite state S and the owner of t is S.
     */
    dispatch def void validate(Transition t)
    {
        val transitionOwner = t.eContainer
        val qn = (t.source as org.eclipse.uml2.uml.Transition).qualifiedName

        var EObject sourceNode = t.sourceVertex
        if (sourceNode instanceof EntryPoint || sourceNode instanceof ExitPoint)
        {
            sourceNode = sourceNode.eContainer
        }
        var EObject targetNode = t.targetVertex
        if (targetNode instanceof EntryPoint || targetNode instanceof ExitPoint)
        {
            targetNode = sourceNode.eContainer
        }

        // check to see if this is through transition
        if (sourceNode instanceof CompositeState && sourceNode == targetNode)
        {
            // check through transition
            if (t.sourceVertex instanceof EntryPoint || t.targetVertex instanceof ExitPoint)
            {
                if (targetNode !== transitionOwner)
                {
                    throw new DetailedException("Through transition(" + qn +
                        ") must have same owner state as source and target vertex")
                }
            }
            else
            {
                // We cannot differentiate through transition with loop transition
                // so just check loosely
                if (targetNode !== transitionOwner && targetNode.eContainer !== transitionOwner)
                {
                    throw new DetailedException("Transition(" + qn +
                        ") does not belong to correct state")
                }
            }

        }
        // check to see if this is entering transition
        else if (t.sourceVertex instanceof EntryPoint || sourceNode === targetNode.eContainer)
        {
            if (sourceNode !== transitionOwner)
            {
                throw new DetailedException(
                    "Entering transition(" + qn + ") must have same owner state as target vertex")
            }
        }
        // check to see if this is exiting transition
        else if (t.targetVertex instanceof ExitPoint || targetNode === sourceNode.eContainer)
        {
            if (targetNode !== transitionOwner)
            {
                throw new DetailedException(
                    "Exiting transition(" + qn + ") must have same owner state as source vertex")
            }
        }
        // check to see if this is sibling transition 
        else if (sourceNode.eContainer == targetNode.eContainer)
        {
            if (targetNode.eContainer !== transitionOwner)
            {
                throw new DetailedException(
                    "Sibling transition:(" + qn +
                        ") must have same owner state with its vertices if the source and target has same state"
                )
            }
        }
    }

    /**
     * Translates a region into a composite state. This is because in UML-RT, (composite) states
     * have exactly one region, so there is a one-to-one correspondance between them.
     */
    dispatch def CompositeState
    create
        if (region == null) null
        else StatemachFactory.eINSTANCE.createCompositeState
    translate( Region region )
    {
        if (region !== null)
        {
            initial = translateOptionalElement( region.initialPoint ) as InitialPoint
            deepHistory = translateOptionalElement( region.deepHistoryPoint ) as DeepHistory
            for (element : region.choicePoints)
            {
                choicePoints.addIfNotNull( translateElement(element) as ChoicePoint )
            }
            for (element : region.junctionPoints)
            {
                junctionPoints.addIfNotNull( translateElement(element) as JunctionPoint )
            }
            for (element : region.substates)
            {
                substates.addIfNotNull( translateElement(element) as org.eclipse.papyrusrt.xtumlrt.statemach.State )
            }
            for (element : region.transitions)
            {
                transitions.addIfNotNull( translateElement(element) as Transition )
            }
        }
    }

    /**
     * Translates a state from the source model into a state in the internal
     * representation.
     *
     * Essentially it just decides which translation to apply depending on
     * whether the state is simple or composite (in UML-RT there are no
     * "sub-state-machine" states.)
     *
     * @param originalState - the UML2 state element in the original model
     */
    dispatch def org.eclipse.papyrusrt.xtumlrt.statemach.State
    create
        if (originalState.isSimpleState)
            translateSimpleState( originalState )
        else
            translateCompositeState( originalState )
    translate( State originalState )
    {
        if (it !== null)
        {
            name = originalState.name
            entryAction = translateFeature( originalState, "entry", Behavior, EntryAction, true ) as EntryAction
            exitAction = translateFeature( originalState, "exit", Behavior, ExitAction, true ) as ExitAction
            for (entryPoint : originalState.entryPoints)
            {
                entryPoints.addIfNotNull( translateElement(entryPoint) as EntryPoint )
            }
            for (exitPoint : originalState.exitPoints)
            {
                exitPoints.addIfNotNull( translateElement(exitPoint) as ExitPoint )
            }
            translateRedefinableElement( originalState, it )
            GenerationProperties.setGenProperties( originalState, it )
        }
    }

    def isSimpleState( State originalState )
    {
        originalState.simple
        || originalState.composite
            && (originalState.regions === null
                || originalState.regions.empty
                || (originalState.regions.size == 1
                    && (originalState.regions.get(0) as Region).isEmpty))
    }

    def boolean isEmpty( Region region )
    {
        if (region.extendedRegion === null)
            region.ownedElements === null || region.ownedElements.empty
        else
            (region.ownedElements === null && region.ownedElements.empty) && region.extendedRegion.isEmpty
    }

    /**
     * Translates a simple state.
     */
    protected def SimpleState translateSimpleState( State originalState )
    {
        StatemachFactory.eINSTANCE.createSimpleState
    }

    /**
     * Translates a composite state by translating its sub-elements (vertices
     * and transitions).
     *
     * <p>When traversing the sub-states, the translation is applied recursively.
     */
    protected def CompositeState
    translateCompositeState( State originalState )
    {
        translate( originalState.ownedRegion ) as CompositeState
    }

    dispatch def Pseudostate
    create
        if (originalPseudostate === null) null
        else
            switch (originalPseudostate.kind)
            {
                case PseudostateKind.INITIAL_LITERAL:
                    StatemachFactory.eINSTANCE.createInitialPoint
                case PseudostateKind.DEEP_HISTORY_LITERAL:
                    StatemachFactory.eINSTANCE.createDeepHistory
                case PseudostateKind.ENTRY_POINT_LITERAL:
                    StatemachFactory.eINSTANCE.createEntryPoint
                case PseudostateKind.EXIT_POINT_LITERAL:
                    StatemachFactory.eINSTANCE.createExitPoint
                case PseudostateKind.CHOICE_LITERAL:
                    StatemachFactory.eINSTANCE.createChoicePoint
                case PseudostateKind.JUNCTION_LITERAL:
                    StatemachFactory.eINSTANCE.createJunctionPoint
                default:
                    throw new RuntimeException( "Pseudostate '" + originalPseudostate.qualifiedName + "' has unsupported kind: '" + originalPseudostate.kind.toString + "'")
            }
    translate( org.eclipse.uml2.uml.Pseudostate originalPseudostate )
    {
        if (it !== null)
        {
            name = originalPseudostate.name
            GenerationProperties.setGenProperties( originalPseudostate, it )
        }
    }

    dispatch def ActionCode
    create
        if (behaviour === null) null
        else if (behaviour.owner !== null)
        {
            if (behaviour.owner instanceof State)
            {
                val state = behaviour.owner as State
                if (behaviour === state.entry)
                    StatemachextFactory.eINSTANCE.createEntryAction
                else if (behaviour === state.exit)
                    StatemachextFactory.eINSTANCE.createExitAction
            }
            else if (behaviour.owner instanceof org.eclipse.uml2.uml.Transition)
            {
                val transition = behaviour.owner as org.eclipse.uml2.uml.Transition
                if (behaviour == transition.effect)
                    StatemachextFactory.eINSTANCE.createTransitionAction
            }
        }
        else
            throw new RuntimeException( "Behavior '" + behaviour.qualifiedName + "' has no owner.")
    translate( Behavior behaviour )
    {
        if (it !== null && behaviour !== null)
        {
            name = behaviour.name
            source = UML2CppUtil.getCppCode( behaviour )
            GenerationProperties.setGenProperties( behaviour, it )
        }
    }

    dispatch def Transition
    create StatemachFactory.eINSTANCE.createTransition
    translate( org.eclipse.uml2.uml.Transition originalTransition )
    {
        name = originalTransition.name
        sourceVertex = translateFeature( originalTransition, "source", org.eclipse.uml2.uml.Vertex, Vertex ) as Vertex
        targetVertex = translateFeature( originalTransition, "target", org.eclipse.uml2.uml.Vertex, Vertex ) as Vertex
        for (trigger : originalTransition.triggers)
        {
            triggers.addIfNotNull( translateElement(trigger) as Trigger )
        }
        guard        = translateFeature( originalTransition, "guard", Constraint, Guard, true ) as Guard
        actionChain  = StatemachFactory.eINSTANCE.createActionChain
        actionChain.actions.addIfNotNull( translateFeature( originalTransition, "effect", Behavior, TransitionAction, true ) as TransitionAction )
        translateRedefinableElement( originalTransition, it )
        GenerationProperties.setGenProperties( originalTransition, it )
    }

    /**
     * @param trigger  - A {@link org.eclipse.uml2.uml.Trigger}
     * @return A {@link Trigger}
     */
    dispatch def Trigger
    create
        // if (trigger.isRTTrigger) // This test should be here, but the current profile doesn't have the RTTrigger stereotype
            translateRTTrigger( trigger )
    translate( org.eclipse.uml2.uml.Trigger trigger )
    {
        name = trigger.name
        GenerationProperties.setGenProperties( trigger, it )
    }

    protected def RTTrigger translateRTTrigger( org.eclipse.uml2.uml.Trigger trigger )
    {
        val it = UmlrtFactory.eINSTANCE.createRTTrigger
        if (trigger.ports !== null)
        {
            for (port : trigger.ports)
            {
                val triggerPort = uml2xtumlrtTranslator.translateElement(port) as RTPort
                if (PortKind.RELAY == triggerPort.kind) {
                    CodeGenPlugin.warning("Ignoring replay port from the trigger '" + trigger.qualifiedName + "'")
                } 
                else
                {
                    ports.addIfNotNull(triggerPort)
                }
            }
        }
        val event = trigger.event
        if (event === null)
            CodeGenPlugin.error( "The trigger '" + trigger.qualifiedName + "' has a null event" )
        if (event instanceof AnyReceiveEvent)
        {
            signal = UmlrtFactory.eINSTANCE.createAnyEvent
        }
        else if (event instanceof CallEvent)
        {
            val operation = event.operation
            if (operation === null)
                CodeGenPlugin.error( "The event '" + event.qualifiedName + "' has a null operation" )
            signal = uml2xtumlrtTranslator.translateElement( operation ) as Signal
        }
        else
            CodeGenPlugin.error( "The event of trigger '" + trigger.qualifiedName + "' is not a CallEvent or AnyReceiveEvent" )
        it
    }

    dispatch def create
        if (originalGuard === null) null
        else StatemachFactory.eINSTANCE.createGuard
    translate( Constraint originalGuard )
    {
        if (it !== null)
        {
            name = originalGuard?.name
            body = StatemachextFactory.eINSTANCE.createGuardAction;
            (body as ActionCode).source = UML2CppUtil.getCppCode( originalGuard )
            GenerationProperties.setGenProperties( originalGuard, it )
        }
    }

    override translateEnum(Enum<?> kind)
    {
        throw new UnsupportedOperationException("TODO: auto-generated method stub")
    }

    override resetTranslateCache(EObject element)
    {
        val key = newArrayList( element )
        _createCache_translate.remove( key )
    }

}